$ErrorActionPreference = "Stop"

$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$appRoot = Resolve-Path (Join-Path $scriptDir "..")
$venvPath = Join-Path $appRoot ".venv"

Write-Host "AI Tag Server setup" -ForegroundColor Cyan
Write-Host "App root: $appRoot"

if (-not (Get-Command py -ErrorAction SilentlyContinue)) {
    Write-Host "Python launcher (py) not found. Install Python 3.11 from python.org and rerun." -ForegroundColor Red
    exit 1
}

$py311 = $false
try {
    & py -3.11 -c "import sys; print(sys.version)" | Out-Null
    $py311 = $true
} catch {
    $py311 = $false
}

if (-not $py311) {
    Write-Host "Python 3.11 is required. Install it from python.org and rerun." -ForegroundColor Red
    exit 1
}

# Create venv
if (Test-Path $venvPath) {
    Remove-Item -Recurse -Force $venvPath
}
py -3.11 -m venv $venvPath

$activate = Join-Path $venvPath "Scripts\Activate.ps1"
. $activate

python -m pip install --upgrade pip

# Install torch + torchvision (prefer CUDA if available)
$hasNvidia = $false
try {
    $hasNvidia = $null -ne (Get-Command nvidia-smi -ErrorAction SilentlyContinue)
} catch { $hasNvidia = $false }

$useCuda = $false
if ($hasNvidia) {
    $resp = Read-Host "NVIDIA GPU detected. Install CUDA build? [Y/n]"
    if ($resp -eq "" -or $resp -eq "Y" -or $resp -eq "y") { $useCuda = $true }
}

if ($useCuda) {
    Write-Host "Select CUDA index-url (Enter for cu121):" -ForegroundColor Cyan
    Write-Host "  1) cu121" -ForegroundColor DarkGray
    Write-Host "  2) cu124" -ForegroundColor DarkGray
    $cudaChoice = Read-Host "Choose 1/2 or press Enter for cu121"
    $cudaIndex = "https://download.pytorch.org/whl/cu121"
    if ($cudaChoice -eq "2") { $cudaIndex = "https://download.pytorch.org/whl/cu124" }
    python -m pip install torch torchvision --index-url $cudaIndex
} else {
    python -m pip install torch torchvision --index-url https://download.pytorch.org/whl/cpu
}

# Install app deps (includes pydantic-core, etc.)
python -m pip install -r (Join-Path $appRoot "requirements.txt")

# OpenSSL for cert generation
if (-not (Get-Command openssl -ErrorAction SilentlyContinue)) {
    Write-Host "OpenSSL not found. Installing via winget..." -ForegroundColor Yellow
    winget install --id ShiningLight.OpenSSL.Light -e --accept-source-agreements --accept-package-agreements
    $env:Path += ";C:\Program Files\OpenSSL-Win64\bin"
}

# Generate cert and trust it
& (Join-Path $scriptDir "generate_cert.ps1")

Write-Host "Setup complete. Run: .\\run.ps1" -ForegroundColor Green
