"""
AI Tag Server — All-in-One Launcher
Setup · Settings · Start / Stop · Clean — single window, no extra dependencies.
Double-click launcher.bat to start.
"""

import json
import os
import shutil
import signal
import subprocess
import sys
import threading
import tkinter as tk
from tkinter import ttk, messagebox
from pathlib import Path

# ── paths ──────────────────────────────────────────────────────────────
SCRIPT_DIR = Path(__file__).resolve().parent
APP_ROOT = SCRIPT_DIR.parent
SETTINGS_PATH = APP_ROOT / "run_settings.json"
VENV_DIR = APP_ROOT / ".venv"
VENV_PYTHON = VENV_DIR / "Scripts" / "python.exe"
CERT_DIR = APP_ROOT / "cert"
CERT_PEM = CERT_DIR / "cert.pem"
KEY_PEM = CERT_DIR / "key.pem"
OPENSSL_CNF = SCRIPT_DIR / "openssl.cnf"
REQUIREMENTS = APP_ROOT / "requirements.txt"

# model files that can be reset
MODEL_FILES = [
    "tag_db.json", "tag_embeddings.pt", "tag_classifier.pt",
    "finetuned_clip.pt", "vlm_lora_adapter",
]
QUEUE_DIR = APP_ROOT / "training_queue"

# ── defaults ───────────────────────────────────────────────────────────
DEFAULTS = {
    "FINETUNE_ENABLED": "1",
    "FINETUNE_STEPS": "4",
    "FINETUNE_LR": "1e-5",
    "FREEZE_VISUAL": "0",
    "BATCH_SIZE": "4",
    "OPENCLIP_MODEL": "ViT-B-32",
    "OPENCLIP_PRETRAINED": "openai",
    "USE_BLIP2": "0",
    "BLIP2_MODEL_ID": "",
    "HF_TOKEN": "",
    "REQUIRE_CUDA": "0",
    "USE_HTTPS": "1",
    "TAGGING_MODE": "clip",
    "VLM_ENABLED": "0",
    "VLM_MODEL_ID": "vikhyatk/moondream2",
    "VLM_LORA_ENABLED": "0",
    "VLM_LORA_LR": "1e-4",
    "VLM_LORA_RANK": "8",
    "VLM_MAX_TAGS": "10",
}

# ── dropdown options ───────────────────────────────────────────────────
CLIP_MODELS = {
    "ViT-B-32  (~400 MB, 2-3 GB VRAM) — Fast": "ViT-B-32",
    "ViT-B-16  (~600 MB, 3-4 GB VRAM) — Better quality": "ViT-B-16",
    "ViT-L-14  (~1.5 GB, 4-6 GB VRAM) — \u2b50 Best balance": "ViT-L-14",
    "ViT-L-14-336  (~1.6 GB, 5-7 GB VRAM) — Hi-res input": "ViT-L-14-336",
    "ViT-H-14  (~2.5 GB, 8-10 GB VRAM) — High quality": "ViT-H-14",
    "ViT-g-14  (~1.7 GB, 6-8 GB VRAM) — Alt. to H-14": "ViT-g-14",
    "ViT-bigG-14  (~5+ GB, 12-16 GB VRAM) — Top quality": "ViT-bigG-14",
    "ViT-SO400M-14  (~2+ GB, 8-10 GB VRAM)": "ViT-SO400M-14",
    "RN50  (~100 MB, 1-2 GB VRAM) — CPU-friendly": "RN50",
    "RN101  (~200 MB, 2 GB VRAM)": "RN101",
}

CLIP_PRETRAINED = {
    "openai — \u2b50 Best for most use cases": "openai",
    "laion2b_s32b_b82k — Trained on 2B images": "laion2b_s32b_b82k",
    "laion400m_e32 — Smaller dataset": "laion400m_e32",
    "scratch — \u26a0 No weights (experts only)": "scratch",
}

BLIP2_MODELS = {
    "blip2-flan-t5-base  (~4 GB) \u2b50": "Salesforce/blip2-flan-t5-base",
    "blip2-flan-t5-xl  (~8 GB)": "Salesforce/blip2-flan-t5-xl",
    "blip2-flan-t5-xxl  (~16 GB)": "Salesforce/blip2-flan-t5-xxl",
    "blip2-opt-2.7b  (~7 GB)": "Salesforce/blip2-opt-2.7b",
    "blip2-opt-2.7b-coco  (~7 GB)": "Salesforce/blip2-opt-2.7b-coco",
    "blip2-opt-6.7b  (~14 GB)": "Salesforce/blip2-opt-6.7b",
}

VLM_MODELS = {
    "moondream2  (~4 GB) \u2b50 Best 6-10 GB": "vikhyatk/moondream2",
    "Florence-2-base  (~3 GB) Lightweight": "microsoft/Florence-2-base",
    "LLaVA-1.5-7b  (~8 GB) 12 GB+": "llava-hf/llava-1.5-7b-hf",
    "Qwen2-VL-7B  (~8 GB) 12 GB+": "Qwen/Qwen2-VL-7B-Instruct",
}

LORA_RANKS = ["4", "8", "16"]

# reverse maps for display
def _rev(m):
    return {v: k for k, v in m.items()}


# ── JSON helpers ───────────────────────────────────────────────────────
def load_settings():
    if SETTINGS_PATH.exists():
        try:
            return json.loads(SETTINGS_PATH.read_text("utf-8"))
        except Exception:
            pass
    return {}


def save_settings(data):
    SETTINGS_PATH.write_text(json.dumps(data, indent=2), "utf-8")


# ── colours ────────────────────────────────────────────────────────────
BG       = "#1e1e2e"
BG2      = "#181825"
SURFACE  = "#313244"
FG       = "#cdd6f4"
SUBTEXT  = "#a6adc8"
ACCENT   = "#89b4fa"
GREEN    = "#a6e3a1"
YELLOW   = "#f9e2af"
RED      = "#f38ba8"
MAUVE    = "#cba6f7"


# ======================================================================
#                           MAIN APPLICATION
# ======================================================================
class App(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("AI Tag Server")
        self.minsize(720, 620)
        self.configure(bg=BG)
        self._center(780, 660)
        self._set_dark_titlebar()

        self._server_proc = None
        self.vars = {}            # key → tk.StringVar
        self._map_vars = {}       # key → (StringVar, fwd_map)
        self._dyn_frames = {}     # key → frame to show/hide
        self._tooltip_win = None

        self._apply_style()

        # ── tab bar ──
        self.notebook = ttk.Notebook(self)
        self.notebook.pack(fill="both", expand=True, padx=4, pady=4)

        self._build_setup_tab()
        self._build_settings_tab()
        self._build_server_tab()
        self._build_tools_tab()

        # If venv doesn't exist, jump to Setup
        if not VENV_PYTHON.exists():
            self.notebook.select(0)
        else:
            self.notebook.select(1)

        self.protocol("WM_DELETE_WINDOW", self._on_close)

    # ── styling ────────────────────────────────────────────────────────
    def _apply_style(self):
        s = ttk.Style(self)
        s.theme_use("clam")
        s.configure(".", background=BG, foreground=FG, fieldbackground=SURFACE,
                     selectbackground="#45475a", insertcolor=FG, borderwidth=0)
        s.configure("TFrame", background=BG)
        s.configure("TLabel", background=BG, foreground=FG, font=("Segoe UI", 9))
        s.configure("Sub.TLabel", background=BG, foreground=SUBTEXT, font=("Segoe UI", 8))
        s.configure("TLabelframe", background=BG, foreground=ACCENT)
        s.configure("TLabelframe.Label", background=BG, foreground=ACCENT,
                     font=("Segoe UI", 10, "bold"))
        s.configure("TEntry", fieldbackground=SURFACE, foreground=FG, insertcolor=FG)
        s.configure("TCombobox", fieldbackground="#cdd6f4", foreground="#1e1e2e",
                     selectbackground="#45475a", selectforeground=FG)
        s.map("TCombobox", fieldbackground=[("readonly", "#cdd6f4")],
              foreground=[("readonly", "#1e1e2e")])
        self.option_add("*TCombobox*Listbox.background", SURFACE)
        self.option_add("*TCombobox*Listbox.foreground", FG)
        self.option_add("*TCombobox*Listbox.selectBackground", "#45475a")
        self.option_add("*TCombobox*Listbox.selectForeground", FG)
        s.configure("TCheckbutton", background=BG, foreground=FG, font=("Segoe UI", 9))
        s.configure("TNotebook", background=BG2, borderwidth=0)
        s.configure("TNotebook.Tab", background=SURFACE, foreground=SUBTEXT,
                     font=("Segoe UI", 10, "bold"), padding=[14, 6])
        s.map("TNotebook.Tab",
              background=[("selected", BG)], foreground=[("selected", ACCENT)])
        s.configure("TButton", background="#45475a", foreground=FG,
                     font=("Segoe UI", 10, "bold"), padding=6)
        s.map("TButton",
              background=[("active", ACCENT), ("pressed", "#74c7ec")],
              foreground=[("active", "#1e1e2e")])
        s.configure("Green.TButton", background=GREEN, foreground="#1e1e2e",
                     font=("Segoe UI", 11, "bold"), padding=8)
        s.map("Green.TButton", background=[("active", "#94e2d5")])
        s.configure("Blue.TButton", background=ACCENT, foreground="#1e1e2e",
                     font=("Segoe UI", 11, "bold"), padding=8)
        s.map("Blue.TButton", background=[("active", "#74c7ec")])
        s.configure("Red.TButton", background=RED, foreground="#1e1e2e",
                     font=("Segoe UI", 10, "bold"), padding=6)
        s.map("Red.TButton", background=[("active", "#eba0ac")])

    def _center(self, w, h):
        x = (self.winfo_screenwidth() - w) // 2
        y = (self.winfo_screenheight() - h) // 2
        self.geometry(f"{w}x{h}+{x}+{y}")

    def _set_dark_titlebar(self):
        try:
            import ctypes
            hwnd = ctypes.windll.user32.GetParent(self.winfo_id())
            ctypes.windll.dwmapi.DwmSetWindowAttribute(
                hwnd, 20, ctypes.byref(ctypes.c_int(1)), 4)
        except Exception:
            pass

    # ══════════════════════════════════════════════════════════════════
    #                         SETUP TAB
    # ══════════════════════════════════════════════════════════════════
    def _build_setup_tab(self):
        tab = ttk.Frame(self.notebook)
        self.notebook.add(tab, text="  Setup  ")

        top = ttk.Frame(tab)
        top.pack(fill="x", padx=16, pady=(12, 4))

        ttk.Label(top, text="Environment Setup", font=("Segoe UI", 14, "bold"),
                  foreground=ACCENT).pack(anchor="w")
        ttk.Label(top, text="Creates virtualenv, installs PyTorch + dependencies, generates HTTPS certificate.",
                  style="Sub.TLabel").pack(anchor="w", pady=(2, 8))

        # Status indicators
        status_frame = ttk.LabelFrame(tab, text="  Status  ", padding=10)
        status_frame.pack(fill="x", padx=16, pady=4)

        self._setup_status = {}
        for label_text, key in [("Python 3.11", "python"), ("Virtual Environment", "venv"),
                                 ("Dependencies", "deps"), ("HTTPS Certificate", "cert")]:
            row = ttk.Frame(status_frame)
            row.pack(fill="x", pady=2)
            lbl = ttk.Label(row, text="\u2022 " + label_text, width=24, anchor="w")
            lbl.pack(side="left")
            val = ttk.Label(row, text="Checking...", foreground=SUBTEXT)
            val.pack(side="left")
            self._setup_status[key] = val

        # CUDA selector
        cuda_frame = ttk.LabelFrame(tab, text="  CUDA Version  ", padding=10)
        cuda_frame.pack(fill="x", padx=16, pady=4)
        self._cuda_var = tk.StringVar(value="cu121")
        for val, txt in [("cu121", "CUDA 12.1 (recommended)"),
                          ("cu124", "CUDA 12.4"),
                          ("cpu", "CPU only (no NVIDIA GPU)")]:
            ttk.Radiobutton(cuda_frame, text=txt, variable=self._cuda_var, value=val).pack(anchor="w")

        # Buttons
        btn_frame = ttk.Frame(tab)
        btn_frame.pack(fill="x", padx=16, pady=8)
        ttk.Button(btn_frame, text="\u25b6  Run Full Setup", style="Green.TButton",
                   command=self._run_setup).pack(side="left", padx=(0, 8))
        ttk.Button(btn_frame, text="\U0001f504  Regenerate Certificate",
                   command=self._regen_cert).pack(side="left")

        # Output log
        log_frame = ttk.LabelFrame(tab, text="  Output  ", padding=4)
        log_frame.pack(fill="both", expand=True, padx=16, pady=(4, 12))

        self._setup_log = tk.Text(log_frame, bg=BG2, fg=FG, font=("Consolas", 9),
                                   wrap="word", bd=0, padx=6, pady=6, state="disabled")
        self._setup_log.pack(fill="both", expand=True)

        self.after(200, self._refresh_setup_status)

    def _refresh_setup_status(self):
        # Python
        try:
            subprocess.run(["py", "-3.11", "-c", "pass"], capture_output=True, timeout=5)
            self._setup_status["python"].config(text="\u2713 Found", foreground=GREEN)
        except Exception:
            self._setup_status["python"].config(text="\u2717 Not found", foreground=RED)

        # Venv
        if VENV_PYTHON.exists():
            self._setup_status["venv"].config(text="\u2713 Ready", foreground=GREEN)
        else:
            self._setup_status["venv"].config(text="\u2717 Not created", foreground=RED)

        # Deps — quick check for a key package
        if VENV_PYTHON.exists():
            try:
                r = subprocess.run(
                    [str(VENV_PYTHON), "-c", "import torch, fastapi, open_clip; print('ok')"],
                    capture_output=True, text=True, timeout=60,
                    cwd=str(APP_ROOT))
                if r.returncode == 0 and "ok" in r.stdout:
                    self._setup_status["deps"].config(text="\u2713 Installed", foreground=GREEN)
                else:
                    err = r.stderr.strip().split("\n")[-1] if r.stderr.strip() else "unknown"
                    self._setup_status["deps"].config(
                        text=f"\u2717 Missing: {err[:60]}", foreground=YELLOW)
            except subprocess.TimeoutExpired:
                self._setup_status["deps"].config(text="\u2713 Installed (slow check)", foreground=GREEN)
            except Exception:
                self._setup_status["deps"].config(text="\u2717 Check failed", foreground=YELLOW)
        else:
            self._setup_status["deps"].config(text="\u2013 Needs venv first", foreground=SUBTEXT)

        # Cert
        if CERT_PEM.exists() and KEY_PEM.exists():
            self._setup_status["cert"].config(text="\u2713 Found", foreground=GREEN)
        else:
            self._setup_status["cert"].config(text="\u2717 Missing", foreground=RED)

    def _log(self, text):
        self._setup_log.config(state="normal")
        self._setup_log.insert("end", text)
        self._setup_log.see("end")
        self._setup_log.config(state="disabled")

    def _run_cmd_async(self, cmds, callback=None):
        """Run a list of (description, [cmd]) sequentially in a thread, logging output."""
        def worker():
            for desc, cmd in cmds:
                self.after(0, lambda d=desc: self._log(f"\n{'='*50}\n{d}\n{'='*50}\n"))
                try:
                    proc = subprocess.Popen(
                        cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
                        text=True, cwd=str(APP_ROOT), creationflags=subprocess.CREATE_NO_WINDOW)
                    for line in proc.stdout:
                        self.after(0, lambda l=line: self._log(l))
                    proc.wait()
                    if proc.returncode != 0:
                        self.after(0, lambda: self._log(f"\n\u26a0 Command exited with code {proc.returncode}\n"))
                except Exception as exc:
                    self.after(0, lambda e=exc: self._log(f"\nError: {e}\n"))
            self.after(0, self._refresh_setup_status)
            if callback:
                self.after(0, callback)
        threading.Thread(target=worker, daemon=True).start()

    def _run_setup(self):
        cuda = self._cuda_var.get()
        if cuda == "cpu":
            torch_url = "https://download.pytorch.org/whl/cpu"
        elif cuda == "cu124":
            torch_url = "https://download.pytorch.org/whl/cu124"
        else:
            torch_url = "https://download.pytorch.org/whl/cu121"

        vpy = str(VENV_PYTHON)
        cmds = []

        # 1 — create venv
        if not VENV_PYTHON.exists():
            cmds.append(("Creating virtual environment...",
                         ["py", "-3.11", "-m", "venv", str(VENV_DIR)]))

        # 2 — upgrade pip
        cmds.append(("Upgrading pip...",
                      [vpy, "-m", "pip", "install", "--upgrade", "pip"]))

        # 3 — torch
        cmds.append(("Installing PyTorch...",
                      [vpy, "-m", "pip", "install", "torch", "torchvision", "--index-url", torch_url]))

        # 4 — requirements
        cmds.append(("Installing dependencies...",
                      [vpy, "-m", "pip", "install", "-r", str(REQUIREMENTS)]))

        # 5 — cert
        if not CERT_PEM.exists():
            cmds.append(("Generating HTTPS certificate...",
                          self._cert_cmd()))

        self._log("Starting setup...\n")
        self._run_cmd_async(cmds, lambda: self._log("\n\u2705 Setup complete!\n"))

    def _cert_cmd(self):
        """Return the command list to generate a certificate."""
        CERT_DIR.mkdir(exist_ok=True)
        return [
            "openssl", "req", "-x509", "-newkey", "rsa:2048",
            "-keyout", str(KEY_PEM), "-out", str(CERT_PEM),
            "-days", "365", "-nodes", "-config", str(OPENSSL_CNF),
        ]

    def _regen_cert(self):
        for p in (CERT_PEM, KEY_PEM):
            if p.exists():
                p.unlink()
        CERT_DIR.mkdir(exist_ok=True)
        self._log("\nRegenerating certificate...\n")
        self._run_cmd_async([
            ("Generating certificate...", self._cert_cmd()),
            ("Trusting certificate...",
             ["certutil", "-user", "-addstore", "-f", "Root", str(CERT_PEM)]),
        ], lambda: self._log("\u2705 Certificate regenerated.\n"))

    # ══════════════════════════════════════════════════════════════════
    #                       SETTINGS TAB
    # ══════════════════════════════════════════════════════════════════
    def _build_settings_tab(self):
        tab = ttk.Frame(self.notebook)
        self.notebook.add(tab, text="  Settings  ")

        # scrollable canvas
        canvas = tk.Canvas(tab, bg=BG, highlightthickness=0, bd=0)
        sb = ttk.Scrollbar(tab, orient="vertical", command=canvas.yview)
        self._settings_inner = ttk.Frame(canvas)
        self._settings_inner.bind("<Configure>",
                                   lambda e: canvas.configure(scrollregion=canvas.bbox("all")))
        canvas.create_window((0, 0), window=self._settings_inner, anchor="nw")
        canvas.configure(yscrollcommand=sb.set)
        canvas.pack(side="left", fill="both", expand=True)
        sb.pack(side="right", fill="y")
        canvas.bind_all("<MouseWheel>",
                         lambda e: canvas.yview_scroll(int(-1 * (e.delta / 120)), "units"))

        parent = self._settings_inner
        settings = {**DEFAULTS, **load_settings()}

        # ── Fine-Tuning ──────────────────────────────────────────────
        self._add_toggle(parent, "FINETUNE_ENABLED", "Enable Fine-Tuning",
                         "Train model on your tagged images", settings)

        ft_frame = self._make_dynamic_frame(parent, "FINETUNE_ENABLED")
        self._add_entry(ft_frame, "FINETUNE_STEPS", "Steps per Image",
                        "Iterations per image. 4=fast, 8=quality, 16=thorough.", settings)
        self._add_entry(ft_frame, "FINETUNE_LR", "Learning Rate",
                        "MUST be tiny like 1e-5. Values \u22650.01 destroy weights!", settings)
        self._add_entry(ft_frame, "BATCH_SIZE", "Batch Size",
                        "4GB\u21921-2 \u2022 6GB\u21922-4 \u2022 8GB\u21924-8 \u2022 12GB+\u21928-16", settings)
        self._add_toggle(ft_frame, "FREEZE_VISUAL", "Freeze Image Encoder",
                         "0=train both (best), 1=freeze (safe for small data)", settings)

        # ── CLIP Model ───────────────────────────────────────────────
        sec = ttk.LabelFrame(parent, text="  CLIP Model  ", padding=(12, 8))
        sec.pack(fill="x", padx=12, pady=(10, 2))
        self._add_combo_map(sec, "OPENCLIP_MODEL", "Architecture", CLIP_MODELS, settings)
        self._add_combo_map(sec, "OPENCLIP_PRETRAINED", "Pretrained Weights", CLIP_PRETRAINED, settings)

        # ── BLIP-2 ───────────────────────────────────────────────────
        self._add_toggle(parent, "USE_BLIP2", "Enable BLIP-2 Captioning",
                         "Adds 2-8 GB+ extra VRAM. Helps generate better tags.", settings)

        blip_frame = self._make_dynamic_frame(parent, "USE_BLIP2")
        self._add_combo_map(blip_frame, "BLIP2_MODEL_ID", "BLIP-2 Model", BLIP2_MODELS, settings)
        self._add_entry(blip_frame, "HF_TOKEN", "HuggingFace Token",
                        "Required to download BLIP-2 and some VLM models. "
                        "Free at huggingface.co/settings/tokens", settings)

        # ── Tagging Mode (always visible) ────────────────────────────
        vlm_sec = ttk.LabelFrame(parent, text="  Vision Language Model (VLM)  ", padding=(12, 8))
        vlm_sec.pack(fill="x", padx=12, pady=(10, 2))
        self._add_combo_map(vlm_sec, "TAGGING_MODE", "Tagging Mode",
                            {"clip — CLIP only (fast, ~50ms)": "clip",
                             "vlm — VLM only (smart, ~1-3s)": "vlm",
                             "hybrid — Both merged (best, ~2-4s)": "hybrid"},
                            settings)
        self._add_toggle(vlm_sec, "VLM_ENABLED", "Enable VLM",
                         "Load VLM at startup. Uses 3-8 GB+ extra VRAM.", settings)

        # Dynamic VLM settings — visible when VLM_ENABLED=1 OR TAGGING_MODE in vlm/hybrid
        vlm_detail = ttk.Frame(vlm_sec)
        self._dyn_frames["_vlm_detail"] = vlm_detail
        self._add_combo_map(vlm_detail, "VLM_MODEL_ID", "VLM Model", VLM_MODELS, settings)
        self._add_entry(vlm_detail, "VLM_MAX_TAGS", "Max Tags",
                        "Max tags VLM returns per image (default 10)", settings)

        self._add_toggle(vlm_detail, "VLM_LORA_ENABLED", "Enable LoRA Training",
                         "Fine-tune VLM on your images. +2 GB train, <100 MB on disk.", settings)

        lora_frame = self._make_dynamic_frame(vlm_detail, "VLM_LORA_ENABLED")
        self._add_entry(lora_frame, "VLM_LORA_LR", "LoRA Learning Rate",
                        "Default: 1e-4", settings)
        self._add_combo_map(lora_frame, "VLM_LORA_RANK", "LoRA Rank",
                            {"4 — small / fast": "4", "8 — balanced": "8", "16 — quality": "16"},
                            settings)

        # HF token (if not already shown via BLIP-2)
        self._add_entry(vlm_detail, "HF_TOKEN", "HuggingFace Token",
                        "Some VLM models require a HuggingFace token", settings,
                        duplicate=True)

        # ── Server ───────────────────────────────────────────────────
        srv_sec = ttk.LabelFrame(parent, text="  Server  ", padding=(12, 8))
        srv_sec.pack(fill="x", padx=12, pady=(10, 2))
        self._add_toggle(srv_sec, "REQUIRE_CUDA", "Require NVIDIA GPU",
                         "0=allow CPU fallback (10-50\u00d7 slower)", settings)
        self._add_toggle(srv_sec, "USE_HTTPS", "HTTPS",
                         "Required for browser access from websites", settings)

        # ── Buttons ──────────────────────────────────────────────────
        btn_f = ttk.Frame(parent)
        btn_f.pack(fill="x", padx=12, pady=(12, 16))
        ttk.Button(btn_f, text="\U0001f4be  Save", style="Green.TButton",
                   command=self._save_settings).pack(side="left", padx=(0, 8))
        ttk.Button(btn_f, text="\u25b6  Save & Start Server", style="Blue.TButton",
                   command=self._save_and_start).pack(side="left", padx=(0, 8))
        ttk.Button(btn_f, text="Reset to Defaults",
                   command=self._reset_settings).pack(side="right")

        # ── wire dynamic visibility ──────────────────────────────────
        self._vlm_detail_frame = vlm_detail
        self._wire_vlm_visibility()
        # Initial visibility state
        self._update_all_dynamic()

    # ── widget builders ────────────────────────────────────────────────
    def _add_toggle(self, parent, key, label_text, tip, settings):
        row = ttk.Frame(parent)
        row.pack(fill="x", padx=4, pady=3)
        lbl = ttk.Label(row, text=label_text, width=22, anchor="w")
        lbl.pack(side="left")
        var = self.vars.get(key)
        if var is None:
            var = tk.StringVar(value=settings.get(key, DEFAULTS.get(key, "0")))
            self.vars[key] = var
        cb = ttk.Checkbutton(row, text="Enabled", variable=var, onvalue="1", offvalue="0")
        cb.pack(side="left", padx=(4, 0))
        if tip:
            self._add_info(row, tip)

    def _add_entry(self, parent, key, label_text, tip, settings, duplicate=False):
        # If this is a duplicate key (e.g. HF_TOKEN shown in two places), reuse the var
        row = ttk.Frame(parent)
        row.pack(fill="x", padx=4, pady=3)
        lbl = ttk.Label(row, text=label_text, width=22, anchor="w")
        lbl.pack(side="left")
        var = self.vars.get(key)
        if var is None:
            var = tk.StringVar(value=settings.get(key, DEFAULTS.get(key, "")))
            self.vars[key] = var
        ent = ttk.Entry(row, textvariable=var, width=36)
        ent.pack(side="left", padx=(4, 0), fill="x", expand=True)
        if tip:
            self._add_info(row, tip)

    def _add_combo_map(self, parent, key, label_text, fwd_map, settings):
        row = ttk.Frame(parent)
        row.pack(fill="x", padx=4, pady=3)
        lbl = ttk.Label(row, text=label_text, width=22, anchor="w")
        lbl.pack(side="left")

        rev = _rev(fwd_map)
        raw_val = settings.get(key, DEFAULTS.get(key, ""))
        display_val = rev.get(raw_val, raw_val)

        var = self.vars.get(key)
        if var is None:
            var = tk.StringVar(value=display_val)
            self.vars[key] = var
            self._map_vars[key] = (var, fwd_map)
        else:
            var.set(display_val)
            if key not in self._map_vars:
                self._map_vars[key] = (var, fwd_map)

        cb = ttk.Combobox(row, textvariable=var, values=list(fwd_map.keys()), width=48, state="readonly")
        cb.pack(side="left", padx=(4, 0), fill="x", expand=True)

    def _add_info(self, parent, text):
        info = ttk.Label(parent, text=" \u2139", foreground=ACCENT,
                         font=("Segoe UI", 11, "bold"), cursor="hand2")
        info.pack(side="right", padx=(4, 0))
        info.bind("<Enter>", lambda e: self._show_tip(e, text))
        info.bind("<Leave>", lambda e: self._hide_tip())

    # ── dynamic show / hide ────────────────────────────────────────────
    def _make_dynamic_frame(self, parent, trigger_key):
        """Create a frame that shows only when trigger_key == '1'."""
        f = ttk.Frame(parent)
        self._dyn_frames[trigger_key] = f
        var = self.vars.get(trigger_key)
        if var:
            var.trace_add("write", lambda *_: self._update_dynamic(trigger_key))
        return f

    def _update_dynamic(self, trigger_key):
        f = self._dyn_frames.get(trigger_key)
        if not f:
            return
        var = self.vars.get(trigger_key)
        if var and var.get() == "1":
            f.pack(fill="x", padx=(20, 0), pady=(0, 4))
        else:
            f.pack_forget()
        # VLM detail has compound visibility
        if trigger_key in ("VLM_ENABLED", "TAGGING_MODE"):
            self._update_vlm_visibility()

    def _wire_vlm_visibility(self):
        for key in ("VLM_ENABLED", "TAGGING_MODE"):
            var = self.vars.get(key)
            if var:
                var.trace_add("write", lambda *_, k=key: self._update_vlm_visibility())

    def _update_vlm_visibility(self):
        vlm_on = self.vars.get("VLM_ENABLED", tk.StringVar(value="0")).get() == "1"
        mode_var = self.vars.get("TAGGING_MODE", tk.StringVar(value="clip"))
        # Resolve display value to raw value
        mode_raw = mode_var.get()
        if "TAGGING_MODE" in self._map_vars:
            _, fwd = self._map_vars["TAGGING_MODE"]
            mode_raw = fwd.get(mode_raw, mode_raw)
        needs_vlm = mode_raw in ("vlm", "hybrid")
        f = self._dyn_frames.get("_vlm_detail")
        if f:
            if vlm_on or needs_vlm:
                f.pack(fill="x", padx=(20, 0), pady=(0, 4))
            else:
                f.pack_forget()

    def _update_all_dynamic(self):
        for key in list(self._dyn_frames.keys()):
            if key.startswith("_"):
                continue
            self._update_dynamic(key)
        self._update_vlm_visibility()

    # ── tooltip ────────────────────────────────────────────────────────
    def _show_tip(self, event, text):
        self._hide_tip()
        tw = tk.Toplevel(self)
        tw.wm_overrideredirect(True)
        tw.configure(bg="#45475a")
        tw.wm_geometry(f"+{event.x_root + 16}+{event.y_root + 8}")
        tk.Label(tw, text=text, justify="left", bg="#45475a", fg=YELLOW,
                 font=("Segoe UI", 9), wraplength=380, padx=8, pady=6).pack()
        self._tooltip_win = tw

    def _hide_tip(self):
        if self._tooltip_win:
            self._tooltip_win.destroy()
            self._tooltip_win = None

    # ── collect / save ─────────────────────────────────────────────────
    def _collect(self):
        data = {}
        for key, var in self.vars.items():
            if key in self._map_vars:
                _, fwd = self._map_vars[key]
                data[key] = fwd.get(var.get(), var.get())
            else:
                data[key] = var.get()
        return data

    def _save_settings(self):
        data = self._collect()
        # Validate learning rates
        for lr_key in ("FINETUNE_LR", "VLM_LORA_LR"):
            try:
                if float(data.get(lr_key, "0")) >= 0.01:
                    if not messagebox.askyesno(
                            "Warning",
                            f"{lr_key} = {data[lr_key]} is very high.\n"
                            "Values \u22650.01 can destroy model weights.\n\nContinue?"):
                        return
            except ValueError:
                pass
        save_settings(data)
        messagebox.showinfo("Saved", "Settings saved.")

    def _save_and_start(self):
        self._save_settings()
        self.notebook.select(2)  # switch to Server tab
        self.after(100, self._start_server)

    def _reset_settings(self):
        if not messagebox.askyesno("Reset", "Reset ALL settings to defaults?"):
            return
        for key, var in self.vars.items():
            default = DEFAULTS.get(key, "")
            if key in self._map_vars:
                _, fwd = self._map_vars[key]
                rev = _rev(fwd)
                var.set(rev.get(default, default))
            else:
                var.set(default)

    # ══════════════════════════════════════════════════════════════════
    #                        SERVER TAB
    # ══════════════════════════════════════════════════════════════════
    def _build_server_tab(self):
        tab = ttk.Frame(self.notebook)
        self.notebook.add(tab, text="  Server  ")

        top = ttk.Frame(tab)
        top.pack(fill="x", padx=16, pady=(12, 4))
        ttk.Label(top, text="Server Control", font=("Segoe UI", 14, "bold"),
                  foreground=ACCENT).pack(anchor="w")

        self._srv_status = ttk.Label(top, text="\u25cf Stopped", foreground=RED,
                                      font=("Segoe UI", 11, "bold"))
        self._srv_status.pack(anchor="w", pady=(4, 8))

        btn_row = ttk.Frame(tab)
        btn_row.pack(fill="x", padx=16, pady=4)
        self._btn_start = ttk.Button(btn_row, text="\u25b6  Start Server",
                                      style="Green.TButton", command=self._start_server)
        self._btn_start.pack(side="left", padx=(0, 8))
        self._btn_stop = ttk.Button(btn_row, text="\u25a0  Stop Server",
                                     style="Red.TButton", command=self._stop_server,
                                     state="disabled")
        self._btn_stop.pack(side="left")

        log_frame = ttk.LabelFrame(tab, text="  Server Output  ", padding=4)
        log_frame.pack(fill="both", expand=True, padx=16, pady=(8, 12))

        self._srv_log = tk.Text(log_frame, bg=BG2, fg=FG, font=("Consolas", 9),
                                 wrap="word", bd=0, padx=6, pady=6, state="disabled")
        self._srv_log.pack(fill="both", expand=True)

    def _srv_log_append(self, text):
        self._srv_log.config(state="normal")
        self._srv_log.insert("end", text)
        self._srv_log.see("end")
        self._srv_log.config(state="disabled")

    def _start_server(self):
        if self._server_proc and self._server_proc.poll() is None:
            messagebox.showinfo("Running", "Server is already running.")
            return

        if not VENV_PYTHON.exists():
            messagebox.showerror("Error", "Virtual environment not found.\nRun Setup first.")
            return

        settings = {**DEFAULTS, **load_settings()}
        env = os.environ.copy()
        for k, v in settings.items():
            env[k] = str(v)

        cmd = [str(VENV_PYTHON), "-m", "uvicorn", "app:app",
               "--host", "0.0.0.0", "--port", "8443"]

        if settings.get("USE_HTTPS") == "1" and CERT_PEM.exists() and KEY_PEM.exists():
            cmd += ["--ssl-certfile", str(CERT_PEM), "--ssl-keyfile", str(KEY_PEM)]
            proto = "https"
        else:
            proto = "http"

        self._srv_log.config(state="normal")
        self._srv_log.delete("1.0", "end")
        self._srv_log.config(state="disabled")

        self._srv_log_append(f"Starting server on {proto}://localhost:8443 ...\n\n")

        try:
            self._server_proc = subprocess.Popen(
                cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
                text=True, env=env, cwd=str(APP_ROOT),
                creationflags=subprocess.CREATE_NO_WINDOW)
        except Exception as exc:
            self._srv_log_append(f"Failed to start: {exc}\n")
            return

        self._srv_status.config(text="\u25cf Running", foreground=GREEN)
        self._btn_start.config(state="disabled")
        self._btn_stop.config(state="normal")
        threading.Thread(target=self._tail_server, daemon=True).start()

    def _tail_server(self):
        proc = self._server_proc
        try:
            for line in proc.stdout:
                self.after(0, lambda l=line: self._srv_log_append(l))
        except Exception:
            pass
        self.after(0, self._on_server_stopped)

    def _on_server_stopped(self):
        self._srv_status.config(text="\u25cf Stopped", foreground=RED)
        self._btn_start.config(state="normal")
        self._btn_stop.config(state="disabled")
        self._srv_log_append("\n--- Server stopped ---\n")

    def _stop_server(self):
        if self._server_proc and self._server_proc.poll() is None:
            self._server_proc.terminate()
            try:
                self._server_proc.wait(timeout=5)
            except subprocess.TimeoutExpired:
                self._server_proc.kill()

    # ══════════════════════════════════════════════════════════════════
    #                         TOOLS TAB
    # ══════════════════════════════════════════════════════════════════
    def _build_tools_tab(self):
        tab = ttk.Frame(self.notebook)
        self.notebook.add(tab, text="  Tools  ")

        ttk.Label(tab, text="Maintenance Tools", font=("Segoe UI", 14, "bold"),
                  foreground=ACCENT).pack(anchor="w", padx=16, pady=(12, 8))

        # Model files status
        mf = ttk.LabelFrame(tab, text="  Trained Model Files  ", padding=10)
        mf.pack(fill="x", padx=16, pady=4)
        self._model_labels = {}
        for name in MODEL_FILES:
            row = ttk.Frame(mf)
            row.pack(fill="x", pady=1)
            ttk.Label(row, text=name, width=24, anchor="w").pack(side="left")
            val = ttk.Label(row, text="...", foreground=SUBTEXT)
            val.pack(side="left")
            self._model_labels[name] = val
        ttk.Button(mf, text="\U0001f504  Refresh", command=self._refresh_model_status).pack(
            anchor="w", pady=(6, 0))

        # Actions
        act = ttk.LabelFrame(tab, text="  Actions  ", padding=10)
        act.pack(fill="x", padx=16, pady=8)

        ttk.Button(act, text="\U0001f5d1  Reset Model Weights",
                   command=self._reset_models).pack(anchor="w", pady=2)
        ttk.Label(act, text="Deletes all trained weights. Model will retrain from scratch.",
                  style="Sub.TLabel").pack(anchor="w", padx=(28, 0))

        ttk.Button(act, text="\U0001f9f9  Clear Training Queue",
                   command=self._clear_queue).pack(anchor="w", pady=(8, 2))
        ttk.Label(act, text="Removes pending training images from the queue folder.",
                  style="Sub.TLabel").pack(anchor="w", padx=(28, 0))

        ttk.Button(act, text="\U0001f4a3  Full Clean (weights + queue + logs)",
                   style="Red.TButton", command=self._full_clean).pack(anchor="w", pady=(8, 2))
        ttk.Label(act, text="Removes everything trained. Like a fresh install.",
                  style="Sub.TLabel").pack(anchor="w", padx=(28, 0))

        self.after(300, self._refresh_model_status)

    def _refresh_model_status(self):
        for name, lbl in self._model_labels.items():
            p = APP_ROOT / name
            if p.exists():
                if p.is_dir():
                    size = sum(f.stat().st_size for f in p.rglob("*") if f.is_file())
                else:
                    size = p.stat().st_size
                if size > 1_048_576:
                    s = f"{size / 1_048_576:.1f} MB"
                else:
                    s = f"{size / 1024:.0f} KB"
                lbl.config(text=f"\u2713 {s}", foreground=GREEN)
            else:
                lbl.config(text="\u2717 not created", foreground=SUBTEXT)

    def _delete_paths(self, paths):
        deleted = 0
        for p in paths:
            if p.exists():
                if p.is_dir():
                    shutil.rmtree(p, ignore_errors=True)
                else:
                    p.unlink()
                deleted += 1
        return deleted

    def _reset_models(self):
        if not messagebox.askyesno(
                "Reset Models",
                "Delete all trained weights?\n\n"
                "Files to remove:\n" + "\n".join(f"  \u2022 {n}" for n in MODEL_FILES)):
            return
        n = self._delete_paths([APP_ROOT / name for name in MODEL_FILES])
        self._refresh_model_status()
        messagebox.showinfo("Done", f"Deleted {n} file(s). Model will retrain from scratch.")

    def _clear_queue(self):
        if not QUEUE_DIR.exists() or not any(QUEUE_DIR.iterdir()):
            messagebox.showinfo("Queue", "Training queue is already empty.")
            return
        if not messagebox.askyesno("Clear Queue", "Remove all pending training files?"):
            return
        shutil.rmtree(QUEUE_DIR, ignore_errors=True)
        QUEUE_DIR.mkdir(exist_ok=True)
        messagebox.showinfo("Done", "Training queue cleared.")

    def _full_clean(self):
        if not messagebox.askyesno(
                "Full Clean",
                "This will delete ALL trained weights, training queue, and logs.\n\n"
                "Are you sure?"):
            return
        paths = [APP_ROOT / name for name in MODEL_FILES]
        paths.append(QUEUE_DIR)
        paths.append(APP_ROOT / "logs.txt")
        n = self._delete_paths(paths)
        QUEUE_DIR.mkdir(exist_ok=True)
        self._refresh_model_status()
        messagebox.showinfo("Done", f"Full clean complete. Removed {n} item(s).")

    # ── close handler ──────────────────────────────────────────────────
    def _on_close(self):
        if self._server_proc and self._server_proc.poll() is None:
            if messagebox.askyesno("Server Running",
                                    "Server is still running. Stop it and exit?"):
                self._stop_server()
            else:
                return
        self.destroy()


if __name__ == "__main__":
    app = App()
    app.mainloop()
