<?php

declare(strict_types=1);

namespace NewSite\Visitor;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Security\IpService;
use NewSite\Auth\AdminAuth;
use NewSite\Geo\GeoService;
use NewSite\Settings\SettingsService;

/**
 * Visitor tracking and analytics service.
 *
 * Security: All database queries use prepared statements with bound parameters;
 * IP addresses are GDPR-anonymised before storage via IpService::gdprStore();
 * no user input is interpolated into SQL; admin IPs are flagged but not excluded.
 */
final class VisitorService
{
    private const CHAT_IMAGE_ROUTE_PREFIX = '/chat-image/';
    private const PROFILE_PHOTO_ROUTE_PREFIX = '/profile-photo/';

    public static function isTrackablePath(string $cleanPath): bool
    {
        $nonPagePrefixes = [
            '/cart-api', '/search-api', '/set-currency', '/set-language',
            '/notifications-api', '/messages-api', '/messenger-api',
            '/checkout-api', '/stripe-webhook', '/cookie-consent',
            '/friends-search', '/friends-action', '/friends-list',
            '/product-likes-api', '/forum-posts-api', '/forum-comments-api', '/game-leaderboard-api',
            '/invoice', '/log-error',
            '/site-file/', '/admin-file/', self::CHAT_IMAGE_ROUTE_PREFIX, '/forum-image/',
            '/contact-file/', self::PROFILE_PHOTO_ROUTE_PREFIX,
            '/assets/',
        ];
        $nonPageExact = ['/sw.js', '/manifest.json', '/robots.txt', '/sitemap.xml', '/favicon.ico'];
        foreach ($nonPagePrefixes as $prefix) {
            if (str_starts_with($cleanPath, $prefix)) {
                return false;
            }
        }
        if (in_array($cleanPath, $nonPageExact, true)) {
            return false;
        }
        return preg_match('/\.(js|css|png|jpg|jpeg|gif|webp|svg|ico|woff2?|ttf|eot|map|json)$/i', $cleanPath) !== 1;
    }

    public static function shouldSample(int $samplePercent): bool
    {
        if ($samplePercent <= 0) {
            return false;
        }
        if ($samplePercent >= 100) {
            return true;
        }
        return mt_rand(1, 100) <= $samplePercent;
    }

    public static function recordPageView(string $readablePage): int
    {
        $ip = IpService::getClientIP();
        $isAdminIp = AdminAuth::isAllowedIp($ip);
        $storedIp = IpService::gdprStore($ip);
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $referrer = $_SERVER['HTTP_REFERER'] ?? '';
        $sessionId = session_id();
        $visitorId = 0;
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT id FROM visitors WHERE ip_address = ? AND (session_id = ? OR session_id IS NULL) ORDER BY last_activity DESC LIMIT 1");
        $stmt->execute([$storedIp, $sessionId]);
        $visitor = $stmt->fetch();
        if ($visitor) {
            $now = DbHelper::nowString();
            $stmt = $db->prepare("UPDATE visitors SET last_activity = ?, visit_count = visit_count + 1, session_id = ?, is_admin_ip = ? WHERE id = ?");
            $stmt->execute([$now, $sessionId, $isAdminIp ? 1 : 0, $visitor['id']]);
            $visitorId = (int)$visitor['id'];
        } else {
            $countryCode = GeoService::resolveCountry($ip);
            $stmt = $db->prepare("INSERT INTO visitors (ip_address, country_code, user_agent, referrer, session_id, is_admin_ip) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$storedIp, $countryCode, $userAgent, $referrer, $sessionId, $isAdminIp ? 1 : 0]);
            $visitorId = DbHelper::lastInsertId($db, 'visitors');
        }
        $stmt = $db->prepare("INSERT INTO visitor_logs (visitor_id, page_url, referrer) VALUES (?, ?, ?)");
        $stmt->execute([$visitorId, $readablePage, $referrer]);
        return $visitorId;
    }

    public static function track(?string $pageUrl = null): int
    {
        $visitorId = 0;
        if ($pageUrl === null) {
            $pageUrl = $_SERVER['REQUEST_URI'] ?? '/';
        }
        $cleanPath = strtok($pageUrl, '?');
        $cleanPath = rtrim($cleanPath, '/') ?: '/';
        $samplePercent = (int)(SettingsService::get('visitor_tracking_sample_percent', '100') ?? '100');
        if (self::isTrackablePath($cleanPath) && self::shouldSample($samplePercent)) {
            $readablePage = $cleanPath === '/' ? '/' : $cleanPath;
            $visitorId = self::recordPageView($readablePage);
        }
        return $visitorId;
    }

    public static function getActive(int $minutes = 15): array
    {
        $db = DatabaseManager::getReadConnection();
        $cutoff = date('Y-m-d H:i:s', time() - ($minutes * 60));
        $stmt = $db->prepare("SELECT v.*, (SELECT COUNT(*) FROM visitor_logs WHERE visitor_id = v.id) as page_count, (SELECT page_url FROM visitor_logs WHERE visitor_id = v.id AND page_url NOT LIKE '/assets/%' ORDER BY created_at DESC LIMIT 1) as last_page FROM visitors v WHERE v.last_activity > ? ORDER BY v.last_activity DESC");
        $stmt->execute([$cutoff]);
        return $stmt->fetchAll();
    }

    public static function getLogs(int $visitorId, int $limit = 50): array
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare("SELECT * FROM visitor_logs WHERE visitor_id = ? ORDER BY created_at DESC LIMIT ?");
        $stmt->execute([$visitorId, $limit]);
        return $stmt->fetchAll();
    }

    public static function getCountryStats(): array
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->query("SELECT country_code, COUNT(*) as visitor_count, ROUND(COUNT(*) * 100.0 / (SELECT COUNT(*) FROM visitors), 1) as percentage FROM visitors WHERE country_code != 'XX' AND country_code != 'LO' GROUP BY country_code ORDER BY visitor_count DESC");
        return $stmt->fetchAll();
    }

    public static function cleanOldData(int $days = 30): bool
    {
        $db = DatabaseManager::getWriteConnection();
        $cutoff = date('Y-m-d H:i:s', time() - ($days * 86400));
        $db->prepare("DELETE FROM visitor_logs WHERE created_at < ?")->execute([$cutoff]);
        $db->prepare("DELETE FROM visitors WHERE last_activity < ?")->execute([$cutoff]);
        return true;
    }
}
