<?php

declare(strict_types=1);

namespace NewSite\Util;

/**
 * Static country data provider.
 *
 * Security notes:
 * - All data is hard-coded; no user input is interpolated into the lists.
 * - getName() returns the raw ISO code when a mapping is not found,
 *   so callers must still escape output before rendering in HTML.
 * - No inline JavaScript or inline styles are emitted (strict CSP compliance).
 */
final class CountryData
{
    /**
     * Return a flat list of country names for use in select/option dropdowns.
     *
     * @return array<int, string>
     */
    public static function getOptions(): array
    {
        return [
            'United States', 'Canada', 'United Kingdom', 'Ireland', 'Australia', 'New Zealand',
            'Germany', 'France', 'Spain', 'Italy', 'Netherlands', 'Belgium', 'Switzerland',
            'Austria', 'Sweden', 'Norway', 'Denmark', 'Finland', 'Poland', 'Portugal',
            'Czech Republic', 'Hungary', 'Romania', 'Bulgaria', 'Greece', 'Turkey',
            'Israel', 'United Arab Emirates', 'Saudi Arabia', 'Qatar', 'Kuwait',
            'Japan', 'South Korea', 'China', 'Taiwan', 'Singapore', 'Malaysia', 'Thailand',
            'Vietnam', 'Philippines', 'Indonesia', 'India', 'Pakistan', 'Bangladesh',
            'Sri Lanka', 'Nepal', 'South Africa', 'Nigeria', 'Kenya', 'Egypt', 'Morocco',
            'Brazil', 'Argentina', 'Chile', 'Colombia', 'Peru', 'Mexico', 'Costa Rica'
        ];
    }

    /**
     * Resolve an ISO 3166-1 alpha-2 country code to a human-readable name.
     *
     * @param string $code Two-letter country code (e.g. "US", "GB").
     * @return string The country name, or the original code if not found.
     */
    public static function getName(string $code): string
    {
        $countries = [
            'AF' => 'Afghanistan', 'AL' => 'Albania', 'DZ' => 'Algeria', 'AR' => 'Argentina', 'AU' => 'Australia',
            'AT' => 'Austria', 'BD' => 'Bangladesh', 'BE' => 'Belgium', 'BR' => 'Brazil', 'BG' => 'Bulgaria',
            'CA' => 'Canada', 'CL' => 'Chile', 'CN' => 'China', 'CO' => 'Colombia', 'HR' => 'Croatia',
            'CZ' => 'Czech Republic', 'DK' => 'Denmark', 'EG' => 'Egypt', 'FI' => 'Finland', 'FR' => 'France',
            'DE' => 'Germany', 'GR' => 'Greece', 'HK' => 'Hong Kong', 'HU' => 'Hungary', 'IN' => 'India',
            'ID' => 'Indonesia', 'IR' => 'Iran', 'IQ' => 'Iraq', 'IE' => 'Ireland', 'IL' => 'Israel',
            'IT' => 'Italy', 'JP' => 'Japan', 'KZ' => 'Kazakhstan', 'KE' => 'Kenya', 'KR' => 'South Korea',
            'MY' => 'Malaysia', 'MX' => 'Mexico', 'MA' => 'Morocco', 'NL' => 'Netherlands', 'NZ' => 'New Zealand',
            'NG' => 'Nigeria', 'NO' => 'Norway', 'PK' => 'Pakistan', 'PE' => 'Peru', 'PH' => 'Philippines',
            'PL' => 'Poland', 'PT' => 'Portugal', 'RO' => 'Romania', 'RU' => 'Russia', 'SA' => 'Saudi Arabia',
            'RS' => 'Serbia', 'SG' => 'Singapore', 'ZA' => 'South Africa', 'ES' => 'Spain', 'SE' => 'Sweden',
            'CH' => 'Switzerland', 'TW' => 'Taiwan', 'TH' => 'Thailand', 'TR' => 'Turkey', 'UA' => 'Ukraine',
            'AE' => 'United Arab Emirates', 'GB' => 'United Kingdom', 'US' => 'United States', 'VN' => 'Vietnam',
            'XX' => 'Unknown', 'LO' => 'Local/Private'
        ];
        return $countries[$code] ?? $code;
    }
}
