<?php

declare(strict_types=1);

namespace NewSite\Util;

/**
 * Asset URL versioning for cache busting.
 *
 * Security notes:
 * - Only appends a numeric query-string parameter (?v=timestamp) to asset paths;
 *   no user input is interpolated into the generated URLs.
 * - Relies on the PUBLIC_PATH constant defined in the _init.php bootstrap.
 * - No inline JavaScript or inline styles are emitted (strict CSP compliance).
 */
final class AssetVersioning
{
    /**
     * Generate versioned asset URL for cache busting.
     * Appends file modification time as version query string.
     * Uses PUBLIC_PATH constant (defined in bootstrap).
     *
     * @param string $path The root-relative asset path (e.g. "/assets/css/style.css").
     * @return string The path with a ?v= cache-busting query string appended.
     */
    public static function url(string $path): string
    {
        // Strip query string before checking the filesystem
        $queryPos = strpos($path, '?');
        $basePath = ($queryPos !== false) ? substr($path, 0, $queryPos) : $path;
        $filePath = PUBLIC_PATH . $basePath;

        $version  = file_exists($filePath) ? (string) filemtime($filePath) : (string) time();

        // Use &v= if the path already contains a query string, otherwise ?v=
        $separator = ($queryPos !== false) ? '&' : '?';

        return $path . $separator . 'v=' . $version;
    }

    /**
     * Generate versioned asset URL for admin assets.
     *
     * @param string $path The root-relative admin asset path.
     * @return string The path with a ?v= cache-busting query string appended.
     */
    public static function adminUrl(string $path): string
    {
        return self::url($path);
    }
}
