<?php

declare(strict_types=1);

namespace NewSite\User;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Security\IpService;

/**
 * User account management service.
 *
 * Security: All database access uses prepared statements with bound parameters.
 * IP addresses are stored through IpService::gdprStore() for GDPR compliance.
 * Tokens are hashed with SHA-256 before storage; only hashes are persisted.
 */
final class UserService
{
    public static function getByNickname(string $nickname): array|false
    {
        $db = DatabaseManager::getReadConnection();
        $nickname = ltrim($nickname, '@');
        $stmt = $db->prepare("SELECT * FROM site_users WHERE nickname = ? AND is_active = 1");
        $stmt->execute([$nickname]);
        return $stmt->fetch();
    }

    public static function getById(int $userId): array|false
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare("SELECT * FROM site_users WHERE id = ?");
        $stmt->execute([$userId]);
        return $stmt->fetch();
    }

    public static function getByEmail(string $email): array|false
    {
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT * FROM site_users WHERE email = ? LIMIT 1");
        $stmt->execute([$email]);
        return $stmt->fetch();
    }

    /**
     * Look up a user by their linked Google account ID.
     */
    public static function getByGoogleId(string $googleId): array|false
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare("SELECT * FROM site_users WHERE google_id = ? LIMIT 1");
        $stmt->execute([$googleId]);
        return $stmt->fetch();
    }

    /**
     * Link a Google account ID to an existing user.
     */
    public static function linkGoogleId(int $userId, string $googleId): void
    {
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("UPDATE site_users SET google_id = ? WHERE id = ?");
        $stmt->execute([$googleId, $userId]);
    }

    public static function generateUniqueNickname(\PDO $db): string
    {
        $base = 'User';
        for ($i = 0; $i < 5; $i++) {
            $suffix = str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);
            $nickname = $base . $suffix;
            $stmt = $db->prepare("SELECT id FROM site_users WHERE nickname = ? LIMIT 1");
            $stmt->execute([$nickname]);
            if (!$stmt->fetch()) {
                return $nickname;
            }
        }
        return $base . bin2hex(random_bytes(3));
    }

    public static function needsLegalAcceptance(array $user): bool
    {
        $termsAccepted = !empty($user['terms_accepted_at']);
        $privacyAccepted = !empty($user['privacy_accepted_at']);
        return !$termsAccepted || !$privacyAccepted;
    }

    public static function create(
        string $email,
        ?string $ip = null,
        bool $acceptLegal = false,
    ): int {
        $db = DatabaseManager::getWriteConnection();
        $createdAt = time();
        $storedIp = $ip !== null ? IpService::gdprStore($ip) : null;
        $termsAcceptedAt = $acceptLegal ? $createdAt : null;
        $termsAcceptedIp = $acceptLegal ? $storedIp : null;
        $privacyAcceptedAt = $acceptLegal ? $createdAt : null;
        $privacyAcceptedIp = $acceptLegal ? $storedIp : null;

        $stmt = $db->prepare(
            "INSERT INTO site_users
                (email, created_at, created_ip, terms_accepted_at, terms_accepted_ip, privacy_accepted_at, privacy_accepted_ip)
             VALUES (?, ?, ?, ?, ?, ?, ?)"
        );
        $stmt->execute([
            $email,
            $createdAt,
            $ip,
            $termsAcceptedAt,
            $termsAcceptedIp,
            $privacyAcceptedAt,
            $privacyAcceptedIp,
        ]);
        $userId = DbHelper::lastInsertId($db, 'site_users');

        // Send welcome messages
        $welcomeDb = DatabaseManager::getReadConnection();
        $welcomeStmt = $welcomeDb->query(
            "SELECT * FROM welcome_messages WHERE is_active = 1 ORDER BY created_at DESC"
        );
        $welcomeMessages = $welcomeStmt->fetchAll();

        foreach ($welcomeMessages as $message) {
            $msgStmt = $db->prepare(
                "INSERT INTO user_messages
                    (user_id, subject, body, sender_type, sender_label, message_type, is_read, created_at)
                 VALUES (?, ?, ?, 'admin', 'System', 'notice', 0, ?)"
            );
            $msgStmt->execute([
                $userId,
                $message['subject'] ?? 'Welcome',
                $message['body'] ?? '',
                time(),
            ]);
        }

        return $userId;
    }

    public static function recordLegalAcceptance(int $userId, ?string $ip = null): void
    {
        $db = DatabaseManager::getWriteConnection();
        $storedIp = $ip !== null ? IpService::gdprStore($ip) : null;
        $now = time();

        $stmt = $db->prepare(
            "UPDATE site_users
             SET terms_accepted_at   = COALESCE(terms_accepted_at, ?),
                 terms_accepted_ip   = COALESCE(terms_accepted_ip, ?),
                 privacy_accepted_at = COALESCE(privacy_accepted_at, ?),
                 privacy_accepted_ip = COALESCE(privacy_accepted_ip, ?)
             WHERE id = ?"
        );
        $stmt->execute([$now, $storedIp, $now, $storedIp, $userId]);
    }

    public static function createLoginToken(
        int $userId,
        string $ip,
        string $userAgent,
        int $ttlSeconds = 1800,
    ): string {
        $token = bin2hex(random_bytes(32));
        $hash = hash('sha256', $token);
        $expiresAt = time() + $ttlSeconds;

        $db = DatabaseManager::getWriteConnection();
        $storedIp = IpService::gdprStore($ip);

        $stmt = $db->prepare(
            "INSERT INTO user_login_tokens
                (user_id, token_hash, expires_at, created_at, ip_address, user_agent)
             VALUES (?, ?, ?, ?, ?, ?)"
        );
        $stmt->execute([$userId, $hash, $expiresAt, time(), $storedIp, $userAgent]);

        return $token;
    }

    public static function verifyLoginToken(string $token): ?int
    {
        $hash = hash('sha256', $token);
        $db = DatabaseManager::getWriteConnection();

        $stmt = $db->prepare("SELECT * FROM user_login_tokens WHERE token_hash = ? LIMIT 1");
        $stmt->execute([$hash]);
        $row = $stmt->fetch();

        $isValid = $row && empty($row['used_at']) && (int) $row['expires_at'] >= time();

        if ($isValid) {
            $stmt = $db->prepare("UPDATE user_login_tokens SET used_at = ? WHERE id = ?");
            $stmt->execute([time(), $row['id']]);
            return (int) $row['user_id'];
        }

        return null;
    }

    public static function resolveDisplayName(array $user, string $fallback = 'there'): string
    {
        $name = trim((string) ($user['display_name'] ?? ''));
        if ($name === '') {
            $name = trim((string) ($user['nickname'] ?? ''));
        }
        return $name !== '' ? $name : $fallback;
    }
}
