<?php

declare(strict_types=1);

namespace NewSite\User;

use NewSite\Database\DatabaseManager;

/**
 * Service for user messages and notification counts.
 *
 * Security: All database access uses prepared statements with bound parameters.
 * No user-supplied values are interpolated into queries.
 */
final class UserMessageService
{
    public static function getMessages(int $userId, int $limit = 5): array
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare(
            "SELECT * FROM user_messages
             WHERE user_id = ?
               AND (message_type IS NULL OR message_type != 'chat')
             ORDER BY created_at DESC
             LIMIT ?"
        );
        $stmt->bindValue(1, $userId, \PDO::PARAM_INT);
        $stmt->bindValue(2, $limit, \PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    public static function getWelcomeMessages(): array
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->query(
            "SELECT * FROM welcome_messages WHERE is_active = 1 ORDER BY created_at DESC"
        );
        return $stmt->fetchAll();
    }

    public static function getUnreadCount(int $userId): int
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare(
            "SELECT COUNT(*) FROM user_messages
             WHERE user_id = ?
               AND is_read = 0
               AND (message_type IS NULL OR message_type != 'chat')"
        );
        $stmt->execute([$userId]);
        return (int) $stmt->fetchColumn();
    }

    public static function getAdminUnreadTicketCount(): int
    {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->query(
            "SELECT COUNT(*) FROM user_messages
             WHERE (message_type = 'ticket' OR LOWER(subject) LIKE '%ticket%')
               AND sender_type = 'user'
               AND (admin_read = 0 OR admin_read IS NULL)"
        );
        return (int) $stmt->fetchColumn();
    }
}
