<?php

declare(strict_types=1);

namespace NewSite\User;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\User\FriendService;
use NewSite\User\UserService;

/**
 * User privacy-settings management.
 *
 * Security: All database access uses prepared statements with bound parameters.
 * Visibility values are validated against an allowlist before storage.
 */
final class PrivacyService
{
    public static function getDefaults(): array
    {
        return [
            'display_name'  => 'everyone',
            'nickname'      => 'everyone',
            'profile_photo' => 'everyone',
            'description'   => 'everyone',
            'website'       => 'everyone',
            'company'       => 'friends',
            'email'         => 'hidden',
            'phone'         => 'hidden',
            'messenger'     => 'friends',
            'gender'        => 'friends',
            'age'           => 'friends',
            'address'       => 'hidden',
            'country'       => 'friends',
            'postal_code'   => 'hidden',
        ];
    }

    public static function getUserSettings(int $userId): array
    {
        $db = DatabaseManager::getWriteConnection();
        $defaults = self::getDefaults();

        $stmt = $db->prepare(
            "SELECT field_name, visibility FROM user_privacy_settings WHERE user_id = ?"
        );
        $stmt->execute([$userId]);
        $settings = $stmt->fetchAll();

        foreach ($settings as $setting) {
            $defaults[$setting['field_name']] = $setting['visibility'];
        }

        return $defaults;
    }

    public static function setUserSetting(int $userId, string $fieldName, string $visibility): bool
    {
        $db = DatabaseManager::getWriteConnection();
        $now = DbHelper::nowString();

        $validVisibilities = ['everyone', 'friends', 'hidden'];
        if (!in_array($visibility, $validVisibilities, true)) {
            return false;
        }

        $stmt = $db->prepare(
            "INSERT INTO user_privacy_settings (user_id, field_name, visibility, updated_at)
             VALUES (?, ?, ?, ?)
             ON CONFLICT (user_id, field_name)
             DO UPDATE SET visibility = EXCLUDED.visibility,
                           updated_at = EXCLUDED.updated_at"
        );
        return $stmt->execute([$userId, $fieldName, $visibility, $now]);
    }

    /**
     * Determine whether $viewerId may see a specific profile field of $profileUserId.
     */
    public static function canViewField(int $viewerId, int $profileUserId, string $fieldName): bool
    {
        if ($viewerId > 0 && $viewerId === $profileUserId) {
            return true;
        }
        if ($viewerId > 0 && FriendService::isBlockedEitherWay($viewerId, $profileUserId)) {
            return false;
        }

        $privacy = self::getUserSettings($profileUserId);
        $visibility = $privacy[$fieldName] ?? 'hidden';

        $canView = false;
        if ($visibility === 'everyone') {
            $canView = true;
        } elseif ($visibility === 'friends') {
            $canView = $viewerId > 0 && FriendService::areFriends($viewerId, $profileUserId);
        }

        return $canView;
    }

    /**
     * Return the visible profile data for $profileUserId as seen by $viewerId.
     *
     * @return array<string, mixed>|null
     */
    public static function getVisibleProfileData(int $profileUserId, ?int $viewerId = null): ?array
    {
        $user = UserService::getById($profileUserId);
        if (!$user) {
            return null;
        }

        if ($viewerId !== null && $viewerId > 0 && FriendService::isBlockedEitherWay($viewerId, $profileUserId)) {
            return null;
        }

        $privacy  = self::getUserSettings($profileUserId);
        $isOwner  = $viewerId !== null && $viewerId > 0 && $viewerId === $profileUserId;
        $isFriend = $viewerId !== null && $viewerId > 0 && FriendService::areFriends($viewerId, $profileUserId);

        $data = [
            'id'           => $user['id'],
            'member_since' => $user['created_at'],
        ];

        $profileFields = [
            'display_name', 'nickname', 'profile_photo', 'description', 'website',
            'company', 'email', 'phone', 'gender', 'gender_custom', 'age',
            'address', 'country', 'postal_code',
        ];

        foreach ($profileFields as $field) {
            $visibility = $privacy[$field] ?? 'hidden';
            $canView    = $isOwner
                || $visibility === 'everyone'
                || ($visibility === 'friends' && $isFriend);

            $data[$field]               = $canView ? ($user[$field] ?? null) : null;
            $data[$field . '_visible']  = $canView;
        }

        return $data;
    }
}
