<?php

declare(strict_types=1);

namespace NewSite\User;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Push\PushNotification;

/**
 * FriendService — handles friendship, blocking, and friend-list operations.
 * Friend-request lifecycle is in FriendRequestHelper; profile visibility is in PrivacyService.
 *
 * Security notes:
 * - Every database query uses prepared statements with bound parameters to prevent SQL injection.
 * - Block checks are enforced before any friendship or profile-visibility operation.
 * - User IDs are validated (non-zero, non-self) at each public entry point.
 * - Privacy settings are respected through PrivacyService lookups before exposing profile data.
 */
final class FriendService
{
    /**
     * Check whether two users are friends.
     */
    public static function areFriends(int $userId1, int $userId2): bool
    {
        if ($userId1 === 0 || $userId2 === 0 || $userId1 === $userId2) {
            return false;
        }
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT COUNT(*) FROM friendships
             WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)"
        );
        $stmt->execute([$userId1, $userId2, $userId2, $userId1]);
        return $stmt->fetchColumn() > 0;
    }

    /**
     * Check whether $userId has blocked $blockedUserId.
     */
    public static function hasBlocked(int $userId, int $blockedUserId): bool
    {
        if ($userId === 0 || $blockedUserId === 0) {
            return false;
        }
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT COUNT(*) FROM user_blocks WHERE user_id = ? AND blocked_user_id = ?"
        );
        $stmt->execute([$userId, $blockedUserId]);
        return $stmt->fetchColumn() > 0;
    }

    /**
     * Check whether either user has blocked the other.
     */
    public static function isBlockedEitherWay(int $userId1, int $userId2): bool
    {
        return self::hasBlocked($userId1, $userId2) || self::hasBlocked($userId2, $userId1);
    }

    /**
     * Send a friend request (full workflow with precondition checks and auto-accept).
     *
     * @return array<string, mixed>
     */
    public static function sendRequest(int $fromUserId, int $toUserId, ?string $message = null): array
    {
        $response = ['success' => false, 'error' => 'Invalid users'];

        $error = FriendRequestHelper::getPreconditionError($fromUserId, $toUserId);
        if ($error !== null) {
            $response['error'] = $error;
        } else {
            $db       = DatabaseManager::getWriteConnection();
            $existing = FriendRequestHelper::getRequestRecord($db, $fromUserId, $toUserId);

            if ($existing && ($existing['status'] ?? '') === 'pending') {
                $response = ['success' => false, 'error' => 'Request already sent'];
            } else {
                $autoAcceptResponse = FriendRequestHelper::maybeAutoAcceptReverse($db, $toUserId, $fromUserId);
                if ($autoAcceptResponse !== null) {
                    $response = $autoAcceptResponse;
                } else {
                    $response = FriendRequestHelper::createOrRefresh($db, $fromUserId, $toUserId, $message, $existing);
                }
            }
        }

        return $response;
    }

    /**
     * Accept or decline a pending friend request.
     *
     * @return array<string, mixed>
     */
    public static function respondToRequest(int $requestId, int $userId, bool $accept): array
    {
        $db  = DatabaseManager::getWriteConnection();
        $now = DbHelper::nowString();

        $stmt = $db->prepare(
            "SELECT * FROM friend_requests WHERE id = ? AND to_user_id = ? AND status = 'pending'"
        );
        $stmt->execute([$requestId, $userId]);
        $request = $stmt->fetch();

        if (!$request) {
            return ['success' => false, 'error' => 'Request not found'];
        }

        if ($accept) {
            $stmt = $db->prepare(
                "INSERT INTO friendships (user_id, friend_id, created_at)
                 VALUES (?, ?, ?) ON CONFLICT DO NOTHING"
            );
            $stmt->execute([$request['from_user_id'], $request['to_user_id'], $now]);
            $stmt->execute([$request['to_user_id'], $request['from_user_id'], $now]);

            $stmt = $db->prepare(
                "UPDATE friend_requests SET status = 'accepted', updated_at = ? WHERE id = ?"
            );
            $stmt->execute([$now, $requestId]);

            $acceptingUser     = UserService::getById($userId);
            $acceptingNickname = is_array($acceptingUser)
                ? ($acceptingUser['nickname'] ?? $acceptingUser['display_name'] ?? 'Someone')
                : 'Someone';

            PushNotification::friendAccepted((int) $request['from_user_id'], $acceptingNickname);

            return ['success' => true, 'message' => 'Friend request accepted'];
        }

        $stmt = $db->prepare(
            "UPDATE friend_requests SET status = 'declined', updated_at = ? WHERE id = ?"
        );
        $stmt->execute([$now, $requestId]);

        return ['success' => true, 'message' => 'Friend request declined'];
    }

    /**
     * Cancel a pending outgoing friend request.
     */
    public static function cancelRequest(int $requestId, int $userId): bool
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "DELETE FROM friend_requests WHERE id = ? AND from_user_id = ? AND status = 'pending'"
        );
        $stmt->execute([$requestId, $userId]);

        return $stmt->rowCount() > 0;
    }

    /**
     * Remove an existing friendship (both directions).
     */
    public static function removeFriend(int $userId, int $friendId): bool
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "DELETE FROM friendships
             WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)"
        );
        $stmt->execute([$userId, $friendId, $friendId, $userId]);

        return $stmt->rowCount() > 0;
    }

    /**
     * Block another user — removes friendship and any pending requests, then inserts block record.
     */
    public static function blockUser(int $userId, int $blockedUserId): bool
    {
        if ($userId === 0 || $blockedUserId === 0 || $userId === $blockedUserId) {
            return false;
        }

        $db  = DatabaseManager::getWriteConnection();
        $now = DbHelper::nowString();

        self::removeFriend($userId, $blockedUserId);

        $stmt = $db->prepare(
            "DELETE FROM friend_requests
             WHERE (from_user_id = ? AND to_user_id = ?) OR (from_user_id = ? AND to_user_id = ?)"
        );
        $stmt->execute([$userId, $blockedUserId, $blockedUserId, $userId]);

        $stmt = $db->prepare(
            "INSERT INTO user_blocks (user_id, blocked_user_id, created_at)
             VALUES (?, ?, ?) ON CONFLICT DO NOTHING"
        );
        $stmt->execute([$userId, $blockedUserId, $now]);

        return true;
    }

    /**
     * Remove a block on another user.
     */
    public static function unblockUser(int $userId, int $blockedUserId): bool
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "DELETE FROM user_blocks WHERE user_id = ? AND blocked_user_id = ?"
        );
        $stmt->execute([$userId, $blockedUserId]);

        return $stmt->rowCount() > 0;
    }

    /**
     * Get all pending incoming friend requests for a user.
     *
     * @return array<int, array<string, mixed>>
     */
    public static function getPendingRequests(int $userId): array
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT fr.*, su.display_name, su.nickname, su.profile_photo
             FROM friend_requests fr
             JOIN site_users su ON su.id = fr.from_user_id
             WHERE fr.to_user_id = ? AND fr.status = 'pending'
             ORDER BY fr.created_at DESC"
        );
        $stmt->execute([$userId]);

        return $stmt->fetchAll();
    }

    /**
     * Get all pending outgoing friend requests for a user.
     *
     * @return array<int, array<string, mixed>>
     */
    public static function getSentRequests(int $userId): array
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT fr.*, su.display_name, su.nickname, su.profile_photo
             FROM friend_requests fr
             JOIN site_users su ON su.id = fr.to_user_id
             WHERE fr.from_user_id = ? AND fr.status = 'pending'
             ORDER BY fr.created_at DESC"
        );
        $stmt->execute([$userId]);

        return $stmt->fetchAll();
    }

    /**
     * Count pending incoming friend requests for a user.
     */
    public static function getPendingCount(int $userId): int
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT COUNT(*) FROM friend_requests WHERE to_user_id = ? AND status = 'pending'"
        );
        $stmt->execute([$userId]);

        return (int) $stmt->fetchColumn();
    }

    /**
     * Get the friend list for a user with optional pagination.
     *
     * @return array<int, array<string, mixed>>
     */
    public static function getList(int $userId, ?int $limit = null, int $offset = 0): array
    {
        $db  = DatabaseManager::getWriteConnection();
        $sql = "SELECT su.id, su.display_name, su.nickname, su.profile_photo,
                       f.created_at AS friends_since
                FROM friendships f
                JOIN site_users su ON su.id = f.friend_id
                WHERE f.user_id = ? AND su.is_active = 1
                ORDER BY su.display_name ASC, su.nickname ASC";

        if ($limit !== null) {
            $sql .= " LIMIT " . $limit . " OFFSET " . $offset;
        }

        $stmt = $db->prepare($sql);
        $stmt->execute([$userId]);

        return $stmt->fetchAll();
    }

    /**
     * Count total friends for a user.
     */
    public static function getCount(int $userId): int
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT COUNT(*) FROM friendships WHERE user_id = ?");
        $stmt->execute([$userId]);

        return (int) $stmt->fetchColumn();
    }

    /**
     * Get mutual friends between two users.
     *
     * @return array<int, array<string, mixed>>
     */
    public static function getMutual(int $userId1, int $userId2, ?int $limit = null): array
    {
        $db  = DatabaseManager::getWriteConnection();
        $sql = "SELECT su.id, su.display_name, su.nickname, su.profile_photo
                FROM friendships f1
                JOIN friendships f2 ON f1.friend_id = f2.friend_id
                JOIN site_users su ON su.id = f1.friend_id
                WHERE f1.user_id = ? AND f2.user_id = ? AND su.is_active = 1
                ORDER BY su.display_name ASC";

        if ($limit !== null) {
            $sql .= " LIMIT " . $limit;
        }

        $stmt = $db->prepare($sql);
        $stmt->execute([$userId1, $userId2]);

        return $stmt->fetchAll();
    }

    /**
     * Count mutual friends between two users.
     */
    public static function getMutualCount(int $userId1, int $userId2): int
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT COUNT(DISTINCT f1.friend_id)
             FROM friendships f1
             JOIN friendships f2 ON f1.friend_id = f2.friend_id
             WHERE f1.user_id = ? AND f2.user_id = ?"
        );
        $stmt->execute([$userId1, $userId2]);

        return (int) $stmt->fetchColumn();
    }

    /**
     * Search active users by nickname or display name, excluding blocked users.
     *
     * @return array<int, array<string, mixed>>
     */
    public static function searchUsers(string $query, ?int $excludeUserId = null, int $limit = 20): array
    {
        $db    = DatabaseManager::getWriteConnection();
        $query = ltrim($query, '@');
        $queryLike = '%' . $query . '%';

        $hasSearchableColumn = false;

        try {
            $checkStmt = $db->query(
                "SELECT column_name FROM information_schema.columns
                 WHERE table_name = 'site_users' AND column_name = 'is_searchable'"
            );
            $hasSearchableColumn = ($checkStmt->fetch() !== false);
        } catch (\Exception $e) {
            // Column check failed — proceed without the filter.
        }

        if ($hasSearchableColumn) {
            $sql = "SELECT id, display_name, nickname, profile_photo
                    FROM site_users
                    WHERE is_active = 1
                      AND (is_searchable IS NULL OR is_searchable = 1)
                      AND (nickname ILIKE ? OR display_name ILIKE ?)";
        } else {
            $sql = "SELECT id, display_name, nickname, profile_photo
                    FROM site_users
                    WHERE is_active = 1
                      AND (nickname ILIKE ? OR display_name ILIKE ?)";
        }

        $params = [$queryLike, $queryLike];

        if ($excludeUserId !== null) {
            $sql .= " AND id != ?";
            $params[] = $excludeUserId;

            $sql .= " AND id NOT IN (SELECT blocked_user_id FROM user_blocks WHERE user_id = ?)";
            $params[] = $excludeUserId;

            $sql .= " AND id NOT IN (SELECT user_id FROM user_blocks WHERE blocked_user_id = ?)";
            $params[] = $excludeUserId;
        }

        $sql .= " ORDER BY CASE WHEN nickname ILIKE ? THEN 0 ELSE 1 END, display_name ASC LIMIT ?";
        $params[] = $query . '%';
        $params[] = $limit;

        $stmt = $db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll();
    }

    /**
     * Get all users blocked by a given user.
     *
     * @return array<int, array<string, mixed>>
     */
    public static function getBlockedUsers(int $userId): array
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT su.id, su.display_name, su.nickname, su.profile_photo,
                    ub.created_at AS blocked_at
             FROM user_blocks ub
             JOIN site_users su ON su.id = ub.blocked_user_id
             WHERE ub.user_id = ?
             ORDER BY ub.created_at DESC"
        );
        $stmt->execute([$userId]);

        return $stmt->fetchAll();
    }

    /**
     * Determine the relationship status between two users.
     *
     * @return string One of: self, blocked, blocked_by, friends, request_sent, request_received, none.
     */
    public static function getStatus(int $userId, int $otherUserId): string
    {
        if ($userId === 0 || $otherUserId === 0 || $userId === $otherUserId) {
            return 'self';
        }

        $status = 'none';
        if (self::hasBlocked($userId, $otherUserId)) {
            $status = 'blocked';
        } elseif (self::hasBlocked($otherUserId, $userId)) {
            $status = 'blocked_by';
        } elseif (self::areFriends($userId, $otherUserId)) {
            $status = 'friends';
        } else {
            $db   = DatabaseManager::getWriteConnection();
            $stmt = $db->prepare(
                "SELECT id, status FROM friend_requests
                 WHERE from_user_id = ? AND to_user_id = ? AND status = 'pending'"
            );
            $stmt->execute([$userId, $otherUserId]);
            if ($stmt->fetch()) {
                $status = 'request_sent';
            } else {
                $stmt->execute([$otherUserId, $userId]);
                if ($stmt->fetch()) {
                    $status = 'request_received';
                }
            }
        }

        return $status;
    }
}
