<?php

declare(strict_types=1);

namespace NewSite\User;

use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Push\PushNotification;

/**
 * FriendRequestHelper — friend-request lifecycle: precondition checks,
 * creation / refresh, auto-accept, and push notifications.
 *
 * Security notes:
 * - Every database query uses prepared statements with bound parameters to prevent SQL injection.
 * - Block and friendship checks are delegated to FriendService before any mutation.
 * - User IDs are validated (non-zero, non-self) at each public entry point.
 */
final class FriendRequestHelper
{
    /**
     * Fetch an existing friend-request record (any status) between two users.
     *
     * @return array{id: int, status: string}|null
     */
    public static function getRequestRecord(\PDO $db, int $fromUserId, int $toUserId): ?array
    {
        $stmt = $db->prepare(
            "SELECT id, status FROM friend_requests WHERE from_user_id = ? AND to_user_id = ?"
        );
        $stmt->execute([$fromUserId, $toUserId]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    /**
     * Fetch a pending incoming request from $requesterUserId to $recipientUserId.
     *
     * @return array{id: int}|null
     */
    public static function getPendingIncoming(\PDO $db, int $requesterUserId, int $recipientUserId): ?array
    {
        $stmt = $db->prepare(
            "SELECT id FROM friend_requests
             WHERE from_user_id = ? AND to_user_id = ? AND status = 'pending'"
        );
        $stmt->execute([$requesterUserId, $recipientUserId]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    /**
     * Insert or update a friend request to pending status.
     */
    public static function savePending(
        \PDO $db,
        int $fromUserId,
        int $toUserId,
        ?string $message,
        string $now,
        ?int $existingId = null
    ): void {
        if ($existingId !== null && $existingId > 0) {
            $stmt = $db->prepare(
                "UPDATE friend_requests SET status = 'pending', message = ?, updated_at = ? WHERE id = ?"
            );
            $stmt->execute([$message, $now, $existingId]);
            return;
        }

        $stmt = $db->prepare(
            "INSERT INTO friend_requests (from_user_id, to_user_id, message, status, created_at, updated_at)
             VALUES (?, ?, ?, 'pending', ?, ?)"
        );
        $stmt->execute([$fromUserId, $toUserId, $message, $now, $now]);
    }

    /**
     * Send a push notification for a new friend request.
     */
    public static function notifyRequest(int $fromUserId, int $toUserId): void
    {
        $fromUser     = UserService::getById($fromUserId);
        $fromNickname = is_array($fromUser)
            ? ($fromUser['nickname'] ?? $fromUser['display_name'] ?? 'Someone')
            : 'Someone';

        PushNotification::friendRequest($toUserId, $fromNickname);
    }

    /**
     * Validate preconditions before sending a friend request.
     */
    public static function getPreconditionError(int $fromUserId, int $toUserId): ?string
    {
        $error = null;
        if ($fromUserId === 0 || $toUserId === 0 || $fromUserId === $toUserId) {
            $error = 'Invalid users';
        } elseif (FriendService::isBlockedEitherWay($fromUserId, $toUserId)) {
            $error = 'Cannot send request';
        } elseif (FriendService::areFriends($fromUserId, $toUserId)) {
            $error = 'Already friends';
        }

        return $error;
    }

    /**
     * If the other user already sent a pending request, auto-accept it instead.
     *
     * @return array<string, mixed>|null
     */
    public static function maybeAutoAcceptReverse(\PDO $db, int $requesterUserId, int $recipientUserId): ?array
    {
        $theirRequest = self::getPendingIncoming($db, $requesterUserId, $recipientUserId);
        if ($theirRequest) {
            return FriendService::respondToRequest((int) $theirRequest['id'], $recipientUserId, true);
        }

        return null;
    }

    /**
     * Persist a new (or refreshed) friend request and send notifications.
     *
     * @return array<string, mixed>
     */
    public static function createOrRefresh(
        \PDO $db,
        int $fromUserId,
        int $toUserId,
        ?string $message,
        ?array $existing
    ): array {
        $response = ['success' => false, 'error' => 'Database error'];

        try {
            self::savePending(
                $db,
                $fromUserId,
                $toUserId,
                $message,
                DbHelper::nowString(),
                $existing !== null ? (int) $existing['id'] : null
            );
            self::notifyRequest($fromUserId, $toUserId);
            LogService::add('friend_request', 'Friend request sent', "From: $fromUserId, To: $toUserId");
            $response = ['success' => true];
        } catch (\PDOException $e) {
            error_log('Friend request DB error: ' . $e->getMessage());
        }

        return $response;
    }
}
