<?php

declare(strict_types=1);

namespace NewSite\Upload;

use NewSite\Settings\SettingsService;

/**
 * File upload validation and storage service.
 *
 * Security: MIME types are verified via finfo and magic-byte signatures;
 * image dimensions are bounded; uploaded files are stored with random names;
 * .htaccess protection is applied to upload directories; only whitelisted
 * extensions and MIME types are accepted; PHP execution is blocked in upload dirs.
 */
final class UploadService
{
    private const MIME_IMAGE_JPEG = 'image/jpeg';
    private const UPLOAD_ERROR_NO_FILE = 'No file uploaded';

    public static function getImageLimits(): array
    {
        $maxWidth = max(1, (int)(SettingsService::get('upload_max_image_width', '4096') ?? '4096'));
        $maxHeight = max(1, (int)(SettingsService::get('upload_max_image_height', '4096') ?? '4096'));
        return [$maxWidth, $maxHeight];
    }

    public static function isFontSignatureValid(string $path, string $ext): bool
    {
        $handle = @fopen($path, 'rb');
        $isValid = false;
        if ($handle !== false) {
            $signature = fread($handle, 4);
            fclose($handle);
            if ($signature !== false && strlen($signature) >= 4) {
                if ($ext === 'woff') {
                    $isValid = $signature === 'wOFF';
                } elseif ($ext === 'woff2') {
                    $isValid = $signature === 'wOF2';
                } elseif ($ext === 'ttf' || $ext === 'otf') {
                    $isValid = $signature === 'OTTO' || $signature === "\x00\x01\x00\x00" || $signature === 'true' || $signature === 'typ1';
                }
            }
        }
        return $isValid;
    }

    public static function buildValidationResult(bool $success, ?string $error, ?string $mime, ?string $ext): array
    {
        return ['success' => $success, 'error' => $error, 'mime' => $mime, 'ext' => $ext];
    }

    public static function validateMimeType(string $tmpPath, string $ext, array $allowedMap): array
    {
        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($tmpPath);
        $allowedMimes = (array)($allowedMap[$ext] ?? []);
        if (in_array($mime, $allowedMimes, true)) {
            return ['valid' => true, 'mime' => $mime];
        }
        $fontExts = ['woff', 'woff2', 'ttf', 'otf'];
        if (in_array($ext, $fontExts, true) && self::isFontSignatureValid($tmpPath, $ext)) {
            return ['valid' => true, 'mime' => $mime];
        }
        return ['valid' => false, 'mime' => $mime];
    }

    public static function validateImageBounds(string $tmpPath, ?string $mime): ?string
    {
        $error = null;
        $requiresCheck = $mime !== null && strpos($mime, 'image/') === 0 && $mime !== 'image/svg+xml';
        if ($requiresCheck) {
            $imageInfo = @getimagesize($tmpPath);
            if ($imageInfo === false) {
                $error = 'File is not a valid image';
            } else {
                [$maxWidth, $maxHeight] = self::getImageLimits();
                $width = (int)($imageInfo[0] ?? 0);
                $height = (int)($imageInfo[1] ?? 0);
                if ($width > 0 && $height > 0 && ($width > $maxWidth || $height > $maxHeight)) {
                    $error = 'Image dimensions are too large';
                }
            }
        }
        return $error;
    }

    public static function getAllowedImageMap(): array
    {
        return [
            'jpg' => [self::MIME_IMAGE_JPEG], 'jpeg' => [self::MIME_IMAGE_JPEG],
            'png' => ['image/png'], 'gif' => ['image/gif'], 'webp' => ['image/webp'],
        ];
    }

    public static function buildStoreResult(bool $success, ?string $path, ?string $error): array
    {
        return ['success' => $success, 'path' => $path, 'error' => $error];
    }

    public static function ensureProtectedDirectory(string $uploadDir): void
    {
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        $htaccessPath = $uploadDir . '/.htaccess';
        if (!is_file($htaccessPath)) {
            file_put_contents($htaccessPath, "Options -Indexes\n<FilesMatch \"\\.php$\">\n    Deny from all\n</FilesMatch>\nAddType application/octet-stream .php .php5 .phtml");
        }
    }

    public static function storeValidatedImage(string $tmpPath, string $folder, string $extension): array
    {
        $safeFolder = preg_replace('/[^a-z0-9_]/i', '', $folder);
        if (!is_string($safeFolder) || $safeFolder === '') {
            return self::buildStoreResult(false, null, 'Invalid upload directory');
        }
        $normalizedExt = trim($extension) !== '' ? trim($extension) : 'jpg';
        $filename = bin2hex(random_bytes(16)) . '.' . $normalizedExt;
        $uploadDir = dirname(__DIR__, 2) . '/data/' . $safeFolder;
        self::ensureProtectedDirectory($uploadDir);
        $destPath = $uploadDir . '/' . $filename;
        $saved = move_uploaded_file($tmpPath, $destPath);
        return $saved
            ? self::buildStoreResult(true, $safeFolder . '/' . $filename, null)
            : self::buildStoreResult(false, null, 'Failed to save file');
    }

    public static function uploadToDataDirectory(array $file, string $folder, int $maxSize): array
    {
        $tmpPath = (string)($file['tmp_name'] ?? '');
        if ($tmpPath === '' || !is_uploaded_file($tmpPath)) {
            return self::buildStoreResult(false, null, self::UPLOAD_ERROR_NO_FILE);
        }
        $size = (int)($file['size'] ?? 0);
        if ($size > $maxSize) {
            return self::buildStoreResult(false, null, 'File too large. Maximum size is 5MB');
        }
        $validation = self::validateFile($file, self::getAllowedImageMap(), true);
        $result = self::buildStoreResult(false, null, (string)($validation['error'] ?? 'Invalid file'));
        if ($validation['success']) {
            $result = self::storeValidatedImage($tmpPath, $folder, (string)($validation['ext'] ?? ''));
        }
        return $result;
    }

    public static function validateFile(array $file, array $allowedMap, bool $enforceImageDimensions = true): array
    {
        $tmpPath = (string)($file['tmp_name'] ?? '');
        $ext = null;
        $mime = null;
        $error = null;
        $success = false;
        if ($tmpPath === '' || !is_uploaded_file($tmpPath)) {
            $error = self::UPLOAD_ERROR_NO_FILE;
        } else {
            $ext = strtolower(pathinfo((string)($file['name'] ?? ''), PATHINFO_EXTENSION));
            if ($ext === '' || !isset($allowedMap[$ext])) {
                $error = 'Invalid file extension';
            } else {
                $mimeCheck = self::validateMimeType($tmpPath, $ext, $allowedMap);
                $mime = $mimeCheck['mime'];
                if (!$mimeCheck['valid']) {
                    $error = 'Invalid file type';
                } elseif ($enforceImageDimensions) {
                    $error = self::validateImageBounds($tmpPath, $mime);
                }
                if ($error === null) {
                    $success = true;
                }
            }
        }
        return self::buildValidationResult($success, $error, $mime, $ext);
    }

    public static function uploadChatImage(array $file, int $maxSize = 5242880): array
    {
        return self::uploadToDataDirectory($file, 'chat_images', $maxSize);
    }

    public static function uploadForumImage(array $file, int $maxSize = 5242880): array
    {
        return self::uploadToDataDirectory($file, 'forum_images', $maxSize);
    }

    /**
     * Upload a return-instruction image (return label, shipping label, etc.).
     * Stored in data/refund_uploads/. Max 5 MB per file, images only.
     */
    public static function uploadRefundImage(array $file, int $maxSize = 5242880): array
    {
        return self::uploadToDataDirectory($file, 'refund_uploads', $maxSize);
    }
}
