<?php

declare(strict_types=1);

namespace NewSite\Upload;

/**
 * Helper for generating public-facing image URLs from stored file paths.
 *
 * Security notes:
 * - basename() is used to strip directory traversal attempts; only the
 *   filename component is included in the generated URL.
 * - Route prefixes are hard-coded constants to prevent open-redirect or
 *   path-injection attacks.
 * - No inline JavaScript or inline styles are emitted (strict CSP compliance).
 */
final class ImageUrlHelper
{
    /**
     * CHAT_IMAGE_ROUTE_PREFIX = '/chat-image/'
     * PROFILE_PHOTO_ROUTE_PREFIX = '/profile-photo/'
     * These constants are defined in _init.php bootstrap.
     */
    private const CHAT_IMAGE_ROUTE_PREFIX = '/chat-image/';

    /**
     * Generate a public URL for a forum image.
     *
     * @param string|null $imagePath The stored file path of the forum image.
     * @return string|null The public URL, or null if the path is empty.
     */
    public static function getForumImageUrl(?string $imagePath): ?string
    {
        if (empty($imagePath)) {
            return null;
        }
        return '/forum-image/' . basename($imagePath);
    }

    /**
     * Generate a public URL for a chat image.
     *
     * @param string|null $imagePath The stored file path of the chat image.
     * @return string|null The public URL, or null if the path is empty.
     */
    public static function getChatImageUrl(?string $imagePath): ?string
    {
        if (empty($imagePath)) {
            return null;
        }
        return self::CHAT_IMAGE_ROUTE_PREFIX . basename($imagePath);
    }
}
