<?php

declare(strict_types=1);

namespace NewSite\Template;

/**
 * Controlled template renderer — wraps include for PHP templates.
 *
 * Templates are PHP files that output HTML directly. They cannot be
 * autoloaded via PSR-4 because they are not class definitions.
 * This renderer provides a single, auditable inclusion point with
 * optional variable injection so callers use a `use`-imported class
 * instead of raw include/include_once statements.
 *
 * Security:
 * - The caller must sanitize $path before passing it (e.g. allowlist check).
 * - Variables injected via $vars use EXTR_SKIP to prevent overwriting
 *   built-in superglobals or method parameters.
 * - No user input reaches this class directly.
 */
class TemplateRenderer
{
    /** @var array<string, true> Tracks real-paths already rendered via renderOnce() */
    private static array $rendered = [];

    /**
     * Render a template file exactly once per request.
     *
     * Subsequent calls with the same resolved path are silently skipped.
     * Use for page templates, API endpoints, and standalone includes.
     *
     * @param string              $path Full filesystem path to the template
     * @param array<string,mixed> $vars Variables to inject into template scope
     */
    public static function renderOnce(string $path, array $vars = []): void
    {
        $real = realpath($path);
        if ($real === false || isset(self::$rendered[$real])) {
            return;
        }
        self::$rendered[$real] = true;
        self::doRender($path, $vars);
    }

    /**
     * Render a template file, allowing repeated inclusion.
     *
     * Use for section templates that may appear multiple times on a page
     * (e.g. two "text_box" sections on the same page).
     *
     * @param string              $path Full filesystem path to the template
     * @param array<string,mixed> $vars Variables to inject into template scope
     */
    public static function render(string $path, array $vars = []): void
    {
        self::doRender($path, $vars);
    }

    /**
     * Internal: extract variables into an isolated scope and include the template.
     *
     * extract() with EXTR_SKIP is safe here because:
     * 1. $vars originates from get_defined_vars() — fully trusted, no user input.
     * 2. EXTR_SKIP prevents overwriting $path or $vars themselves.
     * 3. PHP functions/constants/classes remain accessible regardless of scope.
     */
    private static function doRender(string $__templatePath, array $__templateVars): void
    {
        if ($__templateVars !== []) {
            // Inject calling scope's variables so the template sees them.
            extract($__templateVars, EXTR_SKIP);
        }
        include $__templatePath;
    }
}
