<?php

declare(strict_types=1);

namespace NewSite\Template;

use NewSite\Settings\SettingsService;
use NewSite\Http\Response;

/**
 * ContentRenderer — template/content rendering utilities.
 *
 * Security: all user-facing output is escaped via Response::escape().
 * URLs are validated before embedding in markup. No raw DB input is
 * rendered without sanitisation.
 */
final class ContentRenderer
{
    private const HTTP_URL_PATTERN = '/^https?:\/\//i';
    private const SITE_FILE_PREFIX = '/site-file/';

    public static function renderEmailTemplate(string $template, array $vars): string
    {
        if ($template === '' || empty($vars)) {
            return $template;
        }
        return str_replace(array_keys($vars), array_values($vars), $template);
    }

    public static function renderMultilinePlaceholder(string $value): string
    {
        $value = trim($value);
        return $value !== '' ? nl2br(Response::escape($value)) : '';
    }

    public static function joinNonEmptyLines(array $parts): string
    {
        $normalized = array_map(static function ($part): string {
            return trim((string)$part);
        }, $parts);
        $filtered = array_values(array_filter($normalized, static function (string $part): bool {
            return $part !== '';
        }));
        return implode("\n", $filtered);
    }

    public static function renderSafeUrlLink(string $url): string
    {
        $url = trim($url);
        if ($url !== '' && filter_var($url, FILTER_VALIDATE_URL) && preg_match(self::HTTP_URL_PATTERN, $url)) {
            $safeUrl = Response::escape($url);
            return '<a href="' . $safeUrl . '" target="_blank" rel="noopener noreferrer">' . $safeUrl . '</a>';
        }
        return Response::escape($url);
    }

    public static function normalizeEasyMediaZipDownloadPath(string $downloadUrl): string
    {
        $adminPrefix = '/admin-file/';
        $sitePrefix = self::SITE_FILE_PREFIX;

        $downloadUrl = trim($downloadUrl);
        if ($downloadUrl === '') {
            return '';
        }
        $downloadUrl = str_replace($adminPrefix, $sitePrefix, $downloadUrl);

        $path = (string)(parse_url($downloadUrl, PHP_URL_PATH) ?? '');
        if ($path === '' && preg_match(self::HTTP_URL_PATTERN, $downloadUrl)) {
            $parts = parse_url($downloadUrl);
            $path = is_array($parts) ? (string)($parts['path'] ?? '') : '';
        }
        if ($path === '' && str_starts_with($downloadUrl, '/')) {
            $path = $downloadUrl;
        }
        if ($path !== '' && strpos($path, $adminPrefix) === 0) {
            $path = $sitePrefix . ltrim(substr($path, strlen($adminPrefix)), '/');
        }

        $decodedPath = rawurldecode($path);
        $isSitePath = $path !== '' && strpos($path, $sitePrefix) === 0;
        $hasTraversal = strpos($decodedPath, "\0") !== false || (bool)preg_match('#(^|/)\.\.(/|$)#', $decodedPath);
        $isZip = (bool)preg_match('/\.zip$/i', $path);

        return ($isSitePath && !$hasTraversal && $isZip) ? $path : '';
    }

    public static function getDefaultEasyMediaAiZipDownloadPath(): string
    {
        $defaultFileName = 'AI_TAG_SERVER_V1.zip';
        $defaultFilePath = dirname(__DIR__, 2) . '/data/admin_uploads/' . $defaultFileName;
        if (!is_file($defaultFilePath)) {
            return '';
        }
        return self::normalizeEasyMediaZipDownloadPath(self::SITE_FILE_PREFIX . $defaultFileName);
    }

    public static function normalizeEasyMediaHelpImagePath(string $imageUrl): string
    {
        $adminPrefix = '/admin-file/';
        $sitePrefix = self::SITE_FILE_PREFIX;

        $imageUrl = trim($imageUrl);
        if ($imageUrl === '') {
            return '';
        }
        $imageUrl = str_replace($adminPrefix, $sitePrefix, $imageUrl);

        $path = (string)(parse_url($imageUrl, PHP_URL_PATH) ?? '');
        if ($path === '' && preg_match(self::HTTP_URL_PATTERN, $imageUrl)) {
            $parts = parse_url($imageUrl);
            $path = is_array($parts) ? (string)($parts['path'] ?? '') : '';
        }
        if ($path === '' && str_starts_with($imageUrl, '/')) {
            $path = $imageUrl;
        }
        if ($path !== '' && strpos($path, $adminPrefix) === 0) {
            $path = $sitePrefix . ltrim(substr($path, strlen($adminPrefix)), '/');
        }

        $decodedPath = rawurldecode($path);
        $isSitePath = $path !== '' && strpos($path, $sitePrefix) === 0;
        $hasTraversal = strpos($decodedPath, "\0") !== false || (bool)preg_match('#(^|/)\.\.(/|$)#', $decodedPath);
        $isImage = (bool)preg_match('/\.(jpg|jpeg|png|gif|webp|avif|svg)$/i', $path);

        return ($isSitePath && !$hasTraversal && $isImage) ? $path : '';
    }

    public static function getEasyMediaHelpImagePaths(int $maxImages = 12): array
    {
        $maxImages = max(1, min(20, $maxImages));
        $paths = [];
        for ($i = 1; $i <= $maxImages; $i++) {
            $key = 'easy_media_ai_help_image_' . $i;
            $normalized = self::normalizeEasyMediaHelpImagePath((string)SettingsService::get($key, ''));
            if ($normalized !== '' && !in_array($normalized, $paths, true)) {
                $paths[] = $normalized;
            }
        }
        return $paths;
    }

    public static function ensureEasyMediaAiServerQuickStartSection(string $content): string
    {
        $updatedContent = $content;
        $shouldPatch = $content !== ''
            && strpos($content, 'id="easyMediaApp"') !== false
            && stripos($content, 'AI Server Quick Start') === false;

        if ($shouldPatch) {
            $aboutContentPos = strpos($content, '<div class="em-about-content">');
            if ($aboutContentPos !== false) {
                $quickStartMarkup = "                <h4 class=\"em-about-heading em-about-heading-cyan\">🤖 AI Server Quick Start</h4>\n"
                    . "                <ul class=\"em-about-list\">\n"
                    . "                    <li><strong>Start here:</strong> Open <code>ai_tag_server\\windows</code> and double-click <code>launcher.bat</code>.</li>\n"
                    . "                    <li><strong>If Python is missing:</strong> Install Python (Microsoft Store or python.org), then run <code>launcher.bat</code> again.</li>\n"
                    . "                    <li><strong>Tip:</strong> Keep the launcher window open while using AI features in Easy Media.</li>\n"
                    . "                </ul>";

                $insertionNeedles = [
                    '<h4 class="em-about-heading em-about-heading-purple">✨ Features</h4>',
                    '<h4 class="em-about-heading em-about-heading-purple">Features</h4>',
                    '<div class="em-about-footer">',
                    '<div class="modal-buttons em-modal-footer">'
                ];

                foreach ($insertionNeedles as $needle) {
                    $insertPos = strpos($content, $needle, $aboutContentPos);
                    if ($insertPos !== false) {
                        $updatedContent = substr_replace($content, $quickStartMarkup . "\n\n", $insertPos, 0);
                        break;
                    }
                }
            }
        }

        return $updatedContent;
    }

    public static function renderPageContentPlaceholders(string $content): string
    {
        if ($content === '') {
            return $content;
        }

        $content = self::ensureEasyMediaAiServerQuickStartSection($content);

        $themeSiteName = trim((string)SettingsService::getTheme('site_name', ''));
        $siteName = trim((string)SettingsService::get('site_name', $themeSiteName));
        $siteUrl = self::getBaseUrl();
        $contactEmail = trim((string)SettingsService::get('contact_email', ''));
        $contactName = trim((string)SettingsService::get('contact_name', $siteName));
        $contactName = $contactName !== '' ? $contactName : $siteName;
        $contactPhone = trim((string)SettingsService::get('contact_phone', ''));
        $contactAddress = trim((string)SettingsService::get('contact_address', ''));
        $contactAddressHtml = self::renderMultilinePlaceholder($contactAddress);

        $legalEntityName = trim((string)SettingsService::get('legal_entity_name', $contactName));
        $legalEntityName = $legalEntityName !== '' ? $legalEntityName : $contactName;
        $legalLegalForm = trim((string)SettingsService::get('legal_legal_form', ''));
        $legalRepresentative = trim((string)SettingsService::get('legal_representative', ''));
        $legalStreet = trim((string)SettingsService::get('legal_street', ''));
        $legalPostalCode = trim((string)SettingsService::get('legal_postal_code', ''));
        $legalCity = trim((string)SettingsService::get('legal_city', ''));
        $legalCountry = trim((string)SettingsService::get('legal_country', ''));
        $legalRegisterCourt = trim((string)SettingsService::get('legal_register_court', ''));
        $legalRegisterNumber = trim((string)SettingsService::get('legal_register_number', ''));
        $legalVatId = trim((string)SettingsService::get('legal_vat_id', ''));
        $legalBusinessId = trim((string)SettingsService::get('legal_business_id', ''));
        $legalEditorName = trim((string)SettingsService::get('legal_editor_name', ''));
        $legalEditorAddress = trim((string)SettingsService::get('legal_editor_address', ''));
        $legalEditorAddress = $legalEditorAddress !== '' ? $legalEditorAddress : $contactAddress;
        $legalProfessionName = trim((string)SettingsService::get('legal_profession_name', ''));
        $legalProfessionalTitle = trim((string)SettingsService::get('legal_professional_title', ''));
        $legalProfessionalTitleCountry = trim((string)SettingsService::get('legal_professional_title_country', ''));
        $legalProfessionalRules = trim((string)SettingsService::get('legal_professional_rules', ''));
        $legalProfessionalRulesUrl = trim((string)SettingsService::get('legal_professional_rules_url', ''));
        $legalChamber = trim((string)SettingsService::get('legal_chamber', ''));
        $legalSupervisoryAuthority = trim((string)SettingsService::get('legal_supervisory_authority', ''));
        $legalOdrUrl = trim((string)SettingsService::get('legal_odr_url', 'https://ec.europa.eu/consumers/odr'));
        $legalDisputeParticipation = trim((string)SettingsService::get('legal_dispute_participation', ''));
        $legalHostingProviderName = trim((string)SettingsService::get('legal_hosting_provider_name', ''));
        $legalHostingProviderAddress = trim((string)SettingsService::get('legal_hosting_provider_address', ''));

        $legalCityLine = trim($legalPostalCode . ' ' . $legalCity);
        $legalFullAddressRaw = self::joinNonEmptyLines([$legalStreet, $legalCityLine, $legalCountry]);
        $legalFullAddressHtml = self::renderMultilinePlaceholder($legalFullAddressRaw);
        $legalEditorAddressHtml = self::renderMultilinePlaceholder($legalEditorAddress);
        $legalHostingProviderAddressHtml = self::renderMultilinePlaceholder($legalHostingProviderAddress);
        $legalProfessionalRulesLink = self::renderSafeUrlLink($legalProfessionalRulesUrl);
        $legalOdrLink = self::renderSafeUrlLink($legalOdrUrl);

        $easyMediaAiZipDefault = self::getDefaultEasyMediaAiZipDownloadPath();
        $easyMediaAiZipUrl = self::normalizeEasyMediaZipDownloadPath((string)SettingsService::get('easy_media_ai_zip_url', $easyMediaAiZipDefault));
        $emailLink = '';

        if ($contactEmail !== '' && filter_var($contactEmail, FILTER_VALIDATE_EMAIL)) {
            $safeEmail = Response::escape($contactEmail);
            $emailLink = '<a href="mailto:' . $safeEmail . '">' . $safeEmail . '</a>';
        }

        $e = [Response::class, 'escape'];

        $replacements = [
            '{SITE_NAME}' => $e($siteName),
            '{SITE_URL}' => $e($siteUrl),
            '{CONTACT_EMAIL}' => $emailLink !== '' ? $emailLink : $e($contactEmail),
            '{CONTACT_NAME}' => $e($contactName),
            '{CONTACT_PHONE}' => $e($contactPhone),
            '{CONTACT_ADDRESS}' => $contactAddressHtml,
            '{LEGAL_ENTITY_NAME}' => $e($legalEntityName),
            '{LEGAL_LEGAL_FORM}' => $e($legalLegalForm),
            '{LEGAL_REPRESENTATIVE}' => $e($legalRepresentative),
            '{LEGAL_STREET}' => $e($legalStreet),
            '{LEGAL_POSTAL_CODE}' => $e($legalPostalCode),
            '{LEGAL_CITY}' => $e($legalCity),
            '{LEGAL_COUNTRY}' => $e($legalCountry),
            '{LEGAL_FULL_ADDRESS}' => $legalFullAddressHtml,
            '{LEGAL_REGISTER_COURT}' => $e($legalRegisterCourt),
            '{LEGAL_REGISTER_NUMBER}' => $e($legalRegisterNumber),
            '{LEGAL_VAT_ID}' => $e($legalVatId),
            '{LEGAL_BUSINESS_ID}' => $e($legalBusinessId),
            '{LEGAL_EDITOR_NAME}' => $e($legalEditorName),
            '{LEGAL_EDITOR_ADDRESS}' => $legalEditorAddressHtml,
            '{LEGAL_PROFESSION_NAME}' => $e($legalProfessionName),
            '{LEGAL_PROFESSIONAL_TITLE}' => $e($legalProfessionalTitle),
            '{LEGAL_PROFESSIONAL_TITLE_COUNTRY}' => $e($legalProfessionalTitleCountry),
            '{LEGAL_CHAMBER}' => $e($legalChamber),
            '{LEGAL_PROFESSIONAL_RULES}' => $e($legalProfessionalRules),
            '{LEGAL_PROFESSIONAL_RULES_URL}' => $e($legalProfessionalRulesUrl),
            '{LEGAL_PROFESSIONAL_RULES_LINK}' => $legalProfessionalRulesLink,
            '{LEGAL_SUPERVISORY_AUTHORITY}' => $e($legalSupervisoryAuthority),
            '{LEGAL_ODR_URL}' => $e($legalOdrUrl),
            '{LEGAL_ODR_LINK}' => $legalOdrLink,
            '{LEGAL_DISPUTE_PARTICIPATION}' => $e($legalDisputeParticipation),
            '{LEGAL_HOSTING_PROVIDER_NAME}' => $e($legalHostingProviderName),
            '{LEGAL_HOSTING_PROVIDER_ADDRESS}' => $legalHostingProviderAddressHtml,
            '{LEGAL_CONTACT_EMAIL}' => $emailLink !== '' ? $emailLink : $e($contactEmail),
            '{LEGAL_CONTACT_PHONE}' => $e($contactPhone),
            '{EASY_MEDIA_AI_ZIP_URL}' => $e($easyMediaAiZipUrl)
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $content);
    }

    private static function getBaseUrl(): string
    {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $scheme . '://' . $host;
    }
}
