<?php

declare(strict_types=1);

namespace NewSite\Shop;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Shop\CartService;
use NewSite\Shop\CurrencyService;
use NewSite\Settings\SettingsService;
use NewSite\User\UserService;
use NewSite\Logging\LogService;

/**
 * Order creation and item processing service.
 *
 * Security: All database queries use prepared statements with bound parameters;
 * order numbers use cryptographically secure random bytes; transactions ensure
 * atomicity; no user input is interpolated into SQL.
 */
final class OrderService
{
    public static function resolveItemUnitPrice(array $item, string $baseCurrency, string $currency): float
    {
        $variantPrice = $item['variant_price'] ?? null;
        $productPrice = $item['product_price'] ?? null;
        $priceRaw = $variantPrice !== '' && $variantPrice !== null ? $variantPrice : $productPrice;
        $unitPrice = CurrencyService::parsePrice((string)$priceRaw);
        return CurrencyService::convert($unitPrice, $baseCurrency, $currency);
    }

    public static function calculateTotal(array $items, string $baseCurrency, string $currency): float
    {
        $total = 0.0;
        foreach ($items as $item) {
            $quantity = (int)($item['quantity'] ?? 0);
            $total += self::resolveItemUnitPrice((array)$item, $baseCurrency, $currency) * $quantity;
        }
        return $total;
    }

    public static function insertHeader(\PDO $db, int $userId, string $orderNumber, float $total, string $currency, array $context): int
    {
        $paymentData = (array)($context['payment_data'] ?? []);
        $user = (array)($context['user'] ?? []);
        $now = (string)($context['now'] ?? DbHelper::nowString());
        $stmt = $db->prepare("INSERT INTO orders (user_id, order_number, total_amount, currency, status, payment_provider, stripe_session_id, stripe_payment_intent, shipping_name, shipping_address, shipping_country, shipping_postal_code, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $userId, $orderNumber, $total, $currency,
            $paymentData['status'] ?? 'paid', $paymentData['payment_provider'] ?? 'stripe',
            $paymentData['stripe_session_id'] ?? null, $paymentData['stripe_payment_intent'] ?? null,
            $user['display_name'] ?? '', $user['address'] ?? '',
            $user['country'] ?? '', $user['postal_code'] ?? '',
            $now, $now
        ]);
        return DbHelper::lastInsertId($db, 'orders');
    }

    public static function resolveItemDownload(array $item): array
    {
        $actionType = (string)($item['variant_action_type'] ?: $item['product_action_type']);
        $downloadUrl = (string)($item['variant_download_url'] ?: $item['product_download_url']);
        return ['is_digital' => $actionType === 'download' && $downloadUrl !== '', 'download_url' => $downloadUrl];
    }

    public static function resolveDownloadVersion(array $item, \PDOStatement $getProductVersion): string
    {
        $productVersion = trim((string)($item['product_current_version'] ?? ''));
        if ($productVersion === '') {
            $getProductVersion->execute([(int)$item['product_id']]);
            $productVersion = (string)($getProductVersion->fetchColumn() ?: '1.0');
        }
        $variantVersion = trim((string)($item['variant_current_version'] ?? ''));
        return $variantVersion !== '' ? $variantVersion : $productVersion;
    }

    public static function createItemDownload(\PDOStatement $insertDownload, \PDOStatement $getProductVersion, int $orderItemId, int $userId, array $item, int $downloadExpiry, int $nowUnix): void
    {
        $downloadData = self::resolveItemDownload($item);
        $downloadUrl = (string)($downloadData['download_url'] ?? '');
        $token = bin2hex(random_bytes(16));
        $downloadVersion = self::resolveDownloadVersion($item, $getProductVersion);
        $insertDownload->execute([$orderItemId, $userId, (int)$item['product_id'], $token, $downloadUrl, 0, 0, $downloadExpiry, $nowUnix, $downloadVersion]);
    }

    public static function insertItems(\PDO $db, array $items, array $context): void
    {
        $orderId = (int)($context['order_id'] ?? 0);
        $userId = (int)($context['user_id'] ?? 0);
        $baseCurrency = (string)($context['base_currency'] ?? 'USD');
        $currency = (string)($context['currency'] ?? 'USD');
        $now = (string)($context['now'] ?? DbHelper::nowString());
        $downloadExpiry = (int)($context['download_expiry'] ?? 0);
        $nowUnix = (int)($context['now_unix'] ?? time());

        $insertItem = $db->prepare("INSERT INTO order_items (order_id, product_id, variant_id, product_name, variant_label, unit_price, quantity, is_digital, download_url, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $insertDownload = $db->prepare("INSERT INTO digital_downloads (order_item_id, user_id, product_id, token, download_url, max_downloads, download_count, expires_at, created_at, last_downloaded_version) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $updateSales = $db->prepare("UPDATE products SET sales_count = sales_count + ? WHERE id = ?");
        $getProductVersion = $db->prepare("SELECT current_version FROM products WHERE id = ?");

        foreach ($items as $item) {
            $quantity = (int)($item['quantity'] ?? 0);
            $downloadData = self::resolveItemDownload((array)$item);
            $insertItem->execute([
                $orderId, (int)$item['product_id'],
                $item['variant_id'] ? (int)$item['variant_id'] : null,
                $item['product_name'] ?? '', $item['variant_label'] ?? '',
                self::resolveItemUnitPrice((array)$item, $baseCurrency, $currency),
                $quantity, $downloadData['is_digital'] ? 1 : 0,
                $downloadData['is_digital'] ? $downloadData['download_url'] : null, $now
            ]);
            $orderItemId = DbHelper::lastInsertId($db, 'order_items');
            if ($downloadData['is_digital']) {
                self::createItemDownload($insertDownload, $getProductVersion, $orderItemId, $userId, (array)$item, $downloadExpiry, $nowUnix);
            }
            $updateSales->execute([$quantity, (int)$item['product_id']]);
        }
    }

    public static function createFromCart(int $userId, int $cartId, array $paymentData = []): ?int
    {
        $items = CartService::fetchItems($cartId);
        if (empty($items)) {
            return null;
        }
        $db = DatabaseManager::getWriteConnection();
        $currency = CurrencyService::getCurrent();
        $baseCurrency = SettingsService::get('store_currency', 'USD') ?? 'USD';
        $user = UserService::getById($userId);
        $orderNumber = 'ORD-' . date('Ymd') . '-' . strtoupper(bin2hex(random_bytes(3)));
        $total = self::calculateTotal($items, $baseCurrency, $currency);
        $db->beginTransaction();
        try {
            $now = DbHelper::nowString();
            $orderId = self::insertHeader($db, $userId, $orderNumber, $total, $currency, ['payment_data' => $paymentData, 'user' => (array)($user ?: []), 'now' => $now]);
            $nowUnix = time();
            $downloadExpiry = $nowUnix + (30 * 24 * 60 * 60);
            self::insertItems($db, $items, ['order_id' => $orderId, 'user_id' => $userId, 'base_currency' => $baseCurrency, 'currency' => $currency, 'now' => $now, 'download_expiry' => $downloadExpiry, 'now_unix' => $nowUnix]);
            CartService::clearItems($cartId);
            $db->commit();
            LogService::add('order', "New order created: $orderNumber", "User: $userId, Total: $total $currency, Order ID: $orderId");
            return $orderId;
        } catch (\Throwable $e) {
            $db->rollBack();
            error_log('Order creation failed: ' . $e->getMessage());
            LogService::add('order_error', "Order creation failed", "User: $userId, Error: " . $e->getMessage());
            return null;
        }
    }
}
