<?php

declare(strict_types=1);

namespace NewSite\Shop;

use NewSite\Database\DatabaseManager;

/**
 * Digital download version-checking service.
 *
 * Security: All database queries use prepared statements with bound parameters;
 * no user input is interpolated into SQL.
 */
final class DownloadService
{
    public static function hasUpdate(int $productId, ?string $lastDownloadedVersion): bool
    {
        if ($lastDownloadedVersion === null) {
            return false;
        }
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT current_version FROM products WHERE id = ?");
        $stmt->execute([$productId]);
        $currentVersion = $stmt->fetchColumn();
        if (!$currentVersion) {
            return false;
        }
        return $currentVersion !== $lastDownloadedVersion;
    }

    public static function getUserUpdatesCount(int $userId): int
    {
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("
            SELECT COUNT(DISTINCT dd.id)
            FROM digital_downloads dd
            JOIN order_items oi ON oi.id = dd.order_item_id
            LEFT JOIN products p ON p.id = COALESCE(dd.product_id, oi.product_id)
            LEFT JOIN product_variants pv ON pv.id = oi.variant_id
            WHERE dd.user_id = ?
              AND dd.last_downloaded_version IS NOT NULL
              AND TRIM(dd.last_downloaded_version) != TRIM(
                  COALESCE(
                      NULLIF(
                          CASE
                              WHEN oi.variant_id IS NOT NULL
                                   AND COALESCE(pv.action_type, '') = 'download'
                              THEN COALESCE(pv.current_version, '')
                              ELSE ''
                          END,
                          ''
                      ),
                      p.current_version,
                      dd.last_downloaded_version
                  )
              )
        ");
        $stmt->execute([$userId]);
        return (int)$stmt->fetchColumn();
    }
}
