<?php

declare(strict_types=1);

namespace NewSite\Shop;

use NewSite\Settings\SettingsService;
use NewSite\Util\CurrencyData;

/**
 * CurrencyService — multi-currency conversion, formatting, and session handling.
 *
 * Security: No direct database access; all persistence is delegated to
 * SettingsService (which uses prepared statements). User-supplied price
 * strings are sanitised before conversion.
 */
final class CurrencyService
{
    public static function getCurrent(): string
    {
        $default  = SettingsService::get('store_currency', 'USD');
        $currency = $_SESSION['store_currency'] ?? $default;
        $map      = CurrencyData::getMap();

        if (!isset($map[$currency])) {
            return $default;
        }

        return $currency;
    }

    public static function getSymbol(): string
    {
        $currency        = self::getCurrent();
        $defaultCurrency = SettingsService::get('store_currency', 'USD');
        $symbolSetting   = SettingsService::get('store_currency_symbol', '$');
        $map             = CurrencyData::getMap();

        if ($currency === $defaultCurrency && $symbolSetting) {
            return $symbolSetting;
        }

        return $map[$currency] ?? ($symbolSetting ?: '$');
    }

    public static function getRates(): array
    {
        $rates = [
            'USD' => 1.0,  'EUR' => 0.92,  'GBP' => 0.79, 'JPY' => 147.0,
            'AUD' => 1.52, 'CAD' => 1.36,  'CHF' => 0.88, 'SEK' => 10.4,
            'NOK' => 10.5, 'DKK' => 6.85,  'PLN' => 3.95, 'BRL' => 4.95,
            'MXN' => 17.2, 'INR' => 83.0,  'KRW' => 1325.0, 'CNY' => 7.2,
        ];

        $custom = SettingsService::get('currency_rates', '');
        if ($custom !== null && $custom !== '') {
            $decoded = json_decode($custom, true);
            if (is_array($decoded)) {
                foreach ($decoded as $code => $rate) {
                    $code  = strtoupper(trim((string) $code));
                    $value = (float) $rate;
                    if ($code !== '' && $value > 0) {
                        $rates[$code] = $value;
                    }
                }
            }
        }

        return $rates;
    }

    public static function convert(float $amount, string $fromCurrency, string $toCurrency): float
    {
        $from = strtoupper($fromCurrency);
        $to   = strtoupper($toCurrency);

        if ($from === $to) {
            return $amount;
        }

        $rates   = self::getRates();
        $fromRate = $rates[$from] ?? 0;
        $toRate   = $rates[$to] ?? 0;

        if ($fromRate <= 0 || $toRate <= 0) {
            return $amount;
        }

        $amountUsd = $amount / $fromRate;

        return $amountUsd * $toRate;
    }

    public static function getSymbolFor(string $currency): string
    {
        $map = CurrencyData::getMap();

        return $map[$currency] ?? self::getSymbol();
    }

    public static function parsePrice(string $price): float
    {
        $numeric = preg_replace('/[^0-9.]/', '', $price);

        if ($numeric === '' || $numeric === null) {
            return 0.0;
        }

        return (float) $numeric;
    }

    public static function formatCart(string $price, string $symbol = ''): string
    {
        $value = trim($price);
        if ($value === '') {
            return '';
        }

        $numeric = self::parsePrice($value);
        if ($numeric === 0.0 && $value !== '0' && $value !== '0.0' && $value !== '0.00') {
            return $value;
        }

        $baseCurrency    = SettingsService::get('store_currency', 'USD');
        $currentCurrency = self::getCurrent();
        $converted       = self::convert($numeric, $baseCurrency, $currentCurrency);

        $requestedSymbol = trim($symbol);
        $currencySymbol  = $requestedSymbol !== '' ? $requestedSymbol : self::getSymbolFor($currentCurrency);

        return $currencySymbol . number_format($converted, 2);
    }

    public static function formatValue(string $price, ?string $currency = null): string
    {
        $value = trim($price);
        if ($value === '') {
            return '';
        }

        $numeric = self::parsePrice($value);
        if ($numeric === 0.0 && $value !== '0' && $value !== '0.0' && $value !== '0.00') {
            return $value;
        }

        $baseCurrency   = SettingsService::get('store_currency', 'USD');
        $targetCurrency = $currency ?? self::getCurrent();
        $converted      = self::convert($numeric, $baseCurrency, $targetCurrency);
        $symbol         = self::getSymbolFor($targetCurrency);

        return $symbol . number_format($converted, 2);
    }
}
