<?php

declare(strict_types=1);

namespace NewSite\Shop;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;

/**
 * Shopping-cart session and item management.
 *
 * Security: All database access uses prepared statements with bound parameters.
 * Cart is tied to the server-side session ID; no user input reaches queries unbound.
 */
final class CartService
{
    public static function getSessionId(): string
    {
        return session_id();
    }

    public static function getCartId(): int
    {
        $db = DatabaseManager::getWriteConnection();
        $sessionId = self::getSessionId();

        $stmt = $db->prepare("SELECT id FROM cart_sessions WHERE session_id = ?");
        $stmt->execute([$sessionId]);
        $cartId = $stmt->fetchColumn();

        if ($cartId) {
            return (int) $cartId;
        }

        $stmt = $db->prepare("INSERT INTO cart_sessions (session_id) VALUES (?)");
        $stmt->execute([$sessionId]);

        return DbHelper::lastInsertId($db, 'cart_sessions');
    }

    public static function fetchItems(int $cartId): array
    {
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare(
            "SELECT ci.id, ci.quantity, ci.product_id, ci.variant_id,
                    p.name AS product_name, p.price AS product_price, p.quantity_max, p.stock AS product_stock,
                    p.action_type AS product_action_type, p.download_url AS product_download_url,
                    p.current_version AS product_current_version,
                    p.product_slug, pg.slug AS page_slug,
                    v.label AS variant_label, v.price AS variant_price, v.stock AS variant_stock,
                    v.action_type AS variant_action_type, v.download_url AS variant_download_url,
                    v.current_version AS variant_current_version,
                    pm.media_url AS media_url
             FROM cart_items ci
             JOIN products p ON p.id = ci.product_id
             LEFT JOIN pages pg ON pg.id = p.page_id
             LEFT JOIN product_variants v ON v.id = ci.variant_id
             LEFT JOIN product_media pm ON pm.product_id = p.id AND pm.sort_order = 0
             WHERE ci.cart_id = ?
             ORDER BY ci.created_at DESC"
        );
        $stmt->execute([$cartId]);
        return $stmt->fetchAll();
    }

    public static function clearItems(int $cartId): void
    {
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("DELETE FROM cart_items WHERE cart_id = ?");
        $stmt->execute([$cartId]);
        $stmt = $db->prepare("DELETE FROM cart_sessions WHERE id = ?");
        $stmt->execute([$cartId]);
    }

    public static function cleanupEmptySession(int $cartId): void
    {
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT COUNT(*) FROM cart_items WHERE cart_id = ?");
        $stmt->execute([$cartId]);

        if ((int) $stmt->fetchColumn() === 0) {
            $stmt = $db->prepare("DELETE FROM cart_sessions WHERE id = ?");
            $stmt->execute([$cartId]);
        }
    }

    /**
     * Get total item count for the current session without creating a cart.
     *
     * Read-only: does not insert a cart_sessions row if none exists.
     * This avoids unnecessary DB writes on every page load.
     */
    public static function getItemCount(): int
    {
        $sessionId = session_id();
        if ($sessionId === '' || $sessionId === false) {
            return 0;
        }

        $db   = DatabaseManager::getReadConnection();
        $stmt = $db->prepare(
            "SELECT COALESCE(SUM(ci.quantity), 0)
             FROM cart_items ci
             JOIN cart_sessions cs ON cs.id = ci.cart_id
             WHERE cs.session_id = ?"
        );
        $stmt->execute([$sessionId]);

        return (int) $stmt->fetchColumn();
    }
}
