<?php

declare(strict_types=1);

namespace NewSite\Settings;

use NewSite\Cache\CacheService;
use NewSite\Database\DatabaseManager;

/**
 * Settings and theme settings management service.
 *
 * Security: All database queries use prepared statements to prevent SQL injection.
 * Cache keys are derived from trusted input only.
 */
final class SettingsService
{
    public static function get(string $key, ?string $default = null): ?string
    {
        $cacheKey = 'setting:' . $key;
        $cached = CacheService::get($cacheKey, '__MISS__');
        if ($cached !== '__MISS__') {
            return $cached === '__NULL__' ? $default : $cached;
        }
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        $value = $result ? $result['setting_value'] : null;
        CacheService::set($cacheKey, $value === null ? '__NULL__' : $value, 60);
        return $value === null ? $default : $value;
    }

    public static function set(string $key, ?string $value): bool
    {
        $db = DatabaseManager::getWriteConnection();
        $now = date('Y-m-d H:i:s');
        $stmt = $db->prepare(
            "INSERT INTO settings (setting_key, setting_value, updated_at) VALUES (?, ?, ?) "
            . "ON CONFLICT (setting_key) DO UPDATE SET setting_value = EXCLUDED.setting_value, "
            . "updated_at = EXCLUDED.updated_at"
        );
        $result = $stmt->execute([$key, $value, $now]);
        CacheService::delete('setting:' . $key);
        return $result;
    }

    public static function getTheme(string $key, ?string $default = null): ?string
    {
        $cacheKey = 'theme_setting:' . $key;
        $cached = CacheService::get($cacheKey, '__MISS__');
        if ($cached !== '__MISS__') {
            return $cached === '__NULL__' ? $default : $cached;
        }
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare("SELECT setting_value FROM theme_settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        $value = $result ? $result['setting_value'] : null;
        CacheService::set($cacheKey, $value === null ? '__NULL__' : $value, 60);
        return $value === null ? $default : $value;
    }

    public static function setTheme(string $key, ?string $value): bool
    {
        $db = DatabaseManager::getWriteConnection();
        $now = date('Y-m-d H:i:s');
        $stmt = $db->prepare(
            "INSERT INTO theme_settings (setting_key, setting_value, updated_at) VALUES (?, ?, ?) "
            . "ON CONFLICT (setting_key) DO UPDATE SET setting_value = EXCLUDED.setting_value, "
            . "updated_at = EXCLUDED.updated_at"
        );
        $result = $stmt->execute([$key, $value, $now]);
        CacheService::delete('theme_setting:' . $key);
        return $result;
    }

    public static function getAllTheme(): array
    {
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->query("SELECT setting_key, setting_value FROM theme_settings");
        $settings = [];
        while ($row = $stmt->fetch()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        return $settings;
    }
}
