<?php

declare(strict_types=1);

namespace NewSite\Security;

use NewSite\Database\DatabaseManager;
use NewSite\Security\IpService;

/**
 * RateLimiter — per-IP / per-key rate limiting for contact, DMCA, and login forms.
 *
 * Security: Every database interaction uses prepared statements with bound
 * parameters. Rate-limit keys are derived from hashed/normalised IP addresses
 * so raw IPs are never stored in rate-limit tables.
 */
final class RateLimiter
{
    public static function contactOk(string $ip, int $cooldownSeconds = 120): bool
    {
        $db   = DatabaseManager::getWriteConnection();
        $key  = IpService::rateLimitKey($ip);
        $stmt = $db->prepare("SELECT last_time FROM contact_rate_limits WHERE ip = ?");
        $stmt->execute([$key]);
        $last = $stmt->fetchColumn();

        if (!$last) {
            return true;
        }

        return (time() - (int) $last) >= $cooldownSeconds;
    }

    public static function contactUpdate(string $ip): bool
    {
        $db   = DatabaseManager::getWriteConnection();
        $key  = IpService::rateLimitKey($ip);
        $stmt = $db->prepare(
            "INSERT INTO contact_rate_limits (ip, last_time) VALUES (?, ?) "
            . "ON CONFLICT (ip) DO UPDATE SET last_time = EXCLUDED.last_time"
        );

        return $stmt->execute([$key, time()]);
    }

    public static function dmcaStatus(string $rateKey, int $maxCount = 3, int $windowSeconds = 86400): array
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT last_time, count FROM dmca_rate_limits WHERE rate_key = ?");
        $stmt->execute([$rateKey]);
        $row    = $stmt->fetch();
        $status = ['ok' => true, 'wait' => 0];

        if ($row) {
            $last  = (int) ($row['last_time'] ?? 0);
            $count = (int) ($row['count'] ?? 0);
            if ($last > 0 && (time() - $last) < $windowSeconds && $count >= $maxCount) {
                $wait   = $windowSeconds - (time() - $last);
                $status = ['ok' => false, 'wait' => max(0, $wait)];
            }
        }

        return $status;
    }

    public static function dmcaStatusFor(string $ip, string $email, int $maxCount = 3, int $windowSeconds = 86400): array
    {
        $ipKey    = 'ip:' . IpService::rateLimitKey($ip);
        $emailKey = 'email:' . hash('sha256', strtolower(trim($email)));

        $ipStatus    = self::dmcaStatus($ipKey, $maxCount, $windowSeconds);
        $emailStatus = self::dmcaStatus($emailKey, $maxCount, $windowSeconds);

        if (!$ipStatus['ok'] || !$emailStatus['ok']) {
            return ['ok' => false, 'wait' => max($ipStatus['wait'], $emailStatus['wait'])];
        }

        return ['ok' => true, 'wait' => 0];
    }

    public static function dmcaUpdateKey(string $rateKey, int $windowSeconds = 86400): void
    {
        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT last_time, count FROM dmca_rate_limits WHERE rate_key = ?");
        $stmt->execute([$rateKey]);
        $row = $stmt->fetch();
        $now = time();

        if (!$row) {
            $stmt = $db->prepare("INSERT INTO dmca_rate_limits (rate_key, last_time, count) VALUES (?, ?, ?)");
            $stmt->execute([$rateKey, $now, 1]);
            return;
        }

        $last  = (int) ($row['last_time'] ?? 0);
        $count = (int) ($row['count'] ?? 0);

        if ($last <= 0 || ($now - $last) >= $windowSeconds) {
            $count = 0;
        }

        $count++;

        $stmt = $db->prepare("UPDATE dmca_rate_limits SET last_time = ?, count = ? WHERE rate_key = ?");
        $stmt->execute([$now, $count, $rateKey]);
    }

    public static function dmcaUpdate(string $ip, string $email): void
    {
        $ipKey    = 'ip:' . IpService::rateLimitKey($ip);
        $emailKey = 'email:' . hash('sha256', strtolower(trim($email)));

        self::dmcaUpdateKey($ipKey);
        self::dmcaUpdateKey($emailKey);
    }

    public static function loginCooldownForCount(int $count): int
    {
        $count    = max(1, $count);
        $cooldown = 5 * (2 ** ($count - 1));

        return (int) min($cooldown, 60);
    }

    public static function loginStatus(string $ip): array
    {
        $db   = DatabaseManager::getWriteConnection();
        $key  = IpService::rateLimitKey($ip);
        $stmt = $db->prepare("SELECT last_time, count FROM login_rate_limits WHERE ip = ?");
        $stmt->execute([$key]);
        $row    = $stmt->fetch();
        $status = ['ok' => true, 'wait' => 0];

        if ($row) {
            $last  = (int) ($row['last_time'] ?? 0);
            $count = (int) ($row['count'] ?? 0);
            if ($count > 0 && $last > 0) {
                $cooldown = self::loginCooldownForCount($count);
                $elapsed  = time() - $last;
                if ($elapsed >= $cooldown) {
                    $stmt = $db->prepare("UPDATE login_rate_limits SET count = 0 WHERE ip = ?");
                    $stmt->execute([$key]);
                } else {
                    $status = ['ok' => false, 'wait' => $cooldown - $elapsed];
                }
            }
        }

        return $status;
    }

    public static function loginOk(string $ip): bool
    {
        $status = self::loginStatus($ip);

        return $status['ok'];
    }

    public static function loginUpdate(string $ip): bool
    {
        $db   = DatabaseManager::getWriteConnection();
        $key  = IpService::rateLimitKey($ip);
        $stmt = $db->prepare("SELECT last_time, count FROM login_rate_limits WHERE ip = ?");
        $stmt->execute([$key]);
        $row = $stmt->fetch();
        $now = time();

        if (!$row) {
            $stmt = $db->prepare("INSERT INTO login_rate_limits (ip, last_time, count) VALUES (?, ?, ?)");
            return $stmt->execute([$key, $now, 1]);
        }

        $last  = (int) ($row['last_time'] ?? 0);
        $count = (int) ($row['count'] ?? 0);

        $cooldown = $count > 0 ? self::loginCooldownForCount($count) : 0;
        $elapsed  = $now - $last;

        if ($count <= 0 || $elapsed >= $cooldown) {
            $count = 1;
        } else {
            $count = min($count + 1, 5);
        }

        $stmt = $db->prepare("UPDATE login_rate_limits SET last_time = ?, count = ? WHERE ip = ?");

        return $stmt->execute([$now, $count, $key]);
    }
}
