<?php

declare(strict_types=1);

namespace NewSite\Security;

use NewSite\Settings\SettingsService;

/**
 * IP address normalization, anonymization, and GDPR-compliant storage.
 *
 * Security: Never stores raw IPs when GDPR anonymization is enabled.
 * All IP values are validated and normalized before use.
 */
final class IpService
{
    public static function normalize(?string $ip): ?string
    {
        if ($ip === null) {
            return null;
        }
        $ip = trim($ip);
        if ($ip === '' || !filter_var($ip, FILTER_VALIDATE_IP)) {
            return null;
        }
        $result = strtolower($ip);
        $packed = @inet_pton($ip);
        if ($packed !== false) {
            $normalized = inet_ntop($packed);
            if ($normalized !== false && $normalized !== '') {
                $result = strtolower($normalized);
            }
        }
        return $result;
    }

    public static function getClientIP(): string
    {
        $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            $ip = $_SERVER['HTTP_CF_CONNECTING_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            $ip = trim($ips[0]);
        } elseif (!empty($_SERVER['HTTP_X_REAL_IP'])) {
            $ip = $_SERVER['HTTP_X_REAL_IP'];
        }
        return self::normalize($ip) ?? '0.0.0.0';
    }

    public static function isAnonymizationEnabled(): bool
    {
        return SettingsService::get('gdpr_ip_anonymization', '1') === '1';
    }

    public static function mask(string $ip): string
    {
        $normalized = self::normalize($ip);
        if ($normalized === null) {
            return $ip;
        }
        $result = $normalized;
        $packed = @inet_pton($normalized);
        if ($packed !== false) {
            if (strlen($packed) === 4) {
                $packed[3] = "\0";
                $result = strtolower((string)(inet_ntop($packed) ?: $normalized));
            } elseif (strlen($packed) === 16) {
                $packed = substr($packed, 0, 8) . str_repeat("\0", 8);
                $result = strtolower((string)(inet_ntop($packed) ?: $normalized));
            }
        }
        return $result;
    }

    public static function gdprStore(?string $ip): ?string
    {
        if ($ip === null || $ip === '') {
            return $ip;
        }
        $normalized = self::normalize($ip);
        if ($normalized === null) {
            return $ip;
        }
        return self::isAnonymizationEnabled() ? self::mask($normalized) : $normalized;
    }

    public static function rateLimitKey(string $ip): string
    {
        return self::isAnonymizationEnabled() ? hash('sha256', $ip) : $ip;
    }

    public static function geoCacheKey(string $ip): string
    {
        return self::isAnonymizationEnabled() ? hash('sha256', $ip) : $ip;
    }
}
