<?php

declare(strict_types=1);

namespace NewSite\Security;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Security\IpService;

/**
 * IpBanService — IP-based access banning.
 *
 * Security: All database queries use prepared statements with bound
 * parameters. IP addresses are normalised before comparison and storage
 * to prevent bypass via alternate representations.
 */
final class IpBanService
{
    public static function isBanned(?string $ip = null): bool
    {
        if ($ip === null) {
            $ip = IpService::getClientIP();
        }

        $normalizedIp = IpService::normalize($ip);
        if ($normalizedIp === null) {
            return false;
        }

        $candidates = [$normalizedIp];
        $maskedIp   = IpService::gdprStore($normalizedIp);
        if (!empty($maskedIp) && !in_array($maskedIp, $candidates, true)) {
            $candidates[] = $maskedIp;
        }

        $candidateKeys = [];
        foreach ($candidates as $candidateIp) {
            $key = strtolower((string) $candidateIp);
            if ($key !== '' && !in_array($key, $candidateKeys, true)) {
                $candidateKeys[] = $key;
            }
        }

        if (empty($candidateKeys)) {
            return false;
        }

        $db           = DatabaseManager::getWriteConnection();
        $placeholders = implode(', ', array_fill(0, count($candidateKeys), '?'));
        $stmt         = $db->prepare(
            "SELECT id FROM banned_ips WHERE LOWER(ip_address) IN ({$placeholders}) LIMIT 1"
        );
        $stmt->execute($candidateKeys);

        return $stmt->fetch() !== false;
    }

    public static function ban(string $ip, ?string $reason = null): bool
    {
        $normalizedIp = IpService::normalize($ip);
        if ($normalizedIp === null) {
            return false;
        }

        $db   = DatabaseManager::getWriteConnection();
        $now  = DbHelper::nowString();
        $stmt = $db->prepare(
            "INSERT INTO banned_ips (ip_address, reason, banned_at, banned_by) "
            . "VALUES (?, ?, ?, ?) "
            . "ON CONFLICT (ip_address) DO UPDATE SET reason = EXCLUDED.reason, "
            . "banned_at = EXCLUDED.banned_at, banned_by = EXCLUDED.banned_by"
        );

        return $stmt->execute([
            $normalizedIp,
            $reason,
            $now,
            $_SESSION['admin_user_id'] ?? ($_SESSION['admin_id'] ?? null),
        ]);
    }

    public static function unban(string $ip): bool
    {
        $normalizedIp = IpService::normalize($ip);
        if ($normalizedIp === null) {
            return false;
        }

        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("DELETE FROM banned_ips WHERE LOWER(ip_address) = ?");

        return $stmt->execute([strtolower($normalizedIp)]);
    }

    public static function getAll(): array
    {
        $db = DatabaseManager::getReadConnection();

        return $db->query("SELECT * FROM banned_ips ORDER BY banned_at DESC")->fetchAll();
    }
}
