<?php

declare(strict_types=1);

namespace NewSite\Security;

/**
 * CSRF token and CSP nonce management.
 *
 * Security: generates cryptographically secure random tokens.
 * CSP nonce is per-request, CSRF token is per-session.
 */
final class CsrfService
{
    private const CSRF_MESSAGE = 'Invalid CSRF token.';

    public static function getCspNonce(): string
    {
        return $GLOBALS['CSP_NONCE'] ?? '';
    }

    public static function getToken(): string
    {
        return $_SESSION['csrf_token'] ?? '';
    }

    public static function rotateSession(): void
    {
        if (session_status() === PHP_SESSION_ACTIVE) {
            @session_regenerate_id(true);
        }
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }

    public static function getTokenFromRequest(): string
    {
        if (!empty($_SERVER['HTTP_X_CSRF_TOKEN'])) {
            return (string)$_SERVER['HTTP_X_CSRF_TOKEN'];
        }
        return (string)($_POST['csrf_token'] ?? '');
    }

    public static function verify(): void
    {
        $token = self::getTokenFromRequest();
        $sessionToken = self::getToken();
        if ($token === '' || $sessionToken === '' || !hash_equals($sessionToken, $token)) {
            http_response_code(403);
            die(self::CSRF_MESSAGE);
        }
    }

    public static function enforceForPost(): void
    {
        if (php_sapi_name() === 'cli') {
            return;
        }
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        if (strtoupper($method) !== 'POST') {
            return;
        }
        $requestUri = $_SERVER['REQUEST_URI'] ?? '';
        if (strpos($requestUri, '/stripe-webhook') !== false
            || strpos($requestUri, '/templates/api/stripe-webhook.php') !== false
            || strpos($requestUri, 'route=stripe-webhook') !== false) {
            return;
        }
        $token = self::getTokenFromRequest();
        $sessionToken = self::getToken();
        if ($token === '' || $sessionToken === '' || !hash_equals($sessionToken, $token)) {
            $isJson = (isset($_SERVER['HTTP_ACCEPT'])
                    && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false)
                || (isset($_SERVER['CONTENT_TYPE'])
                    && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false)
                || (isset($_SERVER['REQUEST_URI'])
                    && strpos($_SERVER['REQUEST_URI'], '/templates/api/') !== false);
            http_response_code(403);
            if ($isJson) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'error' => self::CSRF_MESSAGE]);
            } else {
                echo self::CSRF_MESSAGE;
            }
            exit;
        }
    }
}
