<?php

/**
 * Push Notification Service
 *
 * Sends real-time notifications to connected WebSocket clients
 * via a file-based push queue. Also invalidates cache when data changes.
 *
 * Security: no user input influences file paths beyond the integer user ID.
 * Queue files are pruned automatically to prevent filesystem abuse.
 */

declare(strict_types=1);

namespace NewSite\Push;

use NewSite\Cache\CachedQueries;
use NewSite\Cache\CacheService;
use NewSite\Config\SetupService;

class PushNotification
{
    private static ?string $wsHost = null;
    private static ?int $wsPort = null;
    private static ?int $maxQueueFiles = null;
    private const QUEUE_GLOB = '/*.json';

    /**
     * Initialize WebSocket connection details from config.
     */
    private static function init(): void
    {
        if (self::$wsHost !== null) {
            return;
        }
        self::$wsHost = SetupService::getEnvOrConfig('WS_INTERNAL_HOST', '127.0.0.1');
        self::$wsPort = (int) SetupService::getEnvOrConfig('WS_PORT', '8080');
        self::$maxQueueFiles = (int) SetupService::getEnvOrConfig('WS_QUEUE_MAX_FILES', '2000');
        if (self::$maxQueueFiles <= 0) {
            self::$maxQueueFiles = 2000;
        }
    }

    /**
     * Notify user of new data (triggers WebSocket push + cache invalidation).
     *
     * @param int    $userId Target user
     * @param string $type   Push event type
     * @param array  $data   Extra payload merged into the push
     */
    public static function notifyUser(int $userId, string $type = 'notification_update', array $data = []): void
    {
        // Invalidate cache so next poll gets fresh data
        CachedQueries::invalidateNotificationCache($userId);

        // Try to push via WebSocket
        self::sendWebSocketPush($userId, array_merge(['type' => $type], $data));
    }

    /**
     * Notify user of a new message.
     */
    public static function newMessage(int $userId, string $subject = '', string $body = ''): void
    {
        self::notifyUser($userId, 'new_message', [
            'subject' => $subject,
            'preview' => substr($body, 0, 100),
        ]);
    }

    /**
     * Notify user of an incoming friend request.
     */
    public static function friendRequest(int $toUserId, string $fromNickname): void
    {
        self::notifyUser($toUserId, 'friend_request', [
            'from' => $fromNickname,
        ]);
    }

    /**
     * Notify user that a friend request was accepted.
     */
    public static function friendAccepted(int $userId, string $friendNickname): void
    {
        CachedQueries::invalidateNotificationCache($userId);
        CacheService::delete("friends_count:$userId");

        self::sendWebSocketPush($userId, [
            'type' => 'friend_accepted',
            'friend' => $friendNickname,
        ]);
    }

    /**
     * Notify user of a new order.
     */
    public static function newOrder(int $userId, string $orderNumber, string $total): void
    {
        self::notifyUser($userId, 'new_order', [
            'order_number' => $orderNumber,
            'total' => $total,
        ]);
    }

    /**
     * Send push via file-based queue.
     *
     * Security: user ID is cast to int; no user-controlled strings
     * enter the filename beyond the sanitised ID and a random uniqid().
     */
    private static function sendWebSocketPush(int $userId, array $data): void
    {
        self::init();

        $pushDir = DATA_PATH . '/push_queue';

        if (!is_dir($pushDir)) {
            @mkdir($pushDir, 0755, true);
        }

        self::pruneQueue($pushDir);

        $pushFile = $pushDir . '/' . time() . '_' . $userId . '_' . uniqid() . '.json';

        @file_put_contents($pushFile, json_encode([
            'user_id'   => $userId,
            'data'      => $data,
            'timestamp' => time(),
        ]));
    }

    /**
     * Prune push queue to avoid filesystem overload.
     *
     * Removes entries older than 5 minutes and caps total count
     * at the configured maximum.
     */
    private static function pruneQueue(string $queueDir): void
    {
        $files = glob($queueDir . self::QUEUE_GLOB);
        if (!$files) {
            return;
        }

        self::removeExpiredFiles($files);
        self::enforceMaxFiles($queueDir);
    }

    /**
     * Delete queue files older than 5 minutes.
     *
     * @param string[] $files Absolute paths from glob()
     */
    private static function removeExpiredFiles(array $files): void
    {
        $now = time();
        foreach ($files as $file) {
            if (!is_file($file)) {
                continue;
            }
            $age = $now - @filemtime($file);
            if ($age > 300) {
                @unlink($file);
            }
        }
    }

    /**
     * Enforce maximum queue file count by removing oldest entries.
     */
    private static function enforceMaxFiles(string $queueDir): void
    {
        $files = glob($queueDir . self::QUEUE_GLOB);
        if (!$files || count($files) <= self::$maxQueueFiles) {
            return;
        }

        usort($files, static function (string $a, string $b): int {
            return @filemtime($a) <=> @filemtime($b);
        });

        $over = count($files) - self::$maxQueueFiles;
        for ($i = 0; $i < $over; $i++) {
            @unlink($files[$i]);
        }
    }

    /**
     * Process queued pushes (called by WebSocket server).
     *
     * @param object $wsServer Server instance with sendToUser() method
     */
    public static function processQueue(object $wsServer): void
    {
        $queueDir = DATA_PATH . '/push_queue';
        if (!is_dir($queueDir)) {
            return;
        }

        self::pruneQueue($queueDir);

        $files = glob($queueDir . self::QUEUE_GLOB);
        if (!$files) {
            return;
        }

        foreach ($files as $file) {
            self::processOneQueueFile($file, $wsServer);
        }
    }

    /**
     * Process a single queue file: read, delete, and forward to WebSocket.
     */
    private static function processOneQueueFile(string $file, object $wsServer): void
    {
        $content = @file_get_contents($file);
        @unlink($file); // Delete immediately to avoid double-processing

        if (!$content) {
            return;
        }

        $push = json_decode($content, true);
        if (!$push) {
            return;
        }

        // Skip old messages (more than 1 minute old)
        if (time() - ($push['timestamp'] ?? 0) > 60) {
            return;
        }

        $wsServer->sendToUser($push['user_id'], $push['data']);
    }
}
