<?php

/**
 * Mini-game Round Verifier
 *
 * Anti-cheat layer for verified leaderboard rounds.
 * Issues time-limited round tokens and validates score submissions
 * against timing windows and point-per-second caps.
 *
 * Security:
 * - Round tokens use cryptographically secure random bytes.
 * - Token comparison uses hash_equals() to prevent timing attacks.
 * - Scores are validated against elapsed time × max-points-per-second.
 */

declare(strict_types=1);

namespace NewSite\Minigames;

use NewSite\Cache\CacheService;
use Throwable;

class MinigameRoundVerifier
{
    private const CACHE_PREFIX_ROUND  = 'mini_game_round:';
    private const CACHE_PREFIX_ACTIVE = 'mini_game_round_active:';

    /**
     * Issue a new verified round for a game + user.
     *
     * @return array{round_id:string,round_token:string,issued_at:int,expires_at:int,min_submit_seconds:int,max_submit_seconds:int,max_allowed_score:float}|array{}
     */
    public static function issueRound(string $slug, int $userId): array
    {
        $slug    = MinigameCatalog::normalizeSlug($slug);
        $catalog = MinigameCatalog::getCatalog();
        if ($userId <= 0 || $slug === '' || !isset($catalog[$slug])) {
            return [];
        }

        return self::createRoundTokens($slug, $userId);
    }

    /**
     * Validate a verified round submission end-to-end.
     *
     * @return array{ok:bool,status:int,error:string,max_allowed_score?:float,elapsed_seconds?:int}
     */
    public static function validateSubmission(
        string $slug,
        int $userId,
        string $roundId,
        string $roundToken,
        float $score,
    ): array {
        $slug       = MinigameCatalog::normalizeSlug($slug);
        $roundId    = trim($roundId);
        $roundToken = trim($roundToken);

        $context   = self::buildValidationContext($slug, $userId, $roundId, $roundToken, $score);
        $roundKey  = (string) ($context['round_key']  ?? (self::CACHE_PREFIX_ROUND . $slug . ':' . $userId . ':' . $roundId));
        $activeKey = (string) ($context['active_key'] ?? (self::CACHE_PREFIX_ACTIVE . $slug . ':' . $userId));

        if (empty($context['ok'])) {
            self::cleanupRoundCache($roundKey, $activeKey, !empty($context['delete_round']), !empty($context['delete_active']));
            unset($context['delete_round'], $context['delete_active'], $context['policy'], $context['max_score_absolute'], $context['round_key'], $context['active_key'], $context['round_data']);

            return $context;
        }

        $policy           = is_array($context['policy'] ?? null) ? $context['policy'] : [];
        $roundData        = is_array($context['round_data'] ?? null) ? $context['round_data'] : [];
        $maxScoreAbsolute = (float) ($context['max_score_absolute'] ?? 25000.0);

        $result = self::evaluateScoreWindow($policy, $roundData, $score, $maxScoreAbsolute);
        self::cleanupRoundCache($roundKey, $activeKey, !empty($result['delete_round']), !empty($result['delete_active']));
        unset($result['delete_round'], $result['delete_active']);

        return $result;
    }

    // ------------------------------------------------------------------
    // Private helpers
    // ------------------------------------------------------------------

    /**
     * Create round tokens, store in cache, and return the public payload.
     *
     * @return array{round_id:string,round_token:string,issued_at:int,expires_at:int,min_submit_seconds:int,max_submit_seconds:int,max_allowed_score:float}|array{}
     */
    private static function createRoundTokens(string $slug, int $userId): array
    {
        $policy    = MinigameCatalog::getRoundPolicy($slug);
        $now       = time();
        $ttl       = max(60, (int) ($policy['round_ttl_seconds'] ?? 300));
        $expiresAt = $now + $ttl;

        try {
            $roundId    = bin2hex(random_bytes(12));
            $roundToken = bin2hex(random_bytes(24));
        } catch (Throwable) {
            return [];
        }

        $roundKey  = self::CACHE_PREFIX_ROUND . $slug . ':' . $userId . ':' . $roundId;
        $activeKey = self::CACHE_PREFIX_ACTIVE . $slug . ':' . $userId;
        $cacheTtl  = max(90, $ttl + 20);

        CacheService::set($roundKey, [
            'round_id'    => $roundId,
            'round_token' => $roundToken,
            'issued_at'   => $now,
            'expires_at'  => $expiresAt,
        ], $cacheTtl);
        CacheService::set($activeKey, $roundId, $cacheTtl);

        $maxAllowed = self::computeMaxAllowedScore($policy);

        return [
            'round_id'          => $roundId,
            'round_token'       => $roundToken,
            'issued_at'         => $now,
            'expires_at'        => $expiresAt,
            'min_submit_seconds' => (int) ($policy['min_submit_seconds'] ?? 8),
            'max_submit_seconds' => (int) ($policy['max_submit_seconds'] ?? 300),
            'max_allowed_score' => round($maxAllowed, 2),
        ];
    }

    /**
     * Compute the absolute maximum allowed score from a policy.
     */
    private static function computeMaxAllowedScore(array $policy): float
    {
        return min(
            (float) ($policy['max_score'] ?? 25000.0),
            (float) ($policy['base_score_allowance'] ?? 220.0)
            + ((float) ($policy['max_submit_seconds'] ?? 300.0) * (float) ($policy['max_points_per_second'] ?? 130.0))
        );
    }

    /**
     * Build a standard validation result array.
     *
     * @param array<string, mixed> $extra Additional keys merged into the result
     * @return array{ok:bool,status:int,error:string}&array<string,mixed>
     */
    private static function validationResult(bool $ok, int $status, string $error, array $extra = []): array
    {
        return array_merge([
            'ok'     => $ok,
            'status' => $status,
            'error'  => $error,
        ], $extra);
    }

    /**
     * Build the full validation context from cache.
     *
     * @return array<string, mixed>
     */
    private static function buildValidationContext(
        string $slug,
        int $userId,
        string $roundId,
        string $roundToken,
        float $score,
    ): array {
        $roundExpired = 'Round expired. Start a new game.';

        $hasBaseInput = $userId > 0 && $slug !== '' && $roundId !== '' && $roundToken !== '';
        if (!$hasBaseInput) {
            return self::validationResult(false, 400, 'Missing round verification data.', [
                'delete_round' => false,
                'delete_active' => false,
            ]);
        }

        $policy           = MinigameCatalog::getRoundPolicy($slug);
        $maxScoreAbsolute = (float) ($policy['max_score'] ?? 25000.0);
        $scoreInRange     = is_finite($score) && $score >= 0 && $score <= $maxScoreAbsolute;
        if (!$scoreInRange) {
            return self::validationResult(false, 400, 'Score out of verified range.', [
                'delete_round' => false,
                'delete_active' => false,
            ]);
        }

        return self::resolveFromCache($slug, $userId, $roundId, $roundToken, $policy, $maxScoreAbsolute, $roundExpired);
    }

    /**
     * Resolve round validation by checking cached round + active-round data.
     *
     * @param array<string, mixed> $policy
     * @return array<string, mixed>
     */
    private static function resolveFromCache(
        string $slug,
        int $userId,
        string $roundId,
        string $roundToken,
        array $policy,
        float $maxScoreAbsolute,
        string $roundExpiredMessage,
    ): array {
        $roundKey  = self::CACHE_PREFIX_ROUND . $slug . ':' . $userId . ':' . $roundId;
        $activeKey = self::CACHE_PREFIX_ACTIVE . $slug . ':' . $userId;
        $roundData = CacheService::get($roundKey, null);
        $result    = self::validationResult(false, 409, $roundExpiredMessage, [
            'delete_round'  => false,
            'delete_active' => false,
        ]);

        if (is_array($roundData)) {
            $result = self::verifyTokenAndActiveRound(
                $roundData,
                $roundToken,
                $roundId,
                $activeKey,
                $policy,
                $maxScoreAbsolute,
                $roundKey,
            );
        }

        return $result;
    }

    /**
     * Verify the round token and whether this is the active round.
     *
     * @param array<string, mixed> $roundData
     * @param array<string, mixed> $policy
     * @return array<string, mixed>
     */
    private static function verifyTokenAndActiveRound(
        array $roundData,
        string $roundToken,
        string $roundId,
        string $activeKey,
        array $policy,
        float $maxScoreAbsolute,
        string $roundKey,
    ): array {
        $storedToken = (string) ($roundData['round_token'] ?? '');
        $tokenValid  = $storedToken !== '' && hash_equals($storedToken, $roundToken);

        if (!$tokenValid) {
            return self::validationResult(false, 403, 'Invalid round token.', [
                'delete_round'  => false,
                'delete_active' => false,
            ]);
        }

        $activeRoundId = (string) CacheService::get($activeKey, '');
        $isActive      = $activeRoundId !== '' && hash_equals($activeRoundId, $roundId);

        if (!$isActive) {
            return self::validationResult(false, 409, 'Round replaced. Start a new game.', [
                'delete_round'  => true,
                'delete_active' => false,
            ]);
        }

        return self::validationResult(true, 200, '', [
            'delete_round'      => false,
            'delete_active'     => false,
            'policy'            => $policy,
            'max_score_absolute' => $maxScoreAbsolute,
            'round_key'         => $roundKey,
            'active_key'        => $activeKey,
            'round_data'        => $roundData,
        ]);
    }

    /**
     * Evaluate whether a score fits the allowed time window.
     *
     * @param array<string, mixed> $policy
     * @param array<string, mixed> $roundData
     * @return array<string, mixed>
     */
    private static function evaluateScoreWindow(
        array $policy,
        array $roundData,
        float $score,
        float $maxScoreAbsolute,
    ): array {
        $roundExpired = 'Round expired. Start a new game.';
        $result = self::validationResult(false, 409, $roundExpired, [
            'delete_round'  => true,
            'delete_active' => true,
        ]);

        $now       = time();
        $issuedAt  = (int) ($roundData['issued_at']  ?? 0);
        $expiresAt = (int) ($roundData['expires_at'] ?? 0);
        $elapsed   = max(0, $now - $issuedAt);

        $minSubmit = max(1, (int) ($policy['min_submit_seconds'] ?? 8));
        $maxSubmit = max($minSubmit, (int) ($policy['max_submit_seconds'] ?? 300));

        $isExpired = $issuedAt <= 0 || $expiresAt <= 0 || $now > $expiresAt || $elapsed > $maxSubmit;
        if (!$isExpired) {
            $result = self::evaluateNonExpiredRound($policy, $score, $maxScoreAbsolute, $elapsed, $minSubmit);
        }

        return $result;
    }

    /**
     * Score window evaluation for a non-expired round.
     *
     * @param array<string, mixed> $policy
     * @return array<string, mixed>
     */
    private static function evaluateNonExpiredRound(
        array $policy,
        float $score,
        float $maxScoreAbsolute,
        int $elapsed,
        int $minSubmit,
    ): array {
        if ($elapsed < $minSubmit) {
            return self::validationResult(false, 429, 'Score submitted too quickly.', [
                'delete_round'  => false,
                'delete_active' => false,
            ]);
        }

        $baseAllowance  = (float) ($policy['base_score_allowance']  ?? 220.0);
        $pointsPerSecond = (float) ($policy['max_points_per_second'] ?? 130.0);
        $maxAllowed      = min($maxScoreAbsolute, $baseAllowance + ($elapsed * $pointsPerSecond));
        $ok              = $score <= $maxAllowed;

        return self::validationResult($ok, $ok ? 200 : 422, $ok ? '' : 'Score failed verification checks.', [
            'delete_round'      => true,
            'delete_active'     => true,
            'max_allowed_score' => round($maxAllowed, 2),
            'elapsed_seconds'   => $elapsed,
        ]);
    }

    /**
     * Clean up round-related cache keys.
     */
    private static function cleanupRoundCache(
        string $roundKey,
        string $activeKey,
        bool $deleteRound,
        bool $deleteActive,
    ): void {
        if ($deleteRound) {
            CacheService::delete($roundKey);
        }
        if ($deleteActive) {
            CacheService::delete($activeKey);
        }
    }
}
