<?php

/**
 * Mini-game Repository
 *
 * Database operations for game state, gameplay settings, leaderboard,
 * and score submission.
 *
 * Security:
 * - All queries use prepared statements.
 * - Slugs are normalized via MinigameCatalog::normalizeSlug() before DB use.
 * - Scores are clamped to prevent overflow/abuse.
 * - This table stores only game metadata/state (no personal data) — GDPR-safe.
 */

declare(strict_types=1);

namespace NewSite\Minigames;

use NewSite\Database\DbHelper;
use PDO;
use Throwable;

class MinigameRepository
{
    private const UPSERT_UPDATED_AT = 'updated_at = EXCLUDED.updated_at';

    /**
     * Check whether the mini_games table exists.
     */
    public static function tableExists(PDO $db): bool
    {
        try {
            $stmt = $db->query("SELECT to_regclass('public.mini_games')");
            $reg = $stmt ? $stmt->fetchColumn() : null;

            return is_string($reg) && $reg !== '';
        } catch (Throwable) {
            return false;
        }
    }

    /**
     * Check whether the mini_game_leaderboard table exists.
     */
    public static function leaderboardTableExists(PDO $db): bool
    {
        try {
            $stmt = $db->query("SELECT to_regclass('public.mini_game_leaderboard')");
            $reg = $stmt ? $stmt->fetchColumn() : null;

            return is_string($reg) && $reg !== '';
        } catch (Throwable) {
            return false;
        }
    }

    /**
     * Fetch enabled flags directly from the database (no per-request cache).
     *
     * @return array<string, bool> slug => enabled
     */
    public static function fetchDbStates(PDO $db): array
    {
        $states = [];
        if (!self::tableExists($db)) {
            return $states;
        }

        try {
            $stmt = $db->query('SELECT slug, enabled FROM mini_games');
            if (!$stmt) {
                return $states;
            }

            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
            foreach ($rows as $row) {
                $slug = MinigameCatalog::normalizeSlug((string) ($row['slug'] ?? ''));
                if ($slug === '') {
                    continue;
                }
                $states[$slug] = ((int) ($row['enabled'] ?? 1)) === 1;
            }
        } catch (Throwable) {
            $states = [];
        }

        return $states;
    }

    /**
     * Fetch enabled flags with a tiny per-request cache.
     *
     * Admin toggles take effect on the next request.
     *
     * @return array<string, bool> slug => enabled
     */
    public static function getDbStates(PDO $db): array
    {
        /** @var array<string, bool>|null */
        static $cached = null;
        static $cachedAt = 0;

        if (is_array($cached) && (time() - (int) $cachedAt) < 2) {
            return $cached;
        }

        $cached   = self::fetchDbStates($db);
        $cachedAt = time();

        return $cached;
    }

    /**
     * Check whether a single game is enabled.
     */
    public static function isEnabled(PDO $db, string $slug, bool $default = true): bool
    {
        $slug = MinigameCatalog::normalizeSlug($slug);
        if ($slug === '') {
            return false;
        }

        $states = self::getDbStates($db);

        return array_key_exists($slug, $states) ? (bool) $states[$slug] : $default;
    }

    /**
     * Return enabled mini-games in display order.
     *
     * @return array<int, array{slug:string,title:string,description:string,icon:string,sort:int}>
     */
    public static function getEnabled(PDO $db): array
    {
        $catalog = MinigameCatalog::getCatalog();
        $states  = self::getDbStates($db);
        $games   = [];

        foreach ($catalog as $slug => $game) {
            if (!is_array($game)) {
                continue;
            }
            $s = MinigameCatalog::normalizeSlug((string) ($game['slug'] ?? $slug));
            if ($s === '') {
                continue;
            }
            if (!($states[$s] ?? true)) {
                continue;
            }

            $games[] = [
                'slug'        => $s,
                'title'       => (string) ($game['title'] ?? $s),
                'description' => (string) ($game['description'] ?? ''),
                'icon'        => (string) ($game['icon'] ?? ''),
                'sort'        => (int) ($game['sort'] ?? 0),
            ];
        }

        usort($games, static function (array $a, array $b): int {
            $cmp = ((int) ($a['sort'] ?? 0)) <=> ((int) ($b['sort'] ?? 0));

            return $cmp !== 0 ? $cmp : strcmp((string) ($a['title'] ?? ''), (string) ($b['title'] ?? ''));
        });

        return $games;
    }

    /**
     * Admin helper: upsert enabled flag (and keep titles in sync with catalog).
     */
    public static function setEnabled(PDO $db, string $slug, bool $enabled): bool
    {
        $slug    = MinigameCatalog::normalizeSlug($slug);
        $catalog = MinigameCatalog::getCatalog();
        $ok      = false;

        $isKnown = $slug !== '' && isset($catalog[$slug]);
        if ($isKnown && self::tableExists($db)) {
            $title = (string) ($catalog[$slug]['title'] ?? $slug);
            $sort  = (int)    ($catalog[$slug]['sort']  ?? 0);
            $now   = DbHelper::nowString();

            $stmt = $db->prepare(
                'INSERT INTO mini_games (slug, title, enabled, sort_order, created_at, updated_at) '
                . 'VALUES (?, ?, ?, ?, ?, ?) '
                . 'ON CONFLICT (slug) DO UPDATE SET title = EXCLUDED.title, '
                . 'enabled = EXCLUDED.enabled, sort_order = EXCLUDED.sort_order, '
                . self::UPSERT_UPDATED_AT
            );

            $ok = (bool) $stmt->execute([
                $slug,
                $title,
                $enabled ? 1 : 0,
                $sort,
                $now,
                $now,
            ]);
        }

        return $ok;
    }

    /**
     * Sanitize gameplay settings against defaults and clamp to safe ranges.
     *
     * @param array<string, mixed> $settings Raw input
     * @return array{multiplier_step:float,start_time_seconds:int,time_bonus_on_make:int,time_penalty_on_miss:int,max_time_seconds:int}|array{}
     */
    public static function sanitizeGameplaySettings(string $slug, array $settings): array
    {
        $defaults = MinigameCatalog::getGameplayDefaults($slug);
        if (empty($defaults)) {
            return [];
        }

        $clean = self::clampGameplayValues($settings, $defaults);

        // Max time must not be less than start time.
        if ($clean['max_time_seconds'] < $clean['start_time_seconds']) {
            $clean['max_time_seconds'] = $clean['start_time_seconds'];
        }

        return $clean;
    }

    /**
     * Fetch gameplay settings from DB with fallback to defaults.
     *
     * @return array{multiplier_step:float,start_time_seconds:int,time_bonus_on_make:int,time_penalty_on_miss:int,max_time_seconds:int}|array{}
     */
    public static function getGameplaySettings(PDO $db, string $slug): array
    {
        $slug     = MinigameCatalog::normalizeSlug($slug);
        $defaults = MinigameCatalog::getGameplayDefaults($slug);
        if (empty($defaults)) {
            return [];
        }

        $raw = self::fetchGameplayRow($db, $slug);

        return !empty($raw) ? self::sanitizeGameplaySettings($slug, $raw) : $defaults;
    }

    /**
     * Save gameplay settings to the database.
     *
     * @param array<string, mixed> $settings
     */
    public static function setGameplaySettings(PDO $db, string $slug, array $settings): bool
    {
        $slug    = MinigameCatalog::normalizeSlug($slug);
        $catalog = MinigameCatalog::getCatalog();
        $ok      = false;

        $isValid = $slug !== '' && isset($catalog[$slug]) && self::tableExists($db);
        if ($isValid) {
            $ok = self::upsertGameplaySettings($db, $slug, $catalog[$slug], $settings);
        }

        return $ok;
    }

    /**
     * Fetch leaderboard entries for a specific game.
     *
     * @return array<int, array{rank:int,user_id:int,name:string,score:float,is_current_user:bool,profile_url:string}>
     */
    public static function getLeaderboard(
        PDO $db,
        string $slug,
        int $limit = 50,
        int $viewerUserId = 0,
    ): array {
        $slug    = MinigameCatalog::normalizeSlug($slug);
        $catalog = MinigameCatalog::getCatalog();
        $isValid = $slug !== '' && isset($catalog[$slug]) && self::leaderboardTableExists($db);

        if (!$isValid) {
            return [];
        }

        $rows = self::fetchLeaderboardRows($db, $slug, $limit);

        return self::buildLeaderboardResult($rows, $viewerUserId);
    }

    /**
     * Submit a score for a mini-game leaderboard.
     */
    public static function submitScore(PDO $db, string $slug, int $userId, float $score): bool
    {
        $slug    = MinigameCatalog::normalizeSlug($slug);
        $catalog = MinigameCatalog::getCatalog();
        $isValid = $userId > 0 && $slug !== '' && isset($catalog[$slug]) && self::leaderboardTableExists($db);

        if (!$isValid) {
            return false;
        }

        $safeScore = round(max(0.0, min($score, 1000000.0)), 2);

        return self::upsertScore($db, $slug, $userId, $safeScore);
    }

    // ------------------------------------------------------------------
    // Private helpers
    // ------------------------------------------------------------------

    /**
     * Clamp raw gameplay values to safe ranges.
     *
     * @param array<string, mixed> $settings
     * @param array<string, mixed> $defaults
     * @return array{multiplier_step:float,start_time_seconds:int,time_bonus_on_make:int,time_penalty_on_miss:int,max_time_seconds:int}
     */
    private static function clampGameplayValues(array $settings, array $defaults): array
    {
        $ms = self::clampFloat($settings, $defaults, 'multiplier_step', 1.01, 3.00, 4);
        $st = self::clampInt($settings, $defaults, 'start_time_seconds', 5, 240);
        $tb = self::clampInt($settings, $defaults, 'time_bonus_on_make', 0, 60);
        $tp = self::clampInt($settings, $defaults, 'time_penalty_on_miss', 0, 120);
        $mt = self::clampInt($settings, $defaults, 'max_time_seconds', 10, 360);

        return [
            'multiplier_step'      => $ms,
            'start_time_seconds'   => $st,
            'time_bonus_on_make'   => $tb,
            'time_penalty_on_miss' => $tp,
            'max_time_seconds'     => $mt,
        ];
    }

    /**
     * Clamp a single float setting to a safe range.
     *
     * @param array<string, mixed> $settings
     * @param array<string, mixed> $defaults
     */
    private static function clampFloat(
        array $settings,
        array $defaults,
        string $key,
        float $min,
        float $max,
        int $precision,
    ): float {
        $raw = $settings[$key] ?? $defaults[$key];
        $val = is_numeric($raw) ? (float) $raw : (float) $defaults[$key];
        if (!is_finite($val)) {
            $val = (float) $defaults[$key];
        }

        return round(min($max, max($min, $val)), $precision);
    }

    /**
     * Clamp a single integer setting to a safe range.
     *
     * @param array<string, mixed> $settings
     * @param array<string, mixed> $defaults
     */
    private static function clampInt(
        array $settings,
        array $defaults,
        string $key,
        int $min,
        int $max,
    ): int {
        $raw = $settings[$key] ?? $defaults[$key];
        $val = is_numeric($raw) ? (int) $raw : (int) $defaults[$key];

        return min($max, max($min, $val));
    }

    /**
     * Fetch raw gameplay settings row from the database.
     *
     * @return array<string, mixed>
     */
    private static function fetchGameplayRow(PDO $db, string $slug): array
    {
        if (!self::tableExists($db)) {
            return [];
        }

        try {
            $stmt = $db->prepare(
                'SELECT multiplier_step, start_time_seconds, time_bonus_on_make, '
                . 'time_penalty_on_miss, max_time_seconds '
                . 'FROM mini_games WHERE slug = ? LIMIT 1'
            );
            $stmt->execute([$slug]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            return is_array($row) ? $row : [];
        } catch (Throwable) {
            return [];
        }
    }

    /**
     * Upsert gameplay settings into the database.
     *
     * @param array<string, mixed> $catalogEntry
     * @param array<string, mixed> $settings
     */
    private static function upsertGameplaySettings(
        PDO $db,
        string $slug,
        array $catalogEntry,
        array $settings,
    ): bool {
        $clean = self::sanitizeGameplaySettings($slug, $settings);
        if (empty($clean)) {
            return false;
        }

        $title = (string) ($catalogEntry['title'] ?? $slug);
        $sort  = (int)    ($catalogEntry['sort']  ?? 0);
        $now   = DbHelper::nowString();

        try {
            $stmt = $db->prepare(
                'INSERT INTO mini_games '
                . '(slug, title, enabled, sort_order, multiplier_step, start_time_seconds, '
                . 'time_bonus_on_make, time_penalty_on_miss, max_time_seconds, created_at, updated_at) '
                . 'VALUES (?, ?, 1, ?, ?, ?, ?, ?, ?, ?, ?) '
                . 'ON CONFLICT (slug) DO UPDATE SET '
                . 'title = EXCLUDED.title, '
                . 'sort_order = EXCLUDED.sort_order, '
                . 'multiplier_step = EXCLUDED.multiplier_step, '
                . 'start_time_seconds = EXCLUDED.start_time_seconds, '
                . 'time_bonus_on_make = EXCLUDED.time_bonus_on_make, '
                . 'time_penalty_on_miss = EXCLUDED.time_penalty_on_miss, '
                . 'max_time_seconds = EXCLUDED.max_time_seconds, '
                . self::UPSERT_UPDATED_AT
            );

            return (bool) $stmt->execute([
                $slug,
                $title,
                $sort,
                $clean['multiplier_step'],
                $clean['start_time_seconds'],
                $clean['time_bonus_on_make'],
                $clean['time_penalty_on_miss'],
                $clean['max_time_seconds'],
                $now,
                $now,
            ]);
        } catch (Throwable) {
            return false;
        }
    }

    /**
     * Fetch raw leaderboard rows from the database.
     *
     * @return array<int, array<string, mixed>>
     */
    private static function fetchLeaderboardRows(PDO $db, string $slug, int $limit): array
    {
        $safeLimit = max(1, min(50, $limit));

        try {
            $stmt = $db->prepare(
                'SELECT l.user_id, l.best_score, l.updated_at, '
                . "NULLIF(u.nickname, '') AS nickname, "
                . "COALESCE(NULLIF(u.nickname, ''), NULLIF(u.display_name, ''), "
                . "('Player #' || l.user_id::text)) AS player_name "
                . 'FROM mini_game_leaderboard l '
                . 'LEFT JOIN site_users u ON u.id = l.user_id '
                . 'WHERE l.game_slug = ? '
                . 'ORDER BY l.best_score DESC, l.updated_at ASC, l.user_id ASC '
                . 'LIMIT ' . $safeLimit
            );
            $stmt->execute([$slug]);

            return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        } catch (Throwable) {
            return [];
        }
    }

    /**
     * Transform raw leaderboard rows into the public result format.
     *
     * @param array<int, array<string, mixed>> $rows
     * @return array<int, array{rank:int,user_id:int,name:string,score:float,is_current_user:bool,profile_url:string}>
     */
    private static function buildLeaderboardResult(array $rows, int $viewerUserId): array
    {
        $results = [];
        $rank    = 1;

        foreach ($rows as $row) {
            $userId   = (int) ($row['user_id'] ?? 0);
            $nickname = trim((string) ($row['nickname'] ?? ''));
            $profileUrl = $nickname !== '' ? '/@' . rawurlencode($nickname) : '';

            $results[] = [
                'rank'            => $rank,
                'user_id'         => $userId,
                'name'            => trim((string) ($row['player_name'] ?? 'Player')),
                'score'           => round((float) ($row['best_score'] ?? 0), 2),
                'is_current_user' => $viewerUserId > 0 && $viewerUserId === $userId,
                'profile_url'     => $profileUrl,
            ];
            $rank++;
        }

        return $results;
    }

    /**
     * Upsert a leaderboard score entry.
     */
    private static function upsertScore(PDO $db, string $slug, int $userId, float $safeScore): bool
    {
        $now = DbHelper::nowString();

        try {
            $stmt = $db->prepare(
                'INSERT INTO mini_game_leaderboard '
                . '(game_slug, user_id, best_score, last_score, total_plays, created_at, updated_at) '
                . 'VALUES (?, ?, ?, ?, 1, ?, ?) '
                . 'ON CONFLICT (game_slug, user_id) DO UPDATE SET '
                . 'best_score = GREATEST(mini_game_leaderboard.best_score, EXCLUDED.best_score), '
                . 'last_score = EXCLUDED.last_score, '
                . 'total_plays = mini_game_leaderboard.total_plays + 1, '
                . self::UPSERT_UPDATED_AT
            );

            return (bool) $stmt->execute([
                $slug,
                $userId,
                $safeScore,
                $safeScore,
                $now,
                $now,
            ]);
        } catch (Throwable) {
            return false;
        }
    }
}
