<?php

/**
 * Mini-game Catalog
 *
 * Pure helpers for slug normalization, catalog lookup, and route mapping.
 * No database or cache interaction — safe to call from any context.
 *
 * Security: slug normalization uses a strict [a-z0-9-] allowlist.
 */

declare(strict_types=1);

namespace NewSite\Minigames;

class MinigameCatalog
{
    /**
     * Canonical mini-game catalog used by both public pages and admin.
     *
     * IMPORTANT: Keep internal slugs stable (used in DB primary key and
     * score tables).
     *
     * @return array<string, array{slug:string,title:string,description:string,icon:string,sort:int}>
     */
    public static function getCatalog(): array
    {
        return [
            'basketball-launch' => [
                'slug'        => 'basketball-launch',
                'title'       => 'Basketball Launch',
                'description' => 'Aim, launch, and rack up multiplier points.',
                'icon'        => '🏀',
                'sort'        => 10,
            ],
            'maze-runner' => [
                'slug'        => 'maze-runner',
                'title'       => 'Hardest Maze game V1',
                'description' => 'Survive relentless door shifts in a brutal precision maze challenge.',
                'icon'        => '🧩',
                'sort'        => 20,
            ],
        ];
    }

    /**
     * Map an internal slug to its public route slug.
     */
    public static function getPublicSlug(string $slug): string
    {
        $internal = self::normalizeSlug($slug);

        return match ($internal) {
            'maze-runner' => 'hardest-maze-game-v1',
            default       => $internal,
        };
    }

    /**
     * Resolve a public route slug back to its internal catalog slug.
     */
    public static function resolveFromRoute(string $routeSlug): string
    {
        $public = self::normalizeSlug($routeSlug);

        return match ($public) {
            'hardest-maze-game-v1' => 'maze-runner',
            default                => $public,
        };
    }

    /**
     * Normalize a slug to strict [a-z0-9-] allowlist.
     *
     * Security: prevents injection via slug parameters.
     */
    public static function normalizeSlug(string $slug): string
    {
        $slug = strtolower(trim($slug));
        $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);

        return is_string($slug) ? $slug : '';
    }

    /**
     * Per-game gameplay setting defaults (no DB lookup).
     *
     * @return array{multiplier_step:float,start_time_seconds:int,time_bonus_on_make:int,time_penalty_on_miss:int,max_time_seconds:int}|array{}
     */
    public static function getGameplayDefaults(string $slug): array
    {
        $slug = self::normalizeSlug($slug);

        if ($slug === 'basketball-launch') {
            return [
                'multiplier_step'      => 1.12,
                'start_time_seconds'   => 30,
                'time_bonus_on_make'   => 5,
                'time_penalty_on_miss' => 10,
                'max_time_seconds'     => 99,
            ];
        }

        return [];
    }

    /**
     * Per-game anti-cheat policy for verified leaderboard rounds.
     *
     * @return array{round_ttl_seconds:int,min_submit_seconds:int,max_submit_seconds:int,max_points_per_second:float,base_score_allowance:float,max_score:float}
     */
    public static function getRoundPolicy(string $slug): array
    {
        $slug = self::normalizeSlug($slug);

        $policy = self::defaultRoundPolicy();

        if ($slug === 'basketball-launch') {
            // Basketball-launch uses the default policy values.
            return $policy;
        }

        return $policy;
    }

    /**
     * Base anti-cheat policy shared by all games.
     *
     * @return array{round_ttl_seconds:int,min_submit_seconds:int,max_submit_seconds:int,max_points_per_second:float,base_score_allowance:float,max_score:float}
     */
    private static function defaultRoundPolicy(): array
    {
        return [
            'round_ttl_seconds'    => 300,
            'min_submit_seconds'   => 8,
            'max_submit_seconds'   => 300,
            'max_points_per_second' => 130.0,
            'base_score_allowance' => 220.0,
            'max_score'            => 25000.0,
        ];
    }
}
