<?php

declare(strict_types=1);

namespace NewSite\Menu;

/**
 * MenuService — hierarchical menu-item retrieval.
 *
 * Security: all database queries use prepared statements with bound
 * parameters. Recursion depth is capped to prevent stack exhaustion.
 */
final class MenuService
{
    public static function getItemsRecursive(\PDO $db, int $menuId, ?int $parentId = null, int $depth = 0, int $maxDepth = 5): array
    {
        if ($depth > $maxDepth) {
            return [];
        }
        if ($parentId === null) {
            $stmt = $db->prepare("
                SELECT mi.*, p.slug as page_slug, p.title as page_title
                FROM menu_items mi
                LEFT JOIN pages p ON mi.page_id = p.id
                WHERE mi.menu_id = ? AND mi.parent_id IS NULL
                ORDER BY mi.sort_order
            ");
            $stmt->execute([$menuId]);
        } else {
            $stmt = $db->prepare("
                SELECT mi.*, p.slug as page_slug, p.title as page_title
                FROM menu_items mi
                LEFT JOIN pages p ON mi.page_id = p.id
                WHERE mi.parent_id = ?
                ORDER BY mi.sort_order
            ");
            $stmt->execute([$parentId]);
        }
        $items = $stmt->fetchAll();
        foreach ($items as &$item) {
            $item['children'] = self::getItemsRecursive($db, $menuId, (int)$item['id'], $depth + 1, $maxDepth);
            $item['depth'] = $depth;
        }
        return $items;
    }
}
