<?php

declare(strict_types=1);

namespace NewSite\Logging;

use NewSite\Logging\LogService;

/**
 * RequestLogger — HTTP request logging with PII redaction and sampling.
 *
 * Security: All POST data is sanitised before persistence; sensitive fields
 * are replaced with [REDACTED]. High-frequency endpoints are sampled at 1 %
 * to limit log volume. No raw credentials or tokens are ever written to logs.
 */
final class RequestLogger
{
    /** @var string[] Endpoints hit very frequently (sampled at 1 %). */
    private const HIGH_FREQ_ENDPOINTS = [
        'cart-api', 'notifications-api', 'messages-api', 'messenger-api',
        'friends-list', 'friends-search', 'friends-action', 'log-error',
    ];

    /** @var string[] Endpoints whose requests are never logged. */
    private const SKIP_LOG_ENDPOINTS = ['log-error'];

    public static function uriContainsAny(string $uri, array $needles): bool
    {
        foreach ($needles as $needle) {
            $needle = (string) $needle;
            if ($needle !== '' && strpos($uri, $needle) !== false) {
                return true;
            }
        }
        return false;
    }

    public static function getRequestLogType(string $uri): string
    {
        if (strpos(INIT_SCRIPT, '/admin/') !== false) {
            return 'admin_request';
        }
        if (strpos(INIT_SCRIPT, '/api/') !== false || strpos($uri, '-api') !== false) {
            return 'api_request';
        }
        return 'request';
    }

    public static function isSensitivePostField(string $key, array $sensitiveFields): bool
    {
        return in_array($key, $sensitiveFields, true)
            || str_contains($key, 'password')
            || str_contains($key, 'token')
            || str_contains($key, 'secret')
            || str_contains($key, 'auth')
            || str_contains($key, 'cookie')
            || str_contains($key, 'session')
            || str_contains($key, 'email')
            || str_contains($key, 'phone')
            || str_contains($key, 'address');
    }

    public static function sanitizePostDataForLogging(array $postData): array
    {
        $sensitiveFields = [
            'password', 'password_confirm', 'current_password', 'new_password',
            'card_number', 'cvv', 'cvc', 'token', 'secret', 'api_key',
            'email', 'phone', 'address', 'name', 'first_name', 'last_name',
            'session', 'session_id', 'cookie', 'auth', 'authorization', 'oauth',
            'csrf', 'csrf_token', 'ssn', 'iban', 'bic',
        ];

        foreach ($postData as $key => $_value) {
            $lowerKey = strtolower((string) $key);
            if (self::isSensitivePostField($lowerKey, $sensitiveFields)) {
                $postData[$key] = '[REDACTED]';
            }
        }

        return $postData;
    }

    public static function formatUploadedFilesForLogging(array $files): array
    {
        $uploads = [];
        foreach ($files as $key => $file) {
            $name = $file['name'] ?? '';
            if (is_array($name)) {
                foreach ($name as $i => $singleName) {
                    $uploads[] = "$key: $singleName (" . ($file['size'][$i] ?? 0) . ' bytes)';
                }
                continue;
            }
            $uploads[] = "$key: {$file['name']} ({$file['size']} bytes)";
        }
        return $uploads;
    }

    public static function logCurrentRequest(): void
    {
        if (empty($GLOBALS['_LOG_REQUEST'])) {
            return;
        }
        $GLOBALS['_LOG_REQUEST'] = false;

        $method  = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $uri     = $_SERVER['REQUEST_URI'] ?? '/';
        $script  = basename(INIT_SCRIPT);
        $referer = $_SERVER['HTTP_REFERER'] ?? 'Direct';

        if (self::uriContainsAny($uri, self::SKIP_LOG_ENDPOINTS)) {
            return;
        }

        $isHighFreq = self::uriContainsAny($uri, self::HIGH_FREQ_ENDPOINTS);
        if ($isHighFreq && mt_rand(1, 100) > 1) {
            return;
        }

        $message = "$method $uri";
        if ($isHighFreq) {
            $message .= ' [SAMPLED 1%]';
        }

        LogService::add(self::getRequestLogType($uri), $message, "Script: $script\nReferer: $referer");

        if ($method === 'POST' && !empty($_POST)) {
            $postData = self::sanitizePostDataForLogging($_POST);
            LogService::add('post_data', "POST to $uri", json_encode($postData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        }

        if (!empty($_FILES)) {
            $uploads = self::formatUploadedFilesForLogging($_FILES);
            LogService::add('file_upload', "Files uploaded to $uri", implode("\n", $uploads));
        }
    }
}
