<?php

declare(strict_types=1);

namespace NewSite\Logging;

use NewSite\Database\DatabaseManager;
use NewSite\Security\IpService;
use NewSite\Settings\SettingsService;

/**
 * Centralized logging service.
 *
 * Security: truncates messages to prevent log injection / overflow.
 * GDPR: stores anonymized IPs via IpService.
 * All database queries use prepared statements.
 */
final class LogService
{
    private static bool $loggingEnabled = true;

    public static function add(string $type, string $message, ?string $details = null): bool
    {
        if (!self::$loggingEnabled) {
            return false;
        }
        try {
            self::$loggingEnabled = false;
            $db = DatabaseManager::getWriteConnection();
            $storedIp = IpService::gdprStore($_SERVER['REMOTE_ADDR'] ?? null);
            if ($details !== null && strpos($details, 'Script:') === false) {
                $details .= "\nScript: " . (defined('INIT_SCRIPT') ? INIT_SCRIPT : 'unknown');
            }
            if ($details !== null && strpos($details, 'URI:') === false) {
                $details .= "\nURI: " . ($_SERVER['REQUEST_URI'] ?? 'unknown');
            }
            $stmt = $db->prepare(
                "INSERT INTO logs (log_type, message, details, ip_address, user_agent, user_id) "
                . "VALUES (?, ?, ?, ?, ?, ?)"
            );
            $stmt->execute([
                $type,
                substr($message, 0, 500),
                substr($details ?? '', 0, 5000),
                $storedIp,
                $_SERVER['HTTP_USER_AGENT'] ?? null,
                $_SESSION['admin_user_id'] ?? $_SESSION['site_user_id'] ?? null
            ]);
            self::$loggingEnabled = true;
            return true;
        } catch (\Exception $e) {
            self::$loggingEnabled = true;
            error_log('LogService::add failed: ' . $e->getMessage());
            return false;
        }
    }

    public static function action(string $action, mixed $details = null): bool
    {
        $userId = $_SESSION['site_user_id'] ?? $_SESSION['admin_user_id'] ?? null;
        $extra = "User: " . ($userId ?? 'guest');
        if ($details !== null) {
            $extra .= "\n" . (is_array($details) ? json_encode($details, JSON_PRETTY_PRINT) : (string)$details);
        }
        return self::add('action', $action, $extra);
    }

    public static function query(string $sql, array $params = [], ?float $duration = null): bool
    {
        if (SettingsService::get('log_query_details', '0') !== '1') {
            return false;
        }
        $compactSql = preg_replace('/\s+/', ' ', trim($sql));
        $details = "SQL: $compactSql";
        $details .= "\nParam count: " . count($params);
        if ($duration !== null) {
            $details .= "\nDuration: " . round($duration * 1000, 2) . "ms";
        }
        return self::add('query', 'DB Query', $details);
    }

    public static function security(string $event, ?string $details = null): bool
    {
        return self::add('security', $event, $details);
    }

    public static function getAll(?string $type = null, int $limit = 100, int $offset = 0): array
    {
        $db = DatabaseManager::getWriteConnection();
        if ($type !== null) {
            $stmt = $db->prepare("SELECT * FROM logs WHERE log_type = ? ORDER BY created_at DESC LIMIT ? OFFSET ?");
            $stmt->execute([$type, $limit, $offset]);
        } else {
            $stmt = $db->prepare("SELECT * FROM logs ORDER BY created_at DESC LIMIT ? OFFSET ?");
            $stmt->execute([$limit, $offset]);
        }
        return $stmt->fetchAll();
    }

    public static function count(?string $type = null): int
    {
        $db = DatabaseManager::getWriteConnection();
        if ($type !== null) {
            $stmt = $db->prepare("SELECT COUNT(*) FROM logs WHERE log_type = ?");
            $stmt->execute([$type]);
        } else {
            $stmt = $db->query("SELECT COUNT(*) FROM logs");
        }
        return (int)$stmt->fetchColumn();
    }

    public static function clearOld(int $days = 30): int
    {
        $db = DatabaseManager::getWriteConnection();
        $cutoff = date('Y-m-d H:i:s', time() - ($days * 86400));
        $stmt = $db->prepare("DELETE FROM logs WHERE created_at < ?");
        $stmt->execute([$cutoff]);
        return $stmt->rowCount();
    }
}
