<?php

declare(strict_types=1);

namespace NewSite\Image;

/**
 * On-the-fly image resizing and format conversion with disk caching.
 *
 * Security: Only processes whitelisted image MIME types; validates all
 * parameters; output paths use deterministic hashes stored in a protected
 * cache directory. No user-supplied filenames appear in cache keys.
 */
final class ImageOptimizer
{
    /** Maximum allowed target height in pixels. */
    private const MAX_HEIGHT = 1200;

    /** Maximum allowed target width in pixels. */
    private const MAX_WIDTH = 2000;

    private const MIME_PNG  = 'image/png';
    private const MIME_JPEG = 'image/jpeg';
    private const MIME_WEBP = 'image/webp';
    private const MIME_GIF  = 'image/gif';

    /** Supported output formats mapped to MIME types. */
    private const FORMAT_MAP = [
        'webp' => self::MIME_WEBP,
        'png'  => self::MIME_PNG,
        'jpg'  => self::MIME_JPEG,
        'jpeg' => self::MIME_JPEG,
    ];

    /** Source MIME types that can be processed. */
    private const ALLOWED_SOURCE_MIMES = [
        self::MIME_PNG,
        self::MIME_JPEG,
        self::MIME_WEBP,
        self::MIME_GIF,
    ];

    /** WebP output quality (0–100). */
    private const WEBP_QUALITY = 80;

    /** JPEG output quality (0–100). */
    private const JPEG_QUALITY = 85;

    /** PNG compression level (0–9, where 9 is max compression). */
    private const PNG_COMPRESSION = 8;

    /**
     * Get an optimized version of an image, resized and/or converted.
     *
     * Returns cached file path + MIME type, or null on failure.
     * Falls back gracefully if GD extension or target format is unavailable.
     *
     * @param string   $sourcePath Absolute path to the source image file
     * @param int|null $height     Target height in pixels (null = keep original height)
     * @param string   $format     Target format: webp, png, jpg, jpeg
     * @param int|null $width      Target width in pixels (null = keep proportional)
     *
     * @return array{path: string, mime: string}|null
     */
    public static function getOptimized(
        string $sourcePath,
        ?int $height,
        string $format = 'webp',
        ?int $width = null
    ): ?array {
        $format = strtolower($format);

        if (!self::isValidRequest($sourcePath, $height, $format, $width)) {
            return null;
        }

        // Fall back to PNG if WebP is not compiled into GD
        if ($format === 'webp' && !function_exists('imagewebp')) {
            $format = 'png';
        }

        $targetMime = self::FORMAT_MAP[$format];
        $cacheDir   = dirname(__DIR__, 2) . '/data/cache/images';
        self::ensureCacheDirectory($cacheDir);

        $cacheKey  = self::buildCacheKey($sourcePath, $height, $format, $width);
        $cacheExt  = ($format === 'jpeg') ? 'jpg' : $format;
        $cachePath = $cacheDir . '/' . $cacheKey . '.' . $cacheExt;

        // Serve from cache if fresh, otherwise generate and cache
        $isCached = is_file($cachePath) && filemtime($cachePath) >= filemtime($sourcePath);
        $isReady  = $isCached || self::processImage($sourcePath, $cachePath, $height, $format, $width);

        return $isReady ? ['path' => $cachePath, 'mime' => $targetMime] : null;
    }

    /**
     * Check if GD extension is available with WebP support.
     */
    public static function isWebpSupported(): bool
    {
        return extension_loaded('gd') && function_exists('imagewebp');
    }

    /**
     * Validate all preconditions for an optimization request.
     */
    private static function isValidRequest(string $sourcePath, ?int $height, string $format, ?int $width): bool
    {
        $formatValid = isset(self::FORMAT_MAP[$format]);
        $heightValid = $height === null || ($height >= 1 && $height <= self::MAX_HEIGHT);
        $widthValid  = $width === null || ($width >= 1 && $width <= self::MAX_WIDTH);
        $fileValid   = is_file($sourcePath) && is_readable($sourcePath);
        $gdAvailable = extension_loaded('gd');

        if (!$formatValid || !$heightValid || !$widthValid || !$fileValid || !$gdAvailable) {
            return false;
        }

        // Security: verify source is an allowed image type via magic bytes
        $finfo      = new \finfo(FILEINFO_MIME_TYPE);
        $sourceMime = $finfo->file($sourcePath);

        return in_array($sourceMime, self::ALLOWED_SOURCE_MIMES, true);
    }

    /**
     * Ensure cache directory exists with security protections.
     */
    private static function ensureCacheDirectory(string $cacheDir): void
    {
        if (!is_dir($cacheDir)) {
            mkdir($cacheDir, 0755, true);
        }

        // Block direct PHP execution in the cache directory
        $htaccessPath = $cacheDir . '/.htaccess';
        if (!is_file($htaccessPath)) {
            file_put_contents(
                $htaccessPath,
                "Options -Indexes\n<FilesMatch \"\\.php$\">\n    Deny from all\n</FilesMatch>\n"
                . "AddType application/octet-stream .php .php5 .phtml\n"
            );
        }
    }

    /**
     * Build a deterministic cache key from source file identity and parameters.
     *
     * Includes file modification time + size so cache auto-invalidates when
     * the source changes.
     */
    private static function buildCacheKey(string $sourcePath, ?int $height, string $format, ?int $width): string
    {
        $sourceIdentity = $sourcePath . '|' . filemtime($sourcePath) . '|' . filesize($sourcePath);
        $heightPart     = ($height !== null) ? 'h' . $height : 'orig';
        $widthPart      = ($width !== null) ? 'w' . $width : '';

        return hash('sha256', $sourceIdentity . '|' . $heightPart . '|' . $widthPart . '|' . $format);
    }

    /**
     * Resize and/or convert an image using GD, writing the result to disk.
     */
    private static function processImage(
        string $sourcePath,
        string $outputPath,
        ?int $targetHeight,
        string $outputFormat,
        ?int $targetWidth = null
    ): bool {
        $sourceImage = self::createFromFile($sourcePath);
        if ($sourceImage === null) {
            return false;
        }

        $resized = self::resizeImage($sourceImage, $targetHeight, $outputFormat, $targetWidth);
        // GdImage objects are freed automatically when out of scope (PHP 8.0+)
        unset($sourceImage);

        $saved = false;
        if ($resized !== null) {
            $saved = self::saveImage($resized, $outputPath, $outputFormat);
            unset($resized);
        }

        return $saved;
    }

    /**
     * Create a resized GD image from a source, preserving alpha if needed.
     */
    private static function resizeImage(
        \GdImage $source,
        ?int $targetHeight,
        string $outputFormat,
        ?int $targetWidth = null
    ): ?\GdImage {
        $origWidth  = imagesx($source);
        $origHeight = imagesy($source);

        if ($origWidth === 0 || $origHeight === 0) {
            return null;
        }

        [$newWidth, $newHeight] = self::calculateDimensions($origWidth, $origHeight, $targetHeight, $targetWidth);
        $canvas = self::createCanvas($newWidth, $newHeight, $outputFormat);

        $result = null;
        if ($canvas !== null) {
            $ok = imagecopyresampled($canvas, $source, 0, 0, 0, 0, $newWidth, $newHeight, $origWidth, $origHeight);
            if ($ok) {
                $result = $canvas;
            }
            // Canvas freed automatically if not returned (PHP 8.0+ GdImage)
        }

        return $result;
    }

    /**
     * Calculate proportional target dimensions, respecting MAX_WIDTH.
     *
     * @return array{0: int, 1: int} [width, height]
     */
    private static function calculateDimensions(
        int $origWidth,
        int $origHeight,
        ?int $targetHeight,
        ?int $targetWidth = null
    ): array {
        $newWidth  = $origWidth;
        $newHeight = $origHeight;

        // Width-based resize takes precedence when provided
        if ($targetWidth !== null && $targetWidth < $origWidth) {
            $ratio     = $targetWidth / $origWidth;
            $newWidth  = $targetWidth;
            $newHeight = max(1, (int) round($origHeight * $ratio));
        } elseif ($targetHeight !== null && $targetHeight < $origHeight) {
            $ratio     = $targetHeight / $origHeight;
            $newWidth  = max(1, (int) round($origWidth * $ratio));
            $newHeight = $targetHeight;
        }

        // Safety caps
        if ($newWidth > self::MAX_WIDTH) {
            $ratio     = self::MAX_WIDTH / $newWidth;
            $newWidth  = self::MAX_WIDTH;
            $newHeight = max(1, (int) round($newHeight * $ratio));
        }
        if ($newHeight > self::MAX_HEIGHT) {
            $ratio     = self::MAX_HEIGHT / $newHeight;
            $newHeight = self::MAX_HEIGHT;
            $newWidth  = max(1, (int) round($newWidth * $ratio));
        }

        return [$newWidth, $newHeight];
    }

    /**
     * Create a blank GD canvas with alpha support for PNG/WebP.
     */
    private static function createCanvas(int $width, int $height, string $format): ?\GdImage
    {
        $canvas = imagecreatetruecolor($width, $height);
        if (!$canvas instanceof \GdImage) {
            return null;
        }

        // Preserve alpha transparency for PNG and WebP output
        if ($format === 'png' || $format === 'webp') {
            imagealphablending($canvas, false);
            imagesavealpha($canvas, true);
            $transparent = imagecolorallocatealpha($canvas, 0, 0, 0, 127);
            if ($transparent !== false) {
                imagefill($canvas, 0, 0, $transparent);
            }
        }

        return $canvas;
    }

    /**
     * Create a GD image resource from a file based on its MIME type.
     */
    private static function createFromFile(string $path): ?\GdImage
    {
        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mime  = $finfo->file($path);

        $image = match ($mime) {
            self::MIME_PNG  => @imagecreatefrompng($path),
            self::MIME_JPEG => @imagecreatefromjpeg($path),
            self::MIME_WEBP => function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($path) : false,
            self::MIME_GIF  => @imagecreatefromgif($path),
            default         => false,
        };

        return ($image instanceof \GdImage) ? $image : null;
    }

    /**
     * Save a GD image to disk in the specified format.
     */
    private static function saveImage(\GdImage $image, string $path, string $format): bool
    {
        return match ($format) {
            'webp'        => imagewebp($image, $path, self::WEBP_QUALITY),
            'png'         => imagepng($image, $path, self::PNG_COMPRESSION),
            'jpg', 'jpeg' => imagejpeg($image, $path, self::JPEG_QUALITY),
            default       => false,
        };
    }
}
