<?php

declare(strict_types=1);

namespace NewSite\Http;

/**
 * HTTP Response utility.
 *
 * Security notes:
 * - escape() uses ENT_QUOTES | UTF-8 to prevent XSS in HTML contexts.
 * - json() sets Content-Type header before output to prevent MIME-sniffing.
 * - No inline JavaScript or inline styles are emitted (strict CSP compliance).
 */
final class Response
{
    /**
     * Send a JSON response and terminate execution.
     *
     * @param mixed $data The data to encode as JSON.
     * @param int   $code HTTP status code (default 200).
     */
    public static function json($data, int $code = 200): never
    {
        http_response_code($code);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }

    /**
     * Escape a string for safe HTML output.
     *
     * @param string|null $string The raw string to escape.
     * @return string The escaped string safe for use in HTML attributes and content.
     */
    public static function escape(?string $string): string
    {
        return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
    }
}
