<?php

declare(strict_types=1);

namespace NewSite\Geo;

use NewSite\Database\DatabaseManager;
use NewSite\Security\IpService;
use NewSite\Settings\SettingsService;

/**
 * IP-to-country geolocation service with caching.
 *
 * Security: All database access uses prepared statements with bound parameters.
 * IP addresses are hashed via IpService::geoCacheKey() before storage.
 * External HTTP requests use a strict 2-second timeout.
 */
final class GeoService
{
    private const HTTP_CLIENT_USER_AGENT = 'NewSite/1.0';

    public static function getCachedCountry(string $ip): ?string
    {
        $db = DatabaseManager::getReadConnection();
        $key = IpService::geoCacheKey($ip);

        $stmt = $db->prepare("SELECT country_code, cached_at FROM geo_cache WHERE ip = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();

        return $result ? $result['country_code'] : null;
    }

    public static function cacheCountry(string $ip, string $countryCode): bool
    {
        $db = DatabaseManager::getWriteConnection();
        $key = IpService::geoCacheKey($ip);

        $stmt = $db->prepare(
            "INSERT INTO geo_cache (ip, country_code, cached_at)
             VALUES (?, ?, ?)
             ON CONFLICT (ip)
             DO UPDATE SET country_code = EXCLUDED.country_code,
                           cached_at    = EXCLUDED.cached_at"
        );
        return $stmt->execute([$key, strtoupper($countryCode), time()]);
    }

    public static function fetchGeoJsonPayload(string $url, bool $tryCurlFirst): string
    {
        $payload = '';

        if ($tryCurlFirst && function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
            curl_setopt($ch, CURLOPT_TIMEOUT, 2);
            curl_setopt($ch, CURLOPT_USERAGENT, self::HTTP_CLIENT_USER_AGENT);
            $response = curl_exec($ch);
            // CurlHandle auto-closes on scope exit (PHP 8.0+)

            if (is_string($response) && $response !== '') {
                $payload = $response;
            }
        }

        if ($payload === '') {
            $response = @file_get_contents($url);
            if (is_string($response) && $response !== '') {
                $payload = $response;
            }
        }

        return $payload;
    }

    public static function fetchCountryCodeFromSource(
        string $url,
        string $field,
        bool $tryCurlFirst,
    ): string {
        $countryCode = 'XX';
        $json = self::fetchGeoJsonPayload($url, $tryCurlFirst);

        if ($json !== '') {
            $data = json_decode($json, true);
            if (!empty($data[$field])) {
                $countryCode = strtoupper((string) $data[$field]);
            }
        }

        return $countryCode;
    }

    public static function getCountryLookupSources(string $ip, bool $isIpv6): array
    {
        if ($isIpv6) {
            return [
                ['url' => "https://ipapi.co/{$ip}/json/", 'field' => 'country_code', 'curl' => true],
                ['url' => "https://ipwho.is/{$ip}?fields=country_code", 'field' => 'country_code', 'curl' => false],
            ];
        }

        return [
            ['url' => "https://ip-api.com/json/{$ip}?fields=countryCode", 'field' => 'countryCode', 'curl' => true],
            ['url' => "http://ip-api.com/json/{$ip}?fields=countryCode", 'field' => 'countryCode', 'curl' => true],
            ['url' => "https://ipapi.co/{$ip}/json/", 'field' => 'country_code', 'curl' => false],
            ['url' => "https://ipwho.is/{$ip}?fields=country_code", 'field' => 'country_code', 'curl' => false],
        ];
    }

    public static function resolveCountryFromSources(array $sources): string
    {
        $country = 'XX';

        foreach ($sources as $source) {
            $url   = (string) ($source['url'] ?? '');
            $field = (string) ($source['field'] ?? '');
            $curl  = !empty($source['curl']);

            if ($url === '' || $field === '') {
                continue;
            }

            $country = self::fetchCountryCodeFromSource($url, $field, $curl);
            if ($country !== 'XX') {
                break;
            }
        }

        return $country;
    }

    public static function resolveCountry(string $ip, bool $force = false): string
    {
        $isPublicIp = filter_var(
            $ip,
            FILTER_VALIDATE_IP,
            FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE
        ) !== false;

        if (!$isPublicIp) {
            return 'LO';
        }

        $country = 'XX';

        if (SettingsService::get('geo_lookup_enabled', '1') === '1') {
            $cached = (!$force) ? self::getCachedCountry($ip) : null;
            if ($cached !== null) {
                $country = $cached;
            } else {
                $isIpv6  = filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) !== false;
                $sources = self::getCountryLookupSources($ip, $isIpv6);
                $country = self::resolveCountryFromSources($sources);
                self::cacheCountry($ip, $country);
            }
        }

        return $country;
    }
}
