<?php

declare(strict_types=1);

namespace NewSite\Gdpr;

use NewSite\Database\DatabaseManager;
use NewSite\Settings\SettingsService;
use NewSite\Logging\LogService;
use NewSite\User\UserService;

/**
 * GdprService — GDPR data-retention cleanup, user-data export and deletion.
 *
 * Security: all database queries use prepared statements. File paths are
 * resolved with basename() to prevent directory traversal. Exported archives
 * are written to the system temp directory and streamed to the client.
 */
final class GdprService
{
    private const PROFILE_PHOTO_ROUTE_PREFIX = '/profile-photo/';
    private const DB_DATETIME_FORMAT = 'Y-m-d H:i:s';

    public static function runRetentionCleanup(bool $force = false): void
    {
        $enabled = SettingsService::get('gdpr_auto_purge', '1') === '1';
        $lastRun = (int)SettingsService::get('gdpr_cleanup_last', '0');
        if (!$force && (!$enabled || ($lastRun > 0 && (time() - $lastRun) < 86400))) {
            return;
        }

        $logDays = max(1, (int)SettingsService::get('gdpr_retention_logs_days', '30'));
        $visitorDays = max(1, (int)SettingsService::get('gdpr_retention_visitors_days', '30'));
        $messageDays = max(1, (int)SettingsService::get('gdpr_retention_messages_days', '365'));
        $contactDays = max(1, (int)SettingsService::get('gdpr_retention_contact_days', '365'));
        $cartDays = max(1, (int)SettingsService::get('gdpr_retention_carts_days', '7'));
        $rateDays = max(1, (int)SettingsService::get('gdpr_retention_rate_limits_days', '7'));
        $geoDays = max(1, (int)SettingsService::get('gdpr_retention_geo_cache_days', '7'));

        $db = DatabaseManager::getWriteConnection();
        $now = time();
        $logCutoff = date(self::DB_DATETIME_FORMAT, $now - ($logDays * 86400));
        $visitorCutoff = date(self::DB_DATETIME_FORMAT, $now - ($visitorDays * 86400));
        $contactCutoff = date(self::DB_DATETIME_FORMAT, $now - ($contactDays * 86400));
        $cartCutoff = date(self::DB_DATETIME_FORMAT, $now - ($cartDays * 86400));
        $messageCutoff = $now - ($messageDays * 86400);
        $rateCutoff = $now - ($rateDays * 86400);
        $geoCutoff = $now - ($geoDays * 86400);

        $db->prepare("DELETE FROM logs WHERE created_at < ?")->execute([$logCutoff]);
        $db->prepare("DELETE FROM visitor_logs WHERE created_at < ?")->execute([$visitorCutoff]);
        $db->prepare("DELETE FROM visitors WHERE last_activity < ?")->execute([$visitorCutoff]);
        $db->prepare("DELETE FROM contact_messages WHERE created_at < ?")->execute([$contactCutoff]);
        $db->prepare("DELETE FROM cart_items WHERE updated_at < ?")->execute([$cartCutoff]);
        $db->prepare("DELETE FROM cart_sessions WHERE updated_at < ?")->execute([$cartCutoff]);
        $db->prepare("DELETE FROM user_messages WHERE created_at < ?")->execute([$messageCutoff]);
        $db->prepare("DELETE FROM user_login_tokens WHERE created_at < ? OR expires_at < ?")->execute([$messageCutoff, $now]);
        $db->prepare("DELETE FROM contact_rate_limits WHERE last_time < ?")->execute([$rateCutoff]);
        $db->prepare("DELETE FROM login_rate_limits WHERE last_time < ?")->execute([$rateCutoff]);
        $db->prepare("DELETE FROM dmca_requests WHERE created_at < ?")->execute([$contactCutoff]);
        $db->prepare("DELETE FROM dmca_rate_limits WHERE last_time < ?")->execute([$rateCutoff]);
        $db->prepare("DELETE FROM geo_cache WHERE cached_at < ?")->execute([$geoCutoff]);

        SettingsService::set('gdpr_cleanup_last', (string)$now);
        LogService::add('gdpr', 'GDPR retention cleanup completed', json_encode([
            'logs_days' => $logDays, 'visitors_days' => $visitorDays,
            'messages_days' => $messageDays, 'contact_days' => $contactDays,
            'carts_days' => $cartDays, 'rate_limits_days' => $rateDays,
            'geo_cache_days' => $geoDays
        ]));
    }

    public static function resolveProfilePhotoPath(string $photoPath): ?string
    {
        if ($photoPath === '') {
            return null;
        }
        $fileName = basename($photoPath);
        if ($fileName === '') {
            return null;
        }
        $resolved = null;
        $path = dirname(__DIR__, 2) . '/data/profile_photos/' . $fileName;
        if (is_file($path)) {
            $resolved = $path;
        } elseif (strpos($photoPath, '/assets/uploads/') === 0) {
            $publicPath = dirname(__DIR__, 2) . '/public' . $photoPath;
            if (is_file($publicPath)) {
                $resolved = $publicPath;
            }
        }
        return $resolved;
    }

    public static function resolveChatImagePath(string $imagePath): ?string
    {
        if ($imagePath === '') {
            return null;
        }
        $fileName = basename($imagePath);
        if ($fileName === '') {
            return null;
        }
        $path = dirname(__DIR__, 2) . '/data/chat_images/' . $fileName;
        return is_file($path) ? $path : null;
    }

    public static function appendMediaFileEntry(array &$files, string $type, string $source, string $filename, string $fullPath, string $zipPath): void
    {
        $files[] = [
            'type' => $type,
            'source' => $source,
            'filename' => $filename,
            'full_path' => $fullPath,
            'zip_path' => $zipPath,
        ];
    }

    public static function collectProfilePhotoMediaFile(int $userId, array &$files, array &$seenPaths): void
    {
        $user = UserService::getById($userId);
        if (!$user) {
            return;
        }
        $profilePhoto = trim((string)($user['profile_photo'] ?? ''));
        $fullPath = self::resolveProfilePhotoPath($profilePhoto);
        if ($fullPath !== null && !isset($seenPaths[$fullPath])) {
            $seenPaths[$fullPath] = true;
            $fileName = basename($fullPath);
            self::appendMediaFileEntry($files, 'profile_photo', $profilePhoto, $fileName, $fullPath, 'media/profile-photos/' . $fileName);
        }
    }

    public static function buildUniqueChatZipPath(string $fileName, array &$usedZipPaths): string
    {
        $zipPath = 'media/chat-images/' . $fileName;
        if (isset($usedZipPaths[$zipPath])) {
            $base = pathinfo($fileName, PATHINFO_FILENAME);
            $ext = pathinfo($fileName, PATHINFO_EXTENSION);
            $suffix = 2;
            do {
                $altName = $base . '-' . $suffix . ($ext !== '' ? '.' . $ext : '');
                $zipPath = 'media/chat-images/' . $altName;
                $suffix++;
            } while (isset($usedZipPaths[$zipPath]));
        }
        $usedZipPaths[$zipPath] = true;
        return $zipPath;
    }

    public static function collectUserMediaFiles(int $userId): array
    {
        $files = [];
        $seenPaths = [];
        $usedZipPaths = [];
        self::collectProfilePhotoMediaFile($userId, $files, $seenPaths);
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare("SELECT DISTINCT image_path FROM user_messages WHERE (user_id = ? OR sender_id = ?) AND image_path IS NOT NULL AND image_path != ''");
        $stmt->execute([$userId, $userId]);
        $imageRows = $stmt->fetchAll(\PDO::FETCH_COLUMN);
        foreach ($imageRows as $rowPath) {
            $imagePath = trim((string)$rowPath);
            if ($imagePath === '') {
                continue;
            }
            $fullPath = self::resolveChatImagePath($imagePath);
            if ($fullPath === null || isset($seenPaths[$fullPath])) {
                continue;
            }
            $seenPaths[$fullPath] = true;
            $fileName = basename($fullPath);
            $zipPath = self::buildUniqueChatZipPath($fileName, $usedZipPaths);
            self::appendMediaFileEntry($files, 'chat_image', $imagePath, basename($zipPath), $fullPath, $zipPath);
        }
        return $files;
    }

    public static function exportUserData(int $userId, ?array $mediaFiles = null): array
    {
        $db = DatabaseManager::getWriteConnection();
        $user = UserService::getById($userId);
        if (!$user) {
            return [];
        }
        $stmt = $db->prepare("SELECT * FROM user_messages WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$userId]);
        $messages = $stmt->fetchAll();
        $stmt = $db->prepare("SELECT * FROM user_login_tokens WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$userId]);
        $tokens = $stmt->fetchAll();
        $stmt = $db->prepare("SELECT * FROM orders WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$userId]);
        $orders = $stmt->fetchAll();
        $orderItems = [];
        $downloads = [];
        if (!empty($orders)) {
            $orderIds = array_column($orders, 'id');
            $placeholders = implode(',', array_fill(0, count($orderIds), '?'));
            $itemStmt = $db->prepare("SELECT * FROM order_items WHERE order_id IN ($placeholders) ORDER BY id ASC");
            $itemStmt->execute($orderIds);
            foreach ($itemStmt->fetchAll() as $item) {
                $orderItems[$item['order_id']][] = $item;
            }
            $downloadStmt = $db->prepare("SELECT dd.*, oi.order_id FROM digital_downloads dd JOIN order_items oi ON oi.id = dd.order_item_id WHERE oi.order_id IN ($placeholders)");
            $downloadStmt->execute($orderIds);
            $downloads = $downloadStmt->fetchAll();
        }
        $email = $user['email'] ?? '';
        $contactMessages = [];
        if ($email !== '') {
            $stmt = $db->prepare("SELECT * FROM contact_messages WHERE email = ? ORDER BY created_at DESC");
            $stmt->execute([$email]);
            $contactMessages = $stmt->fetchAll();
        }
        $mediaFiles = $mediaFiles ?? self::collectUserMediaFiles($userId);
        $mediaManifest = [];
        foreach ($mediaFiles as $file) {
            $mediaManifest[] = [
                'type' => $file['type'] ?? 'file',
                'source' => $file['source'] ?? '',
                'filename' => $file['filename'] ?? '',
                'zip_path' => $file['zip_path'] ?? ''
            ];
        }
        return [
            'generated_at' => gmdate('c'),
            'user' => $user,
            'messages' => $messages,
            'login_tokens' => $tokens,
            'orders' => $orders,
            'order_items' => $orderItems,
            'downloads' => $downloads,
            'contact_messages' => $contactMessages,
            'media_files' => $mediaManifest
        ];
    }

    public static function createExportTempPath(): array
    {
        $tmpBase = tempnam(sys_get_temp_dir(), 'gdpr_export_');
        if ($tmpBase === false) {
            return ['success' => false, 'path' => null, 'error' => 'tmp_unavailable'];
        }
        $zipPath = $tmpBase . '.zip';
        @rename($tmpBase, $zipPath);
        return ['success' => true, 'path' => $zipPath, 'error' => null];
    }

    public static function populateExportArchive(string $zipPath, array $exportData, array $mediaFiles): string
    {
        $error = '';
        $zip = new \ZipArchive();
        if ($zip->open($zipPath, \ZipArchive::OVERWRITE) !== true) {
            return 'zip_open_failed';
        }
        $payload = json_encode($exportData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        if ($payload === false) {
            $zip->close();
            return 'json_failed';
        }
        $zip->addFromString('user-data-export.json', $payload);
        foreach ($mediaFiles as $file) {
            $fullPath = $file['full_path'] ?? '';
            $zipPathInternal = $file['zip_path'] ?? '';
            if ($fullPath !== '' && $zipPathInternal !== '' && is_file($fullPath)) {
                $zip->addFile($fullPath, $zipPathInternal);
            }
        }
        $zip->close();
        return $error;
    }

    public static function createUserExportArchive(int $userId, array $exportData, array $mediaFiles): array
    {
        if (!class_exists('ZipArchive')) {
            return ['success' => false, 'error' => 'zip_unavailable'];
        }
        $tmp = self::createExportTempPath();
        if (empty($tmp['success'])) {
            return ['success' => false, 'error' => (string)($tmp['error'] ?? 'tmp_unavailable')];
        }
        $zipPath = (string)$tmp['path'];
        $error = self::populateExportArchive($zipPath, $exportData, $mediaFiles);
        if ($error !== '') {
            @unlink($zipPath);
            $result = ['success' => false, 'error' => $error];
        } else {
            $fileName = 'user-data-export-' . $userId . '-' . date('Ymd-His') . '.zip';
            $size = is_file($zipPath) ? (int)filesize($zipPath) : 0;
            $result = ['success' => true, 'path' => $zipPath, 'name' => $fileName, 'mime' => 'application/zip', 'size' => $size];
        }
        return $result;
    }

    public static function deleteUserData(int $userId): bool
    {
        $db = DatabaseManager::getWriteConnection();
        $user = UserService::getById($userId);
        if (!$user) {
            return false;
        }
        $photo = $user['profile_photo'] ?? '';
        if ($photo !== '' && strpos($photo, self::PROFILE_PHOTO_ROUTE_PREFIX) === 0) {
            $path = dirname(__DIR__, 2) . '/data/profile_photos/' . basename($photo);
            if (is_file($path)) {
                @unlink($path);
            }
        }
        $email = $user['email'] ?? '';
        if ($email !== '') {
            $stmt = $db->prepare("DELETE FROM contact_messages WHERE email = ?");
            $stmt->execute([$email]);
        }
        $stmt = $db->prepare("DELETE FROM site_users WHERE id = ?");
        $stmt->execute([$userId]);
        return $stmt->rowCount() > 0;
    }
}
