<?php

declare(strict_types=1);

namespace NewSite\Forum;

/**
 * Shared forum utility methods.
 *
 * Eliminates duplicate function definitions across forum templates
 * (forum.php, forum-post.php, forum-posts-api.php, admin/forum.php).
 *
 * Security: all output helpers escape via htmlspecialchars(); query
 * builders sanitize parameters before encoding.
 */
class ForumHelper
{
    /**
     * Build a query string from non-empty parameters.
     *
     * Filters out null, empty-string, and zero values, then returns
     * a URL-encoded query string prefixed with "?" or an empty string.
     *
     * @param array<string, mixed> $params Key-value pairs
     */
    public static function buildQuery(array $params): string
    {
        $params = array_filter(
            $params,
            static fn (mixed $value): bool => $value !== null && $value !== '' && $value !== 0
        );

        return $params === [] ? '' : '?' . http_build_query($params);
    }

    /**
     * Convert a flat list of subcategories into a nested tree.
     *
     * Each node receives a "children" key containing its nested
     * subcategories.  Depth tracking assumes a maximum of two levels.
     *
     * @param list<array<string, mixed>> $subcategories Flat rows from DB
     * @param int|null $parentId                        Current parent (null = root)
     * @return list<array<string, mixed>> Nested tree
     */
    public static function buildSubcategoryTree(array $subcategories, ?int $parentId = null): array
    {
        $branch = [];
        foreach ($subcategories as $sub) {
            $subParentId = isset($sub['parent_id']) && $sub['parent_id'] !== null
                ? (int) $sub['parent_id']
                : null;

            if ($subParentId === $parentId) {
                $sub['children'] = self::buildSubcategoryTree($subcategories, (int) $sub['id']);
                if ($parentId !== null) {
                    $sub['depth'] = 1 + ($sub['depth'] ?? 0);
                } else {
                    $sub['depth'] = $sub['depth'] ?? 0;
                }
                $branch[] = $sub;
            }
        }

        return $branch;
    }

    /**
     * Convert an admin-file path to a public-file path.
     *
     * Replaces the "/admin-file/" segment with "/site-file/" so images
     * stored through the admin uploader are accessible on public pages.
     */
    public static function toPublicFilePath(string $path): string
    {
        if ($path === '') {
            return $path;
        }

        return str_replace('/admin-file/', '/site-file/', $path);
    }
}
