<?php

declare(strict_types=1);

namespace NewSite\Email;

use NewSite\Database\DatabaseManager;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;
use NewSite\User\UserService;
use NewSite\Email\SmtpMailer;
use NewSite\Database\DbHelper;

/**
 * Transactional email service for purchase receipts and product update notifications.
 *
 * Security: All database queries use prepared statements; user-supplied data is
 * never interpolated into SQL; email bodies are plain-text only (no HTML injection).
 */
final class TransactionalEmail
{
    public static function getProductNameForUpdate(\PDO $db, int $productId, ?int $variantId): ?string
    {
        $stmt = $db->prepare("SELECT name FROM products WHERE id = ?");
        $stmt->execute([$productId]);
        $productName = trim((string)$stmt->fetchColumn());
        if ($productName === '') {
            return null;
        }
        $variantLabel = '';
        if ($variantId !== null && $variantId > 0) {
            $variantStmt = $db->prepare("SELECT label FROM product_variants WHERE id = ? AND product_id = ? LIMIT 1");
            $variantStmt->execute([$variantId, $productId]);
            $variantLabel = trim((string)$variantStmt->fetchColumn());
        }
        return $variantLabel !== '' ? ($productName . ' — ' . $variantLabel) : $productName;
    }

    public static function getUpdateRecipients(\PDO $db, int $productId, ?int $variantId): array
    {
        $query = "SELECT DISTINCT su.id, su.email, su.display_name, su.nickname FROM digital_downloads dd JOIN order_items oi ON oi.id = dd.order_item_id JOIN site_users su ON su.id = dd.user_id WHERE (dd.product_id = ? OR oi.product_id = ?) AND su.email IS NOT NULL AND su.email != ''";
        $params = [$productId, $productId];
        if ($variantId !== null && $variantId > 0) {
            $query .= " AND oi.variant_id = ?";
            $params[] = $variantId;
        }
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    private static function getSmtpConfig(): array
    {
        return [
            'host' => SettingsService::get('smtp_host', '') ?? '',
            'port' => SettingsService::get('smtp_port', '587') ?? '587',
            'user' => SettingsService::get('smtp_user', '') ?? '',
            'pass' => SettingsService::get('smtp_pass', '') ?? '',
            'secure' => SettingsService::get('smtp_secure', 'tls') ?? 'tls',
        ];
    }

    private static function getBaseUrl(): string
    {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $scheme . '://' . $host;
    }

    public static function sendPurchaseReceipt(int $orderId): bool
    {
        $sent = false;
        $db = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("SELECT o.*, su.email, su.display_name, su.nickname FROM orders o JOIN site_users su ON su.id = o.user_id WHERE o.id = ? LIMIT 1");
        $stmt->execute([$orderId]);
        $order = $stmt->fetch();
        if ($order && empty($order['order_email_sent'])) {
            $to = trim((string)($order['email'] ?? ''));
            if ($to !== '') {
                $subject = SettingsService::get('purchase_email_subject', 'Thanks for your purchase') ?? 'Thanks for your purchase';
                $body = SettingsService::get('purchase_email_body', "Hi {name},\n\nThanks for your purchase!\nYou can view your purchase anytime by logging into your account: {account_url}\n\nOrder: {order_number}\nTotal: {total}\n\nThank you!\n") ?? '';
                $name = UserService::resolveDisplayName((array)$order, 'there');
                $symbol = CurrencyService::getSymbolFor($order['currency'] ?? CurrencyService::getCurrent());
                $total = $symbol . number_format((float)($order['total_amount'] ?? 0), 2);
                $replacements = [
                    '{name}' => $name,
                    '{order_number}' => (string)($order['order_number'] ?? ''),
                    '{total}' => $total,
                    '{currency}' => (string)($order['currency'] ?? ''),
                    '{account_url}' => self::getBaseUrl() . '/account?tab=purchases'
                ];
                $finalBody = str_replace(array_keys($replacements), array_values($replacements), $body);
                $sent = SmtpMailer::send($to, $subject, $finalBody, '', '', self::getSmtpConfig());
                if ($sent) {
                    $now = DbHelper::nowString();
                    $stmt = $db->prepare("UPDATE orders SET order_email_sent = 1, updated_at = ? WHERE id = ?");
                    $stmt->execute([$now, $orderId]);
                }
            }
        }
        return $sent;
    }

    public static function sendProductUpdate(int $productId, string $newVersion, string $changelog, ?int $variantId = null): array
    {
        $db = DatabaseManager::getWriteConnection();
        $productNameForEmail = self::getProductNameForUpdate($db, $productId, $variantId);
        if ($productNameForEmail === null) {
            return ['sent' => 0, 'failed' => 0];
        }
        $buyers = self::getUpdateRecipients($db, $productId, $variantId);
        if (empty($buyers)) {
            return ['sent' => 0, 'failed' => 0];
        }
        $subject = SettingsService::get('product_update_email_subject', 'New update available: {product_name}') ?? '';
        $body = SettingsService::get('product_update_email_body', "Hi {name},\n\nGreat news! A new update is available for {product_name}.\n\nVersion: {version}\n\n{changelog}\n\nLog in to your account to download the latest version:\n{account_url}\n\nThank you!") ?? '';
        $smtp = self::getSmtpConfig();
        $sent = 0;
        $failed = 0;
        foreach ($buyers as $buyer) {
            $name = UserService::resolveDisplayName((array)$buyer, 'there');
            $changelogFormatted = $changelog !== '' ? "What's new:\n" . $changelog : '';
            $replacements = [
                '{name}' => $name,
                '{product_name}' => $productNameForEmail,
                '{version}' => $newVersion,
                '{changelog}' => $changelogFormatted,
                '{account_url}' => self::getBaseUrl() . '/account?tab=purchases'
            ];
            $finalSubject = str_replace(array_keys($replacements), array_values($replacements), $subject);
            $finalBody = str_replace(array_keys($replacements), array_values($replacements), $body);
            $emailSent = SmtpMailer::send((string)$buyer['email'], $finalSubject, $finalBody, '', '', $smtp);
            $sent += $emailSent ? 1 : 0;
            $failed += $emailSent ? 0 : 1;
        }
        return ['sent' => $sent, 'failed' => $failed];
    }
}
