<?php

declare(strict_types=1);

namespace NewSite\Email;

use NewSite\Logging\LogService;

/**
 * Low-level SMTP mailer.
 *
 * Security: All socket I/O is length-limited; TLS 1.2+ enforced;
 * credentials are never logged; attachment paths are validated with is_file().
 */
final class SmtpMailer
{
    private const SMTP_DOUBLE_EOL = "\r\n\r\n";

    public static function sendCommand($fp, string $cmd): string
    {
        fwrite($fp, $cmd . "\r\n");
        return self::readResponse($fp);
    }

    public static function readResponse($fp): string
    {
        $data = '';
        while (!feof($fp)) {
            $line = fgets($fp, 515);
            if ($line === false) {
                break;
            }
            $data .= $line;
            if (preg_match('/^\d{3} /', $line)) {
                break;
            }
        }
        return $data;
    }

    public static function responseHasCode(string $response, string $code): bool
    {
        return preg_match('/^' . preg_quote($code, '/') . '/', $response) === 1;
    }

    private static function getEhloHost(): string
    {
        return (string)($_SERVER['HTTP_HOST'] ?? 'localhost');
    }

    public static function initializeConnection($fp, string $secure): bool
    {
        $ok = true;
        $host = self::getEhloHost();
        if ($secure === 'tls') {
            $ehlo = self::sendCommand($fp, 'EHLO ' . $host);
            $startTls = self::sendCommand($fp, 'STARTTLS');
            $tlsMethod = defined('STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT')
                ? (STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT | STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT)
                : STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT;
            $cryptoOk = @stream_socket_enable_crypto($fp, true, $tlsMethod) === true;
            $ok = self::responseHasCode($ehlo, '250')
                && self::responseHasCode($startTls, '220')
                && $cryptoOk;
        }
        if ($ok) {
            $ehloAfter = self::sendCommand($fp, 'EHLO ' . $host);
            $ok = self::responseHasCode($ehloAfter, '250');
        }
        return $ok;
    }

    public static function authenticateLogin($fp, string $user, string $pass): bool
    {
        $auth = self::sendCommand($fp, 'AUTH LOGIN');
        $userResp = self::sendCommand($fp, base64_encode($user));
        $passResp = self::sendCommand($fp, base64_encode($pass));
        return self::responseHasCode($auth, '334')
            && self::responseHasCode($userResp, '334')
            && self::responseHasCode($passResp, '235');
    }

    public static function openMessageData($fp, string $user, string $to): bool
    {
        $mailFrom = self::sendCommand($fp, 'MAIL FROM: <' . $user . '>');
        $rcptTo = self::sendCommand($fp, 'RCPT TO: <' . $to . '>');
        $data = self::sendCommand($fp, 'DATA');
        return self::responseHasCode($mailFrom, '250')
            && self::responseHasCode($rcptTo, '250')
            && self::responseHasCode($data, '354');
    }

    public static function buildMessagePayload(string $to, string $subject, string $body, string $user, string $replyTo, string $replyName, $attachment): string
    {
        $headers = [];
        $headers[] = 'From: ' . $user;
        $headers[] = 'To: ' . $to;
        if ($replyTo !== '') {
            $headers[] = 'Reply-To: ' . ($replyName !== '' ? ($replyName . ' <' . $replyTo . '>') : $replyTo);
        }
        $headers[] = 'MIME-Version: 1.0';
        $headers[] = 'Subject: ' . $subject;

        $messageBody = '';
        $hasAttachment = is_array($attachment) && !empty($attachment['path']) && is_file((string)$attachment['path']);

        if ($hasAttachment) {
            $boundary = '=_Part_' . bin2hex(random_bytes(8));
            $headers[] = 'Content-Type: multipart/mixed; boundary="' . $boundary . '"';
            $messageBody .= '--' . $boundary . "\r\n";
            $messageBody .= 'Content-Type: text/plain; charset=UTF-8' . self::SMTP_DOUBLE_EOL;
            $messageBody .= $body . self::SMTP_DOUBLE_EOL;
            $attachmentPath = (string)$attachment['path'];
            $fileName = (string)($attachment['name'] ?? basename($attachmentPath));
            $mime = (string)($attachment['mime'] ?? 'application/octet-stream');
            $fileData = chunk_split(base64_encode((string)file_get_contents($attachmentPath)));
            $messageBody .= '--' . $boundary . "\r\n";
            $messageBody .= 'Content-Type: ' . $mime . '; name="' . $fileName . '"' . "\r\n";
            $messageBody .= "Content-Transfer-Encoding: base64\r\n";
            $messageBody .= 'Content-Disposition: attachment; filename="' . $fileName . '"' . self::SMTP_DOUBLE_EOL;
            $messageBody .= $fileData . "\r\n";
            $messageBody .= '--' . $boundary . "--\r\n";
        } else {
            $headers[] = 'Content-Type: text/plain; charset=UTF-8';
            $messageBody = $body . "\r\n";
        }

        return implode("\r\n", $headers) . self::SMTP_DOUBLE_EOL . $messageBody . "\r\n";
    }

    public static function send(string $to, string $subject, string $body, string $replyTo = '', string $replyName = '', array $smtp = [], $attachment = null): bool
    {
        $host = $smtp['host'] ?? '';
        $port = (int)($smtp['port'] ?? 587);
        $user = $smtp['user'] ?? '';
        $pass = $smtp['pass'] ?? '';
        $secure = $smtp['secure'] ?? 'tls';
        if ($host === '' || $user === '' || $pass === '') {
            return false;
        }
        $transport = ($secure === 'ssl') ? 'ssl://' : '';
        $fp = @fsockopen($transport . $host, $port, $errno, $errstr, 10);
        if (!$fp) {
            LogService::add('error', 'SMTP connection failed', json_encode(['error' => $errstr, 'code' => $errno]));
            return false;
        }
        self::readResponse($fp);
        $ok = self::initializeConnection($fp, (string)$secure)
            && self::authenticateLogin($fp, (string)$user, (string)$pass)
            && self::openMessageData($fp, (string)$user, (string)$to);
        if ($ok) {
            $data = self::buildMessagePayload((string)$to, (string)$subject, (string)$body, (string)$user, (string)$replyTo, (string)$replyName, $attachment);
            fwrite($fp, $data . "\r\n.\r\n");
            $resp = self::readResponse($fp);
            $ok = self::responseHasCode($resp, '250');
        }
        self::sendCommand($fp, 'QUIT');
        fclose($fp);
        LogService::add('email', $ok ? "Email sent to: $to" : "Email FAILED to: $to", "Subject: $subject");
        return $ok;
    }
}
