<?php

declare(strict_types=1);

namespace NewSite\Database;

use NewSite\Minigames\MinigameCatalog;
use NewSite\Settings\SettingsService;
use NewSite\Util\SlugGenerator;
use PDO;
use PDOException;
use Throwable;

/**
 * Database schema manager — creates tables, runs versioned migrations, seeds defaults.
 *
 * Extracted from the legacy initDatabase() function for maintainability.
 * All operations are idempotent: safe to re-run on an existing database.
 *
 * Security: No user input influences SQL. All queries use IF NOT EXISTS
 * or try/catch for column additions. Prepared statements for data seeds.
 */
class SchemaManager
{
    /** Exposed as public so admin info pages can read the target version. */
    public const SCHEMA_TARGET = 39;
    private const TYPE_INT_ZERO = 'INTEGER DEFAULT 0';

    private PDO $db;

    public function __construct(PDO $db)
    {
        $this->db = $db;
    }

    /**
     * Main entry point — replaces the legacy initDatabase() function.
     *
     * Creates the settings table first (needed for schema-version check),
     * then either returns early (schema up-to-date) or runs full setup.
     */
    public function run(): void
    {
        // Settings table must exist before we can read db_schema_version
        $this->db->exec("CREATE TABLE IF NOT EXISTS settings (
            id SERIAL PRIMARY KEY,
            setting_key TEXT UNIQUE NOT NULL,
            setting_value TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $schemaVersion = (int) SettingsService::get('db_schema_version', '0');

        if ($schemaVersion >= self::SCHEMA_TARGET) {
            $this->updateUserActivity();
            return;
        }

        $this->createCoreTables();
        $this->createUserTables();
        $this->createForumTables();
        $this->createProductTables();
        $this->createOrderTables();
        $this->createSiteInfrastructureTables();
        $this->seedDefaults();
        $this->createPerformanceIndexes();
        $this->runMigrations($schemaVersion);

        SettingsService::set('db_schema_version', (string) self::SCHEMA_TARGET);
        $this->updateUserActivity();
    }

    // ── Base table creation ────────────────────────────────────────

    /**
     * Core CMS tables: users, pages, menus, collections, menu_items,
     * theme_settings, sections, files, translations.
     */
    private function createCoreTables(): void
    {
        $this->db->exec("CREATE TABLE IF NOT EXISTS users (
            id SERIAL PRIMARY KEY,
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL,
            email TEXT,
            role TEXT DEFAULT 'admin',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        // Optimistic activity update — may fail if site_users/last_activity
        // column does not exist yet; safe to ignore during initial migration
        $this->updateUserActivity();

        $this->db->exec("CREATE TABLE IF NOT EXISTS pages (
            id SERIAL PRIMARY KEY,
            title TEXT NOT NULL,
            slug TEXT UNIQUE NOT NULL,
            content TEXT,
            meta_title TEXT,
            meta_description TEXT,
            is_published INTEGER DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->addColumnSafe('pages', 'use_layout_override', self::TYPE_INT_ZERO);
        $this->addColumnSafe('pages', 'content_max_width', 'INTEGER');
        $this->addColumnSafe('pages', 'section_padding', 'INTEGER');
        $this->addColumnSafe('pages', 'section_height', 'INTEGER');

        $this->db->exec("CREATE TABLE IF NOT EXISTS menus (
            id SERIAL PRIMARY KEY,
            name TEXT NOT NULL,
            location TEXT NOT NULL,
            sort_order INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->addColumnSafe('menus', 'sort_order', self::TYPE_INT_ZERO);

        $this->db->exec("CREATE TABLE IF NOT EXISTS collections (
            id SERIAL PRIMARY KEY,
            title TEXT NOT NULL,
            collection_slug TEXT,
            description TEXT,
            meta_title TEXT,
            meta_description TEXT,
            image_url TEXT,
            target_page_id INTEGER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->addColumnSafe('collections', 'target_page_id', 'INTEGER');
        $this->addColumnSafe('collections', 'meta_title', 'TEXT');
        $this->addColumnSafe('collections', 'meta_description', 'TEXT');
        $this->addColumnSafe('collections', 'collection_slug', 'TEXT');

        $this->db->exec("CREATE TABLE IF NOT EXISTS menu_items (
            id SERIAL PRIMARY KEY,
            menu_id INTEGER NOT NULL,
            parent_id INTEGER DEFAULT NULL,
            title TEXT NOT NULL,
            url TEXT,
            page_id INTEGER DEFAULT NULL,
            target TEXT DEFAULT '_self',
            icon TEXT DEFAULT NULL,
            sort_order INTEGER DEFAULT 0,
            FOREIGN KEY (menu_id) REFERENCES menus(id) ON DELETE CASCADE,
            FOREIGN KEY (parent_id) REFERENCES menu_items(id) ON DELETE CASCADE,
            FOREIGN KEY (page_id) REFERENCES pages(id) ON DELETE SET NULL
        )");
        $this->addColumnSafe('menu_items', 'icon', 'TEXT DEFAULT NULL');

        $this->db->exec("CREATE TABLE IF NOT EXISTS theme_settings (
            id SERIAL PRIMARY KEY,
            setting_key TEXT UNIQUE NOT NULL,
            setting_value TEXT,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS sections (
            id SERIAL PRIMARY KEY,
            page_id INTEGER NOT NULL,
            section_type TEXT NOT NULL,
            settings TEXT,
            sort_order INTEGER DEFAULT 0,
            is_active INTEGER DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (page_id) REFERENCES pages(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS files (
            id SERIAL PRIMARY KEY,
            filename TEXT NOT NULL,
            original_name TEXT NOT NULL,
            file_path TEXT NOT NULL,
            file_type TEXT,
            file_size INTEGER,
            uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS translations (
            id SERIAL PRIMARY KEY,
            language_code TEXT NOT NULL,
            translation_key TEXT NOT NULL,
            translation_value TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS logs (
            id SERIAL PRIMARY KEY,
            log_type TEXT NOT NULL,
            message TEXT NOT NULL,
            details TEXT,
            ip_address TEXT,
            user_agent TEXT,
            user_id INTEGER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
    }

    /**
     * User-related tables: site_users, tokens, messages, rate limits,
     * privacy settings, friendships, friend requests, user blocks.
     */
    private function createUserTables(): void
    {
        $this->db->exec("CREATE TABLE IF NOT EXISTS site_users (
            id SERIAL PRIMARY KEY,
            email TEXT UNIQUE NOT NULL,
            display_name TEXT,
            description TEXT,
            website TEXT,
            company TEXT,
            phone TEXT,
            gender TEXT,
            gender_custom TEXT,
            age INTEGER,
            is_active INTEGER DEFAULT 1,
            created_at INTEGER,
            last_login_at INTEGER,
            terms_accepted_at INTEGER,
            terms_accepted_ip TEXT,
            privacy_accepted_at INTEGER,
            privacy_accepted_ip TEXT,
            created_ip TEXT,
            last_login_ip TEXT
        )");

        // Column additions for existing databases
        $siteUserColumns = [
            ['address', 'TEXT'], ['country', 'TEXT'], ['postal_code', 'TEXT'],
            ['profile_photo', 'TEXT'], ['nickname', 'TEXT'],
            ['description', 'TEXT'], ['website', 'TEXT'], ['company', 'TEXT'],
            ['phone', 'TEXT'], ['gender', 'TEXT'], ['gender_custom', 'TEXT'],
            ['age', 'INTEGER'], ['created_ip', 'TEXT'], ['last_login_ip', 'TEXT'],
        ];
        foreach ($siteUserColumns as [$col, $type]) {
            $this->addColumnSafe('site_users', $col, $type);
        }

        $this->db->exec("CREATE TABLE IF NOT EXISTS user_login_tokens (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL,
            token_hash TEXT NOT NULL,
            expires_at INTEGER NOT NULL,
            used_at INTEGER,
            created_at INTEGER NOT NULL,
            ip_address TEXT,
            user_agent TEXT,
            FOREIGN KEY (user_id) REFERENCES site_users(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS user_messages (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL,
            subject TEXT,
            body TEXT,
            sender_type TEXT DEFAULT 'admin',
            sender_label TEXT,
            reply_to_id INTEGER,
            message_type TEXT DEFAULT 'notice',
            is_read INTEGER DEFAULT 0,
            created_at INTEGER NOT NULL,
            FOREIGN KEY (user_id) REFERENCES site_users(id) ON DELETE CASCADE
        )");
        $this->addColumnSafe('user_messages', 'sender_type', "TEXT DEFAULT 'admin'");
        $this->addColumnSafe('user_messages', 'sender_label', 'TEXT');
        $this->addColumnSafe('user_messages', 'reply_to_id', 'INTEGER');
        $this->addColumnSafe('user_messages', 'message_type', "TEXT DEFAULT 'notice'");
        $this->addColumnSafe('user_messages', 'image_path', 'TEXT');
        $this->addColumnSafe('user_messages', 'admin_read', 'INTEGER DEFAULT 1');
        $this->execSafe("UPDATE user_messages SET message_type = 'ticket' WHERE message_type IS NULL AND subject = 'Ticket'");

        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_user_messages_user ON user_messages(user_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_user_messages_user_read ON user_messages(user_id, is_read)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_user_messages_created ON user_messages(user_id, created_at DESC)");

        $this->db->exec("CREATE TABLE IF NOT EXISTS welcome_messages (
            id SERIAL PRIMARY KEY,
            subject TEXT,
            body TEXT,
            is_active INTEGER DEFAULT 1,
            created_at INTEGER NOT NULL
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS login_rate_limits (
            ip TEXT PRIMARY KEY,
            last_time INTEGER,
            count INTEGER DEFAULT 0
        )");
        $this->db->exec("CREATE TABLE IF NOT EXISTS gdpr_export_rate_limits (
            user_id INTEGER PRIMARY KEY,
            last_time INTEGER
        )");
        $this->addColumnSafe('login_rate_limits', 'count', self::TYPE_INT_ZERO);

        // ── Schema v8: Friends & privacy ──
        $this->db->exec("CREATE TABLE IF NOT EXISTS user_privacy_settings (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
            field_name TEXT NOT NULL,
            visibility TEXT NOT NULL DEFAULT 'everyone',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, field_name)
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS friendships (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
            friend_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, friend_id)
        )");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_friendships_user ON friendships(user_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_friendships_friend ON friendships(friend_id)");

        $this->db->exec("CREATE TABLE IF NOT EXISTS friend_requests (
            id SERIAL PRIMARY KEY,
            from_user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
            to_user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
            status TEXT NOT NULL DEFAULT 'pending',
            message TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(from_user_id, to_user_id)
        )");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_friend_requests_to ON friend_requests(to_user_id, status)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_friend_requests_from ON friend_requests(from_user_id, status)");

        $this->db->exec("CREATE TABLE IF NOT EXISTS user_blocks (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
            blocked_user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, blocked_user_id)
        )");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_user_blocks_user ON user_blocks(user_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_user_blocks_blocked ON user_blocks(blocked_user_id)");

        // Additional site_users columns from schema v8/v11
        $this->addColumnSafe('site_users', 'nickname', 'TEXT UNIQUE');
        $this->addColumnSafe('site_users', 'profile_photo', 'TEXT');
        $this->addColumnSafe('site_users', 'is_searchable', 'INTEGER DEFAULT 1');
        $this->addColumnSafe('site_users', 'status', "TEXT DEFAULT 'online'");
        $this->addColumnSafe('site_users', 'last_activity', 'INTEGER');

        // Schema v11: messenger — add sender_id to user_messages
        $this->addColumnSafe('user_messages', 'sender_id', 'INTEGER REFERENCES site_users(id) ON DELETE SET NULL');
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_user_messages_chat ON user_messages(user_id, sender_id, message_type, created_at DESC)");
    }

    /**
     * Forum tables: categories, subcategories, posts, comments, images.
     */
    private function createForumTables(): void
    {
        $this->db->exec("CREATE TABLE IF NOT EXISTS forum_categories (
            id SERIAL PRIMARY KEY,
            name TEXT NOT NULL,
            slug TEXT,
            is_active INTEGER DEFAULT 1,
            sort_order INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->db->exec("CREATE TABLE IF NOT EXISTS forum_subcategories (
            id SERIAL PRIMARY KEY,
            category_id INTEGER NOT NULL REFERENCES forum_categories(id) ON DELETE CASCADE,
            name TEXT NOT NULL,
            slug TEXT,
            description TEXT,
            is_active INTEGER DEFAULT 1,
            sort_order INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->db->exec("CREATE TABLE IF NOT EXISTS forum_posts (
            id SERIAL PRIMARY KEY,
            subcategory_id INTEGER NOT NULL REFERENCES forum_subcategories(id) ON DELETE CASCADE,
            user_id INTEGER REFERENCES site_users(id) ON DELETE SET NULL,
            title TEXT NOT NULL,
            body TEXT,
            views INTEGER DEFAULT 0,
            status TEXT DEFAULT 'pending',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->addColumnSafe('forum_posts', 'views', self::TYPE_INT_ZERO);

        $this->db->exec("CREATE TABLE IF NOT EXISTS forum_comments (
            id SERIAL PRIMARY KEY,
            post_id INTEGER NOT NULL REFERENCES forum_posts(id) ON DELETE CASCADE,
            user_id INTEGER REFERENCES site_users(id) ON DELETE SET NULL,
            body TEXT,
            status TEXT DEFAULT 'approved',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->db->exec("CREATE TABLE IF NOT EXISTS forum_post_images (
            id SERIAL PRIMARY KEY,
            post_id INTEGER NOT NULL REFERENCES forum_posts(id) ON DELETE CASCADE,
            image_path TEXT NOT NULL,
            sort_order INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->db->exec("CREATE TABLE IF NOT EXISTS forum_comment_images (
            id SERIAL PRIMARY KEY,
            comment_id INTEGER NOT NULL REFERENCES forum_comments(id) ON DELETE CASCADE,
            image_path TEXT NOT NULL,
            sort_order INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_categories_sort ON forum_categories(sort_order, name)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_subcategories_cat ON forum_subcategories(category_id, sort_order)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_posts_subcat ON forum_posts(subcategory_id, created_at DESC)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_posts_status ON forum_posts(status)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_comments_post ON forum_comments(post_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_post_images_post ON forum_post_images(post_id, sort_order)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_comment_images_comment ON forum_comment_images(comment_id, sort_order)");

        // Schema v18: nested subcategories
        $this->addColumnSafe('forum_subcategories', 'parent_id', 'INTEGER REFERENCES forum_subcategories(id) ON DELETE CASCADE');
        $this->addColumnSafe('forum_subcategories', 'depth', self::TYPE_INT_ZERO);
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_subcategories_parent_id ON forum_subcategories(parent_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_forum_subcategories_category_parent ON forum_subcategories(category_id, parent_id)");
    }

    /**
     * Product catalog: products, likes, versions, attributes, media, blocks, variants.
     */
    private function createProductTables(): void
    {
        $this->db->exec("CREATE TABLE IF NOT EXISTS products (
            id SERIAL PRIMARY KEY,
            name TEXT NOT NULL,
            product_slug TEXT,
            description TEXT,
            price TEXT,
            media_url TEXT,
            action_type TEXT DEFAULT 'cart',
            download_url TEXT,
            external_link_label TEXT,
            page_id INTEGER,
            collection_id INTEGER,
            stock INTEGER,
            quantity_max INTEGER,
            is_featured INTEGER DEFAULT 0,
            sales_count INTEGER DEFAULT 0,
            is_active INTEGER DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $productColumns = [
            ['collection_id', 'INTEGER'], ['stock', 'INTEGER'], ['quantity_max', 'INTEGER'],
            ['is_featured', self::TYPE_INT_ZERO], ['sales_count', self::TYPE_INT_ZERO],
            ['action_type', "TEXT DEFAULT 'cart'"], ['download_url', 'TEXT'],
            ['current_version', "TEXT DEFAULT '1.0'"], ['version_changelog', 'TEXT'],
            ['likes_count', self::TYPE_INT_ZERO], ['dislikes_count', self::TYPE_INT_ZERO],
        ];
        foreach ($productColumns as [$col, $type]) {
            $this->addColumnSafe('products', $col, $type);
        }

        $this->db->exec("CREATE TABLE IF NOT EXISTS product_likes (
            id SERIAL PRIMARY KEY,
            product_id INTEGER NOT NULL,
            user_id INTEGER,
            session_id TEXT,
            ip_address TEXT,
            is_liked BOOLEAN NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
            FOREIGN KEY (user_id) REFERENCES site_users(id) ON DELETE CASCADE
        )");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_product_likes_product ON product_likes(product_id)");
        $this->db->exec("CREATE UNIQUE INDEX IF NOT EXISTS idx_product_likes_user ON product_likes(product_id, user_id) WHERE user_id IS NOT NULL");
        $this->db->exec("CREATE UNIQUE INDEX IF NOT EXISTS idx_product_likes_session ON product_likes(product_id, session_id) WHERE session_id IS NOT NULL");

        $this->db->exec("CREATE TABLE IF NOT EXISTS product_version_history (
            id SERIAL PRIMARY KEY,
            product_id INTEGER NOT NULL,
            version TEXT NOT NULL,
            changelog TEXT,
            file_url TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS product_attributes (
            id SERIAL PRIMARY KEY,
            product_id INTEGER NOT NULL,
            name TEXT NOT NULL,
            value TEXT NOT NULL,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS product_media (
            id SERIAL PRIMARY KEY,
            product_id INTEGER NOT NULL,
            media_url TEXT NOT NULL,
            sort_order INTEGER DEFAULT 0,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS product_blocks (
            id SERIAL PRIMARY KEY,
            product_id INTEGER NOT NULL,
            title TEXT,
            content TEXT,
            sort_order INTEGER DEFAULT 0,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS product_variants (
            id SERIAL PRIMARY KEY,
            product_id INTEGER NOT NULL,
            label TEXT NOT NULL,
            price TEXT,
            is_default INTEGER DEFAULT 0,
            stock INTEGER,
            action_type TEXT DEFAULT 'cart',
            download_url TEXT,
            external_link_label TEXT,
            sort_order INTEGER DEFAULT 0,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
        )");

        $variantColumns = [
            ['is_default', self::TYPE_INT_ZERO], ['stock', 'INTEGER'],
            ['action_type', "TEXT DEFAULT 'cart'"], ['download_url', 'TEXT'],
            ['external_link_label', 'TEXT'],
            ['current_version', "TEXT DEFAULT '1.0'"], ['version_changelog', 'TEXT'],
        ];
        foreach ($variantColumns as [$col, $type]) {
            $this->addColumnSafe('product_variants', $col, $type);
        }
    }

    /**
     * Order processing: carts, orders, order items, digital downloads.
     */
    private function createOrderTables(): void
    {
        $this->db->exec("CREATE TABLE IF NOT EXISTS cart_sessions (
            id SERIAL PRIMARY KEY,
            session_id TEXT UNIQUE NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS cart_items (
            id SERIAL PRIMARY KEY,
            cart_id INTEGER NOT NULL,
            product_id INTEGER NOT NULL,
            variant_id INTEGER DEFAULT NULL,
            quantity INTEGER DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (cart_id) REFERENCES cart_sessions(id) ON DELETE CASCADE,
            FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE,
            FOREIGN KEY (variant_id) REFERENCES product_variants(id) ON DELETE SET NULL
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS orders (
            id SERIAL PRIMARY KEY,
            user_id INTEGER NOT NULL,
            order_number TEXT UNIQUE NOT NULL,
            total_amount REAL DEFAULT 0,
            currency TEXT DEFAULT 'USD',
            status TEXT DEFAULT 'paid',
            payment_provider TEXT DEFAULT 'stripe',
            stripe_session_id TEXT,
            stripe_payment_intent TEXT,
            shipping_name TEXT,
            shipping_address TEXT,
            shipping_country TEXT,
            shipping_postal_code TEXT,
            shipping_status TEXT DEFAULT 'pending',
            tracking_number TEXT,
            tracking_url TEXT,
            order_email_sent INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES site_users(id) ON DELETE CASCADE
        )");
        $this->addColumnSafe('orders', 'order_email_sent', self::TYPE_INT_ZERO);

        $this->db->exec("CREATE TABLE IF NOT EXISTS order_items (
            id SERIAL PRIMARY KEY,
            order_id INTEGER NOT NULL,
            product_id INTEGER NOT NULL,
            variant_id INTEGER,
            product_name TEXT,
            variant_label TEXT,
            unit_price REAL DEFAULT 0,
            quantity INTEGER DEFAULT 1,
            is_digital INTEGER DEFAULT 0,
            download_url TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS digital_downloads (
            id SERIAL PRIMARY KEY,
            order_item_id INTEGER NOT NULL,
            user_id INTEGER NOT NULL,
            token TEXT UNIQUE NOT NULL,
            download_url TEXT NOT NULL,
            max_downloads INTEGER DEFAULT 5,
            download_count INTEGER DEFAULT 0,
            expires_at INTEGER,
            created_at INTEGER,
            last_download_at INTEGER,
            last_downloaded_version TEXT,
            FOREIGN KEY (order_item_id) REFERENCES order_items(id) ON DELETE CASCADE
        )");
        $this->addColumnSafe('digital_downloads', 'last_downloaded_version', 'TEXT');
        $this->addColumnSafe('digital_downloads', 'product_id', 'INTEGER');
    }

    /**
     * Site infrastructure: visitors, geo-cache, contact, bans, admin IPs.
     */
    private function createSiteInfrastructureTables(): void
    {
        $this->db->exec("CREATE TABLE IF NOT EXISTS visitors (
            id SERIAL PRIMARY KEY,
            ip_address TEXT NOT NULL,
            country_code TEXT DEFAULT 'XX',
            user_agent TEXT,
            referrer TEXT,
            first_visit TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            visit_count INTEGER DEFAULT 1,
            session_id TEXT
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS visitor_logs (
            id SERIAL PRIMARY KEY,
            visitor_id INTEGER NOT NULL,
            page_url TEXT NOT NULL,
            referrer TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (visitor_id) REFERENCES visitors(id) ON DELETE CASCADE
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS geo_cache (
            ip TEXT PRIMARY KEY,
            country_code TEXT NOT NULL,
            cached_at INTEGER NOT NULL
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS contact_messages (
            id SERIAL PRIMARY KEY,
            name TEXT NOT NULL,
            email TEXT NOT NULL,
            subject TEXT,
            message TEXT NOT NULL,
            ip_address TEXT,
            user_agent TEXT,
            referrer TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS contact_rate_limits (
            ip TEXT PRIMARY KEY,
            last_time INTEGER
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS banned_ips (
            id SERIAL PRIMARY KEY,
            ip_address TEXT UNIQUE NOT NULL,
            reason TEXT,
            banned_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            banned_by INTEGER
        )");

        $this->db->exec("CREATE TABLE IF NOT EXISTS admin_allowed_ips (
            id SERIAL PRIMARY KEY,
            ip_address TEXT UNIQUE NOT NULL,
            label TEXT,
            added_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");

        $this->addColumnSafe('visitors', 'is_admin_ip', 'BOOLEAN DEFAULT FALSE');
    }

    /**
     * Seed default admin user, theme settings, home page, and menus.
     */
    private function seedDefaults(): void
    {
        // Default admin user (Argon2id)
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM users WHERE username = ?");
        $stmt->execute(['admin']);
        if ((int) $stmt->fetchColumn() === 0) {
            $hash = password_hash('admin123', PASSWORD_ARGON2ID);
            $this->db->prepare("INSERT INTO users (username, password, email, role) VALUES (?, ?, ?, ?)")
                ->execute(['admin', $hash, 'admin@example.com', 'admin']);
        }

        // Default theme settings
        $defaultTheme = [
            'header_style' => 'top',
            'primary_color' => '#00f0ff',
            'secondary_color' => '#ff00ff',
            'accent_color' => '#00ff88',
            'background_color' => '#0a0a1a',
            'text_color' => '#ffffff',
            'font_family' => 'Orbitron, sans-serif',
        ];
        foreach ($defaultTheme as $key => $value) {
            $stmt = $this->db->prepare("INSERT INTO theme_settings (setting_key, setting_value) VALUES (?, ?) ON CONFLICT (setting_key) DO NOTHING");
            $stmt->execute([$key, $value]);
        }

        // Default home page
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM pages WHERE slug = ?");
        $stmt->execute(['home']);
        if ((int) $stmt->fetchColumn() === 0) {
            $this->db->prepare("INSERT INTO pages (title, slug, content, is_published) VALUES (?, ?, ?, ?)")
                ->execute(['Home', 'home', '', 1]);
        }

        // Default menus (header, topbar, footer)
        $defaultMenus = [
            ['Main Menu', 'header'],
            ['Top Bar Menu', 'topbar'],
            ['Footer Menu', 'footer'],
        ];
        foreach ($defaultMenus as [$name, $location]) {
            $stmt = $this->db->prepare(SQL_COUNT_MENUS_BY_LOCATION);
            $stmt->execute([$location]);
            if ((int) $stmt->fetchColumn() === 0) {
                $this->db->prepare(SQL_INSERT_MENU_NAME_LOCATION)
                    ->execute([$name, $location]);
            }
        }
    }

    /**
     * Composite performance indexes across multiple tables.
     */
    private function createPerformanceIndexes(): void
    {
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_user_messages_user_read_type_created ON user_messages(user_id, is_read, message_type, created_at DESC)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_friend_requests_to_status_created ON friend_requests(to_user_id, status, created_at DESC)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_friendships_user_friend ON friendships(user_id, friend_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_cart_sessions_session_id ON cart_sessions(session_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_visitors_last_activity ON visitors(last_activity DESC)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_visitor_logs_visitor_created ON visitor_logs(visitor_id, created_at DESC)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_geo_cache_cached_at ON geo_cache(cached_at)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_logs_created_at ON logs(created_at DESC)");
    }

    // ── Versioned migrations ───────────────────────────────────────

    /**
     * Dispatch versioned migration blocks.
     */
    private function runMigrations(int $currentVersion): void
    {
        $this->migrateV20ToV25($currentVersion);
        $this->migrateV26ToV30($currentVersion);
        $this->migrateV31ToV40($currentVersion);
    }

    /**
     * Schema versions 20–25: file thumbnails, DMCA, terms, product SEO.
     */
    private function migrateV20ToV25(int $v): void
    {
        if ($v < 20) {
            $this->addColumnSafe('files', 'thumbnail_path', 'TEXT');
        }

        if ($v < 21) {
            $this->db->exec("CREATE TABLE IF NOT EXISTS dmca_requests (
                id SERIAL PRIMARY KEY,
                status TEXT DEFAULT 'pending',
                name TEXT,
                email TEXT,
                address TEXT,
                phone TEXT,
                work_description TEXT,
                infringing_urls TEXT,
                good_faith BOOLEAN DEFAULT FALSE,
                perjury BOOLEAN DEFAULT FALSE,
                authorized BOOLEAN DEFAULT FALSE,
                signature TEXT,
                ip_address TEXT,
                user_agent TEXT,
                verify_token_hash TEXT,
                verify_expires_at TIMESTAMP,
                verified_at TIMESTAMP,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_dmca_requests_token ON dmca_requests(verify_token_hash)");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_dmca_requests_created ON dmca_requests(created_at)");

            $this->db->exec("CREATE TABLE IF NOT EXISTS dmca_rate_limits (
                rate_key TEXT PRIMARY KEY,
                last_time INTEGER DEFAULT 0,
                count INTEGER DEFAULT 0
            )");
        }

        if ($v < 22) {
            $this->addColumnSafe('site_users', 'terms_accepted_at', 'INTEGER');
            $this->addColumnSafe('site_users', 'terms_accepted_ip', 'TEXT');
            $this->addColumnSafe('site_users', 'privacy_accepted_at', 'INTEGER');
            $this->addColumnSafe('site_users', 'privacy_accepted_ip', 'TEXT');
        }

        if ($v < 23) {
            $this->addColumnSafe('products', 'external_link_label', 'TEXT');
        }

        if ($v < 24) {
            $this->addColumnSafe('product_variants', 'external_link_label', 'TEXT');
        }

        if ($v < 25) {
            $this->addColumnSafe('products', 'meta_title', 'TEXT');
            $this->addColumnSafe('products', 'meta_description', 'TEXT');
        }
    }

    /**
     * Schema versions 26–30: mini-games, leaderboard, gameplay tuning,
     * collection SEO, free download stats.
     */
    private function migrateV26ToV30(int $v): void
    {
        if ($v < 26) {
            $this->db->exec("CREATE TABLE IF NOT EXISTS mini_games (
                slug TEXT PRIMARY KEY,
                title TEXT NOT NULL,
                enabled INTEGER NOT NULL DEFAULT 1,
                sort_order INTEGER NOT NULL DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_mini_games_enabled_sort ON mini_games(enabled, sort_order, title)");

            $this->seedMiniGameCatalog();
        }

        if ($v < 27) {
            $this->db->exec("CREATE TABLE IF NOT EXISTS mini_game_leaderboard (
                id SERIAL PRIMARY KEY,
                game_slug TEXT NOT NULL,
                user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
                best_score NUMERIC(12,2) NOT NULL DEFAULT 0 CHECK (best_score >= 0),
                last_score NUMERIC(12,2) NOT NULL DEFAULT 0 CHECK (last_score >= 0),
                total_plays INTEGER NOT NULL DEFAULT 0 CHECK (total_plays >= 0),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE(game_slug, user_id)
            )");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_mini_game_leaderboard_game_score ON mini_game_leaderboard(game_slug, best_score DESC, updated_at ASC)");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_mini_game_leaderboard_user ON mini_game_leaderboard(user_id)");
        }

        if ($v < 28) {
            $this->db->exec("ALTER TABLE mini_games ADD COLUMN IF NOT EXISTS multiplier_step NUMERIC(8,4)");
            $this->db->exec("ALTER TABLE mini_games ADD COLUMN IF NOT EXISTS start_time_seconds INTEGER");
            $this->db->exec("ALTER TABLE mini_games ADD COLUMN IF NOT EXISTS time_bonus_on_make INTEGER");
            $this->db->exec("ALTER TABLE mini_games ADD COLUMN IF NOT EXISTS time_penalty_on_miss INTEGER");
            $this->db->exec("ALTER TABLE mini_games ADD COLUMN IF NOT EXISTS max_time_seconds INTEGER");

            $this->seedMiniGameTuning();
        }

        if ($v < 29) {
            $this->db->exec("ALTER TABLE collections ADD COLUMN IF NOT EXISTS meta_title TEXT");
            $this->db->exec("ALTER TABLE collections ADD COLUMN IF NOT EXISTS meta_description TEXT");
        }

        if ($v < 30) {
            $this->db->exec("CREATE TABLE IF NOT EXISTS free_digital_download_stats (
                product_id INTEGER NOT NULL,
                variant_id INTEGER NOT NULL DEFAULT 0,
                download_count BIGINT NOT NULL DEFAULT 0,
                first_download_at INTEGER NOT NULL,
                last_download_at INTEGER NOT NULL,
                PRIMARY KEY (product_id, variant_id),
                FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
            )");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_free_download_stats_count ON free_digital_download_stats(download_count DESC, last_download_at DESC)");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_free_download_stats_product ON free_digital_download_stats(product_id)");
        }
    }

    /**
     * Schema versions 31–40: product comments, slugs, attribute names, file usage notes, Google OAuth.
     */
    private function migrateV31ToV40(int $v): void
    {
        if ($v < 31) {
            $this->db->exec("ALTER TABLE products ADD COLUMN IF NOT EXISTS allow_comments INTEGER DEFAULT 1");
            $this->db->exec("UPDATE products SET allow_comments = 1 WHERE allow_comments IS NULL");

            $this->db->exec("CREATE TABLE IF NOT EXISTS product_comments (
                id SERIAL PRIMARY KEY,
                product_id INTEGER NOT NULL REFERENCES products(id) ON DELETE CASCADE,
                user_id INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
                body TEXT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_product_comments_product_created ON product_comments(product_id, created_at DESC)");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_product_comments_user_created ON product_comments(user_id, created_at DESC)");
        }

        if ($v < 32) {
            $this->backfillSlugs(
                'products',
                'product_slug',
                'name',
                'product',
                'idx_products_product_slug_unique'
            );
        }

        if ($v < 33) {
            $this->backfillSlugs(
                'collections',
                'collection_slug',
                'title',
                'collection',
                'idx_collections_collection_slug_unique'
            );
        }

        if ($v < 34) {
            $this->db->exec("CREATE TABLE IF NOT EXISTS product_attribute_names (
                id SERIAL PRIMARY KEY,
                name TEXT NOT NULL UNIQUE,
                sort_order INTEGER NOT NULL DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )");
            $this->db->exec("CREATE INDEX IF NOT EXISTS idx_product_attr_names_sort ON product_attribute_names(sort_order ASC, name ASC)");

            // Seed from existing product_attributes so nothing is lost
            $this->db->exec("INSERT INTO product_attribute_names (name)
                SELECT DISTINCT name FROM product_attributes
                WHERE name <> ''
                ON CONFLICT (name) DO NOTHING");
        }

        if ($v < 35) {
            // Allow admin to manually mark files as "used" with a free-text note
            $this->addColumnSafe('files', 'manual_usage_note', 'TEXT');
        }

        if ($v < 36) {
            // Google OAuth: store the Google account ID for linked users
            $this->addColumnSafe('site_users', 'google_id', 'TEXT');
            $this->db->exec(
                "CREATE UNIQUE INDEX IF NOT EXISTS idx_site_users_google_id ON site_users(google_id) WHERE google_id IS NOT NULL"
            );
        }

        if ($v < 37) {
            // Refund request tracking for digital purchases
            $this->db->exec("
                CREATE TABLE IF NOT EXISTS refund_requests (
                    id              SERIAL PRIMARY KEY,
                    order_id        INTEGER NOT NULL REFERENCES orders(id) ON DELETE CASCADE,
                    order_item_id   INTEGER NOT NULL REFERENCES order_items(id) ON DELETE CASCADE,
                    user_id         INTEGER NOT NULL REFERENCES site_users(id) ON DELETE CASCADE,
                    status          TEXT NOT NULL DEFAULT 'pending',
                    admin_reason    TEXT DEFAULT '',
                    created_at      TEXT NOT NULL DEFAULT '',
                    updated_at      TEXT NOT NULL DEFAULT ''
                )
            ");
            $this->execSafe("CREATE INDEX IF NOT EXISTS idx_refund_requests_user ON refund_requests(user_id)");
            $this->execSafe("CREATE INDEX IF NOT EXISTS idx_refund_requests_order ON refund_requests(order_id)");
            $this->execSafe("CREATE INDEX IF NOT EXISTS idx_refund_requests_status ON refund_requests(status)");
            // Prevent duplicate refund requests per order item
            $this->execSafe("CREATE UNIQUE INDEX IF NOT EXISTS idx_refund_requests_item ON refund_requests(order_item_id)");
        }

        if ($v < 38) {
            // Store return instruction file paths (JSON array) for physical item refunds
            $this->addColumnSafe('refund_requests', 'return_instructions', "TEXT DEFAULT ''");
        }

        if ($v < 39) {
            // SEO: add slug column to forum_posts for pretty URLs (/forum/post/{slug})
            $this->backfillSlugs('forum_posts', 'slug', 'title', 'post', 'idx_forum_posts_slug_unique');
        }
    }

    // ── Helpers ─────────────────────────────────────────────────────

    /**
     * Add a column to a table, ignoring the error if it already exists.
     * Eliminates repetitive try/catch blocks throughout the schema.
     */
    private function addColumnSafe(string $table, string $column, string $type): void
    {
        try {
            $this->db->exec("ALTER TABLE {$table} ADD COLUMN {$column} {$type}");
        } catch (PDOException) {
            // Column already exists — safe to ignore
        }
    }

    /**
     * Execute a SQL statement, ignoring errors (for idempotent operations).
     */
    private function execSafe(string $sql): void
    {
        try {
            $this->db->exec($sql);
        } catch (PDOException) {
            // Already applied — safe to ignore
        }
    }

    /**
     * Update last_activity for the currently logged-in site user.
     * May fail during early migration if the column doesn't exist yet.
     */
    private function updateUserActivity(): void
    {
        if (empty($_SESSION['site_user_id'])) {
            return;
        }

        $userId = (int) $_SESSION['site_user_id'];
        try {
            $this->db->prepare(SQL_UPDATE_SITE_USER_LAST_ACTIVITY)
                ->execute([time(), $userId]);
        } catch (PDOException $e) {
            error_log('Skipping last_activity update: ' . $e->getMessage());
        }
    }

    /**
     * Seed mini-game catalog from MinigameCatalog (schema v26).
     */
    private function seedMiniGameCatalog(): void
    {
        try {
            $catalog = MinigameCatalog::getCatalog();
            if (!is_array($catalog) || empty($catalog)) {
                return;
            }

            $now = DbHelper::nowString();
            $seed = $this->db->prepare(
                'INSERT INTO mini_games (slug, title, enabled, sort_order, created_at, updated_at) '
                . 'VALUES (?, ?, 1, ?, ?, ?) '
                . 'ON CONFLICT (slug) DO UPDATE SET title = EXCLUDED.title, sort_order = EXCLUDED.sort_order, updated_at = EXCLUDED.updated_at'
            );

            foreach ($catalog as $game) {
                if (!is_array($game)) {
                    continue;
                }
                $slug = isset($game['slug']) ? (string) $game['slug'] : '';
                $slug = preg_replace('/[^a-z0-9\-]/', '', strtolower(trim($slug)));
                if (!is_string($slug) || $slug === '') {
                    continue;
                }
                $title = isset($game['title']) ? (string) $game['title'] : $slug;
                $sort = isset($game['sort']) ? (int) $game['sort'] : 0;
                $seed->execute([$slug, $title, $sort, $now, $now]);
            }
        } catch (Throwable) {
            // Ignore seeding failures; admin page can still manage game state.
        }
    }

    /**
     * Seed mini-game gameplay tuning defaults (schema v28).
     */
    private function seedMiniGameTuning(): void
    {
        try {
            $catalog = MinigameCatalog::getCatalog();
            if (!is_array($catalog) || empty($catalog)) {
                return;
            }

            $seedSettings = $this->db->prepare(
                'UPDATE mini_games SET '
                . 'multiplier_step = COALESCE(multiplier_step, ?), '
                . 'start_time_seconds = COALESCE(start_time_seconds, ?), '
                . 'time_bonus_on_make = COALESCE(time_bonus_on_make, ?), '
                . 'time_penalty_on_miss = COALESCE(time_penalty_on_miss, ?), '
                . 'max_time_seconds = COALESCE(max_time_seconds, ?), '
                . 'updated_at = ? '
                . 'WHERE slug = ?'
            );

            foreach ($catalog as $catalogSlug => $game) {
                if (!is_array($game)) {
                    continue;
                }

                $slug = isset($game['slug']) ? (string) $game['slug'] : (string) $catalogSlug;
                $slug = preg_replace('/[^a-z0-9\-]/', '', strtolower(trim($slug)));
                if (!is_string($slug) || $slug === '') {
                    continue;
                }

                $defaults = MinigameCatalog::getGameplayDefaults($slug);
                if (!is_array($defaults) || empty($defaults)) {
                    continue;
                }

                $seedSettings->execute([
                    (float) ($defaults['multiplier_step'] ?? 1.12),
                    (int) ($defaults['start_time_seconds'] ?? 30),
                    (int) ($defaults['time_bonus_on_make'] ?? 5),
                    (int) ($defaults['time_penalty_on_miss'] ?? 10),
                    (int) ($defaults['max_time_seconds'] ?? 99),
                    DbHelper::nowString(),
                    $slug,
                ]);
            }
        } catch (Throwable) {
            // Ignore default tuning seed failures.
        }
    }

    /**
     * Add a slug column, backfill from a name/title column, ensure uniqueness.
     * Used for both products (v32) and collections (v33).
     */
    private function backfillSlugs(
        string $table,
        string $slugColumn,
        string $nameColumn,
        string $fallbackPrefix,
        string $indexName
    ): void {
        $this->db->exec("ALTER TABLE {$table} ADD COLUMN IF NOT EXISTS {$slugColumn} TEXT");

        $rows = $this->db->query("SELECT id, {$nameColumn}, {$slugColumn} FROM {$table} ORDER BY id ASC")
            ->fetchAll(PDO::FETCH_ASSOC);
        $slugExistsStmt = $this->db->prepare("SELECT id FROM {$table} WHERE {$slugColumn} = ? AND id <> ? LIMIT 1");
        $slugUpdateStmt = $this->db->prepare("UPDATE {$table} SET {$slugColumn} = ? WHERE id = ?");

        foreach ($rows as $row) {
            $entityId = (int) ($row['id'] ?? 0);
            if ($entityId <= 0) {
                continue;
            }

            $currentSlug = trim((string) ($row[$slugColumn] ?? ''));
            $normalizedSlug = SlugGenerator::generate($currentSlug);
            if ($normalizedSlug === '') {
                $normalizedSlug = SlugGenerator::generate((string) ($row[$nameColumn] ?? ''));
            }
            if ($normalizedSlug === '') {
                $normalizedSlug = $fallbackPrefix;
            }

            $candidateSlug = $normalizedSlug;
            $suffix = 2;
            while (true) {
                $slugExistsStmt->execute([$candidateSlug, $entityId]);
                $existsRow = $slugExistsStmt->fetch(PDO::FETCH_ASSOC);
                if (!$existsRow) {
                    break;
                }
                $candidateSlug = $normalizedSlug . '-' . $suffix;
                $suffix++;
            }

            if ($candidateSlug !== $currentSlug) {
                $slugUpdateStmt->execute([$candidateSlug, $entityId]);
            }
        }

        $this->db->exec("CREATE UNIQUE INDEX IF NOT EXISTS {$indexName} ON {$table}({$slugColumn}) WHERE {$slugColumn} IS NOT NULL");
    }
}
