<?php

declare(strict_types=1);

namespace NewSite\Database;

use Exception;
use NewSite\Config\SetupService;
use PDO;

/**
 * Database Connection Manager
 *
 * Supports read replicas for load distribution.
 * Write queries go to primary; read queries go to replicas when configured.
 *
 * Migrated from includes/database.php to PSR-4 namespace NewSite\Database.
 * Legacy call sites may still reference the unqualified `DatabaseManager`
 * class via the backward-compatible alias registered in includes/autoload.php.
 */
class DatabaseManager
{
    private static ?PDO $writeDB = null;

    /** @var PDO[] */
    private static array $readReplicas = [];

    private static int $currentReplicaIndex = 0;

    // ------------------------------------------------------------------
    //  Write connection (primary)
    // ------------------------------------------------------------------

    /**
     * Get write (primary) database connection.
     *
     * A static re-entrancy guard prevents infinite recursion if
     * `SetupService::getEnvOrConfig()` triggers a database call during initialisation.
     */
    public static function getWriteConnection(): ?PDO
    {
        static $initializing = false;

        if ($initializing) {
            return self::$writeDB;
        }

        $initializing = true;

        if (self::$writeDB === null) {
            self::$writeDB = self::createConnection(
                SetupService::getEnvOrConfig('DB_HOST', ''),
                SetupService::getEnvOrConfig('DB_PORT', '5432'),
                SetupService::getEnvOrConfig('DB_NAME', ''),
                SetupService::getEnvOrConfig('DB_USER', ''),
                SetupService::getEnvOrConfig('DB_PASS', ''),
                SetupService::getEnvOrConfig('DB_SSLMODE', 'require')
            );
        }

        $initializing = false;
        return self::$writeDB;
    }

    // ------------------------------------------------------------------
    //  Read connection (replica pool or primary fallback)
    // ------------------------------------------------------------------

    /**
     * Get read database connection (replica pool or primary fallback).
     */
    public static function getReadConnection(): ?PDO
    {
        static $initializing = false;

        if ($initializing) {
            return self::getWriteConnection();
        }

        $initializing = true;

        if (empty(self::$readReplicas)) {
            self::initReplicas();
        }

        if (empty(self::$readReplicas)) {
            $initializing = false;
            return self::getWriteConnection();
        }

        // Round-robin load balancing across replicas
        $replicaCount = count(self::$readReplicas);
        $index        = self::$currentReplicaIndex % $replicaCount;
        self::$currentReplicaIndex++;

        $initializing = false;
        return self::$readReplicas[$index];
    }

    // ------------------------------------------------------------------
    //  Replica initialisation
    // ------------------------------------------------------------------

    /**
     * Initialise read replicas from configuration.
     */
    private static function initReplicas(): void
    {
        static $initializing = false;

        if ($initializing) {
            return;
        }

        $initializing = true;

        self::initSingleReplica();
        self::initNumberedReplicas();

        $initializing = false;
    }

    /**
     * Try to connect a single-replica configuration (DB_READ_HOST).
     */
    private static function initSingleReplica(): void
    {
        $host = SetupService::getEnvOrConfig('DB_READ_HOST', '');

        if ($host === '') {
            return;
        }

        try {
            $replica = self::createConnection(
                $host,
                SetupService::getEnvOrConfig('DB_READ_PORT', SetupService::getEnvOrConfig('DB_PORT', '5432')),
                SetupService::getEnvOrConfig('DB_READ_NAME', SetupService::getEnvOrConfig('DB_NAME', '')),
                SetupService::getEnvOrConfig('DB_READ_USER', SetupService::getEnvOrConfig('DB_USER', '')),
                SetupService::getEnvOrConfig('DB_READ_PASS', SetupService::getEnvOrConfig('DB_PASS', '')),
                SetupService::getEnvOrConfig('DB_READ_SSLMODE', SetupService::getEnvOrConfig('DB_SSLMODE', 'require'))
            );
            self::$readReplicas[] = $replica;
        } catch (Exception $e) {
            error_log('Read replica connection failed: ' . $e->getMessage());
        }
    }

    /**
     * Try to connect numbered replicas (REPLICA_1_HOST … REPLICA_10_HOST).
     */
    private static function initNumberedReplicas(): void
    {
        for ($i = 1; $i <= 10; $i++) {
            $host = SetupService::getEnvOrConfig("REPLICA_{$i}_HOST", '');

            if ($host === '') {
                break;
            }

            try {
                $replica = self::createConnection(
                    $host,
                    SetupService::getEnvOrConfig("REPLICA_{$i}_PORT", SetupService::getEnvOrConfig('DB_PORT', '5432')),
                    SetupService::getEnvOrConfig("REPLICA_{$i}_NAME", SetupService::getEnvOrConfig('DB_NAME', '')),
                    SetupService::getEnvOrConfig("REPLICA_{$i}_USER", SetupService::getEnvOrConfig('DB_USER', '')),
                    SetupService::getEnvOrConfig("REPLICA_{$i}_PASS", SetupService::getEnvOrConfig('DB_PASS', '')),
                    SetupService::getEnvOrConfig("REPLICA_{$i}_SSLMODE", SetupService::getEnvOrConfig('DB_SSLMODE', 'require'))
                );
                self::$readReplicas[] = $replica;
            } catch (Exception $e) {
                error_log("Replica {$i} connection failed: " . $e->getMessage());
            }
        }
    }

    // ------------------------------------------------------------------
    //  Connection factory
    // ------------------------------------------------------------------

    /**
     * Create a validated PDO connection.
     *
     * Security: uses parameterised DSN components from the trusted config
     * layer.  Connection attributes enforce strict error handling and
     * disable emulated prepared statements.
     */
    private static function createConnection(
        string $host,
        string $port,
        string $name,
        string $user,
        string $pass,
        string $sslmode
    ): PDO {
        if ($host === '' || $name === '' || $user === '' || $pass === '') {
            throw new DatabaseConfigurationException(
                'Database configuration incomplete'
            );
        }

        $dsn = "pgsql:host={$host};port={$port};dbname={$name};sslmode={$sslmode}";
        $db  = new PDO($dsn, $user, $pass);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $db->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        $db->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);

        return $db;
    }

    // ------------------------------------------------------------------
    //  Diagnostics
    // ------------------------------------------------------------------

    /**
     * Get status of all connections.
     *
     * @return array{primary: array, replicas: list<array>}
     */
    public static function getStatus(): array
    {
        $status = [
            'primary' => [
                'host'      => SetupService::getEnvOrConfig('DB_HOST', ''),
                'connected' => self::$writeDB !== null,
            ],
            'replicas' => [],
        ];

        foreach (self::$readReplicas as $i => $replica) {
            $status['replicas'][] = [
                'index'     => $i,
                'connected' => true,
            ];
        }

        if (empty($status['replicas'])) {
            $status['replicas'][] = [
                'note' => 'Using primary for reads (no replicas configured)',
            ];
        }

        return $status;
    }
}
