<?php

declare(strict_types=1);

namespace NewSite\Config;

use NewSite\Config\ConfigLoader;
use NewSite\Settings\SettingsService;

/**
 * SetupService — application bootstrap helpers and configuration utilities.
 *
 * Security: configuration values are read from ConfigLoader or environment
 * variables only; no user input is trusted without validation. Setup-redirect
 * logic prevents access to the application before initial configuration is
 * complete.
 */
final class SetupService
{
    public static function getSystemPageSlugs(): array
    {
        return [
            'home', 'contact', 'cart', 'checkout', 'games',
            'product', 'products', 'collections',
            'privacy-policy', 'terms-of-service', 'dmca',
            'refund-policy', 'shipping-policy', 'legal-notice',
            'easy-media-ai', 'text-finder-replacer'
        ];
    }

    public static function isSystemPageSlug(string $slug): bool
    {
        return in_array(strtolower(trim($slug)), self::getSystemPageSlugs(), true);
    }

    public static function shouldRunDailyTask(string $key, int $intervalSeconds = 86400): bool
    {
        $safeKey = preg_replace('/[^a-z0-9_\-]/i', '_', $key);
        $cacheDir = dirname(__DIR__, 2) . '/data/cache';
        if (!is_dir($cacheDir)) {
            @mkdir($cacheDir, 0755, true);
        }
        $path = $cacheDir . '/daily_' . $safeKey . '.json';
        $now = time();
        if (is_file($path)) {
            $data = json_decode((string)@file_get_contents($path), true) ?: [];
            $lastRun = (int)($data['last_run'] ?? 0);
            if ($lastRun > 0 && ($now - $lastRun) < $intervalSeconds) {
                return false;
            }
        }
        @file_put_contents($path, json_encode(['last_run' => $now]), LOCK_EX);
        return true;
    }

    public static function isSetupRequired(): bool
    {
        $jsonPath = dirname(__DIR__, 2) . '/data/config.json';
        $setupLockPath = dirname(__DIR__, 2) . '/data/.setup_complete';
        $setupComplete = file_exists($setupLockPath);
        $configExists = file_exists($jsonPath);
        $configValid = $configExists
            && ConfigLoader::isLoaded()
            && ConfigLoader::getValue('DB_HOST', '') !== '';
        return !$setupComplete && (!$configExists || !$configValid);
    }

    public static function redirectToSetupIfNeeded(): void
    {
        $requestUri = $_SERVER['REQUEST_URI'] ?? '';
        $isSetupPage = strpos($requestUri, 'setup.php') !== false;
        $isAllowlistTool = strpos($requestUri, 'allowlist_tool.php') !== false;
        if (!$isSetupPage && !$isAllowlistTool && self::isSetupRequired()) {
            header('Location: /setup.php');
            exit;
        }
    }

    public static function getConfigValue(string $key, string $default = ''): string
    {
        return (string)ConfigLoader::getValue($key, $default);
    }

    public static function getEnvOrConfig(string $key, string $default = ''): string
    {
        $value = getenv($key);
        if ($value !== false && $value !== '') {
            return (string)$value;
        }
        $configValue = self::getConfigValue($key, $default);
        return $configValue === '' ? $default : $configValue;
    }

    public static function getCurrentLanguage(): string
    {
        return $_SESSION['site_language'] ?? 'en';
    }

    public static function getBaseUrl(): string
    {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $scheme . '://' . $host;
    }
}
