<?php

declare(strict_types=1);

namespace NewSite\Config;

/**
 * Centralized configuration loader — caches the parsed config array in memory.
 *
 * The bootstrap sequence in _init.php loads `data/config.json` once (before the
 * autoloader is registered) and passes the result to {@see init()}.  Every
 * subsequent config read goes through {@see getValue()} without another file
 * include, satisfying both the PSR-4 "use instead of include" rule and the
 * requirement that the config is loaded exactly once.
 *
 * Security: No user input influences the config path. The config array is
 * treated as read-only after init().
 */
class ConfigLoader
{
    /** @var array<string, mixed> Cached config key-value pairs */
    private static array $config = [];

    /** @var bool Whether init() has been called */
    private static bool $initialized = false;

    /**
     * Seed the loader with the already-parsed config array.
     *
     * Must be called exactly once, right after the PSR-4 autoloader is
     * registered and before any consumer calls getValue().
     *
     * @param array<string, mixed> $config Parsed config.php return value
     */
    public static function init(array $config): void
    {
        self::$config = $config;
        self::$initialized = true;
    }

    /**
     * Whether a non-empty config was loaded successfully.
     *
     * Useful for the setup-required check: if the config file was missing or
     * returned an empty array, this returns false.
     */
    public static function isLoaded(): bool
    {
        return self::$initialized && self::$config !== [];
    }

    /**
     * Retrieve a single config value by key.
     *
     * @param string $key     Config key (e.g. 'DB_HOST')
     * @param mixed  $default Returned when the key is absent
     */
    public static function getValue(string $key, mixed $default = ''): mixed
    {
        return array_key_exists($key, self::$config) ? self::$config[$key] : $default;
    }

    /**
     * Return the full config array (read-only copy).
     *
     * @return array<string, mixed>
     */
    public static function getAll(): array
    {
        return self::$config;
    }
}
