<?php

declare(strict_types=1);

namespace NewSite\Cleanup;

use NewSite\Database\DatabaseManager;
use NewSite\Settings\SettingsService;
use NewSite\Logging\LogService;

/**
 * MissingUploadsService — detects and removes references to missing upload files.
 *
 * Security: all database queries use prepared statements. File existence checks
 * use basename() resolution against known safe directories to prevent path
 * traversal. HTML content is sanitised via regex callbacks that only strip
 * broken image/source/video tags.
 */
final class MissingUploadsService
{
    private const PROFILE_PHOTO_ROUTE_PREFIX = '/profile-photo/';
    private const CHAT_IMAGE_ROUTE_PREFIX = '/chat-image/';

    public static function stripFromHtml(string $html): string
    {
        if ($html === '') {
            return $html;
        }
        $publicPath = dirname(__DIR__, 2) . '/public';
        return (string)preg_replace_callback(
            '/<(img|source|video)\b[^>]*\bsrc=["\'](\/assets\/uploads\/[^"\']+)["\'][^>]*>/i',
            static function (array $matches) use ($publicPath): string {
                $path = $matches[2];
                $fullPath = $publicPath . $path;
                return is_file($fullPath) ? $matches[0] : '';
            },
            $html
        );
    }

    public static function isMissingPath(string $path): bool
    {
        if ($path === '') {
            return false;
        }
        $isMissing = false;
        $dataDir = dirname(__DIR__, 2) . '/data/';
        $publicPath = dirname(__DIR__, 2) . '/public';
        if (strpos($path, '/assets/uploads/') === 0) {
            $isMissing = !is_file($publicPath . $path);
        } elseif (strpos($path, '/admin-file/') === 0) {
            $isMissing = !is_file($dataDir . 'admin_uploads/' . basename($path));
        } elseif (strpos($path, self::PROFILE_PHOTO_ROUTE_PREFIX) === 0) {
            $isMissing = !is_file($dataDir . 'profile_photos/' . basename($path));
        } elseif (strpos($path, self::CHAT_IMAGE_ROUTE_PREFIX) === 0) {
            $isMissing = !is_file($dataDir . 'chat_images/' . basename($path));
        } elseif (strpos($path, '/contact-upload/') === 0) {
            $isMissing = !is_file($dataDir . 'contact_uploads/' . basename($path));
        }
        return $isMissing;
    }

    public static function stripFromSettings(array $settings, bool &$changed = false): array
    {
        foreach ($settings as $key => $value) {
            if (is_array($value)) {
                $settings[$key] = self::stripFromSettings($value, $changed);
            } elseif (is_string($value) && self::isMissingPath($value)) {
                $settings[$key] = '';
                $changed = true;
            }
        }
        return $settings;
    }

    public static function cleanupReferences(bool $force = false): void
    {
        $lastRun = (int)SettingsService::get('missing_uploads_cleanup', '0');
        if (!$force && $lastRun > 0 && (time() - $lastRun) < 86400) {
            return;
        }
        $db = DatabaseManager::getWriteConnection();
        $changedCount = 0;

        // Theme settings
        $stmt = $db->query("SELECT setting_key, setting_value FROM theme_settings");
        while ($row = $stmt->fetch()) {
            if (is_string($row['setting_value']) && self::isMissingPath($row['setting_value'])) {
                SettingsService::setTheme($row['setting_key'], '');
                $changedCount++;
            }
        }

        // Sections settings
        $sections = $db->query("SELECT id, settings FROM sections")->fetchAll();
        foreach ($sections as $section) {
            $settings = json_decode($section['settings'] ?? '{}', true);
            if (!is_array($settings)) {
                continue;
            }
            $changed = false;
            $settings = self::stripFromSettings($settings, $changed);
            if ($changed) {
                $stmtUpdate = $db->prepare("UPDATE sections SET settings = ? WHERE id = ?");
                $stmtUpdate->execute([json_encode($settings), $section['id']]);
                $changedCount++;
            }
        }

        // Page content
        $pages = $db->query("SELECT id, content FROM pages WHERE content LIKE '%/assets/uploads/%'")->fetchAll();
        foreach ($pages as $page) {
            $cleaned = self::stripFromHtml((string)($page['content'] ?? ''));
            if ($cleaned !== $page['content']) {
                $stmtUpdate = $db->prepare("UPDATE pages SET content = ? WHERE id = ?");
                $stmtUpdate->execute([$cleaned, $page['id']]);
                $changedCount++;
            }
        }

        SettingsService::set('missing_uploads_cleanup', (string)time());
        if ($changedCount > 0) {
            LogService::add('info', 'Removed missing upload references', json_encode(['count' => $changedCount]));
        }
    }
}
