<?php

declare(strict_types=1);

namespace NewSite\Cache;

use NewSite\Database\DatabaseManager;
use PDO;

/**
 * Domain-specific cached database queries.
 *
 * Wraps frequently-accessed read queries with short TTL caching via
 * {@see CacheService}.  Extracted from the former global helper
 * functions in includes/cache.php to satisfy the PSR-4 migration and
 * the maximum-methods-per-class rule.
 */
class CachedQueries
{
    /**
     * Get notification counts with caching (30-second TTL).
     */
    public static function getNotificationData(int $userId): array
    {
        $cacheKey = "notifications:{$userId}";

        $result = CacheService::remember(
            $cacheKey,
            30,
            static function () use ($userId): array {
                return self::fetchNotificationData($userId);
            }
        );

        return is_array($result) ? $result : [];
    }

    /**
     * Fetch notification data from database (no cache).
     */
    private static function fetchNotificationData(int $userId): array
    {
        $db = DatabaseManager::getReadConnection();

        // Unread NOTICE messages only (not tickets/support chat)
        $stmt = $db->prepare(
            "SELECT COUNT(*) FROM user_messages
              WHERE user_id = ? AND is_read = 0 AND message_type = 'notice'"
        );
        $stmt->execute([$userId]);
        $unread = (int) $stmt->fetchColumn();

        // Unread ticket messages (support chat)
        $stmt = $db->prepare(
            "SELECT COUNT(*) FROM user_messages
              WHERE user_id = ? AND is_read = 0 AND message_type = 'ticket'"
        );
        $stmt->execute([$userId]);
        $ticketUnread = (int) $stmt->fetchColumn();

        $items          = self::fetchUnreadNoticeItems($db, $userId);
        $pendingFriends = self::fetchPendingFriendCount($db, $userId);
        $latestRequester = $pendingFriends > 0
            ? self::fetchLatestFriendRequester($db, $userId)
            : null;

        return [
            'success'                 => true,
            'unread'                  => $unread,
            'ticket_unread'           => $ticketUnread,
            'unread_total'            => $unread + $ticketUnread,
            'pending_friend_requests' => $pendingFriends,
            'latest_friend_requester' => $latestRequester,
            'items'                   => $items,
        ];
    }

    /**
     * Fetch unread notice items (max 5).
     */
    private static function fetchUnreadNoticeItems(PDO $db, int $userId): array
    {
        $stmt = $db->prepare(
            "SELECT id, subject, body, is_read, message_type
               FROM user_messages
              WHERE user_id = ? AND message_type = 'notice' AND is_read = 0
           ORDER BY created_at DESC
              LIMIT 5"
        );
        $stmt->execute([$userId]);
        $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $items = [];
        foreach ($messages as $msg) {
            $items[] = [
                'id'           => (int) $msg['id'],
                'subject'      => $msg['subject'] ?? 'Message',
                'body'         => $msg['body'] ?? '',
                'is_read'      => !empty($msg['is_read']),
                'message_type' => $msg['message_type'] ?? 'notice',
            ];
        }

        return $items;
    }

    /**
     * Count pending friend requests.
     */
    private static function fetchPendingFriendCount(PDO $db, int $userId): int
    {
        $stmt = $db->prepare(
            "SELECT COUNT(*) FROM friend_requests
              WHERE to_user_id = ? AND status = 'pending'"
        );
        $stmt->execute([$userId]);
        return (int) $stmt->fetchColumn();
    }

    /**
     * Get the nickname of the latest friend requester.
     */
    private static function fetchLatestFriendRequester(PDO $db, int $userId): ?string
    {
        $stmt = $db->prepare(
            "SELECT u.nickname, u.display_name, u.email
               FROM friend_requests fr
               JOIN site_users u ON fr.from_user_id = u.id
              WHERE fr.to_user_id = ? AND fr.status = 'pending'
           ORDER BY fr.created_at DESC
              LIMIT 1"
        );
        $stmt->execute([$userId]);
        $requester = $stmt->fetch();

        if (!$requester) {
            return null;
        }

        return $requester['nickname']
            ?? $requester['display_name']
            ?? explode('@', $requester['email'])[0]
            ?? 'Someone';
    }

    /**
     * Invalidate notification cache when data changes.
     */
    public static function invalidateNotificationCache(int $userId): void
    {
        CacheService::delete("notifications:{$userId}");
    }

    /**
     * Get user profile with caching (5-minute TTL).
     */
    public static function getUserProfile(int $userId): ?array
    {
        $cacheKey = "user:{$userId}";

        return CacheService::remember(
            $cacheKey,
            300,
            static function () use ($userId): ?array {
                $db   = DatabaseManager::getReadConnection();
                $stmt = $db->prepare(
                    "SELECT id, nickname, display_name, profile_photo, is_active
                       FROM site_users WHERE id = ?"
                );
                $stmt->execute([$userId]);
                return $stmt->fetch() ?: null;
            }
        );
    }

    /**
     * Invalidate user cache.
     */
    public static function invalidateUserCache(int $userId): void
    {
        CacheService::delete("user:{$userId}");
    }

    /**
     * Get friend count with caching (1-minute TTL).
     */
    public static function getFriendsCount(int $userId): int
    {
        $cacheKey = "friends_count:{$userId}";

        $result = CacheService::remember(
            $cacheKey,
            60,
            static function () use ($userId): int {
                $db   = DatabaseManager::getReadConnection();
                $stmt = $db->prepare(
                    "SELECT COUNT(*) FROM friendships WHERE user_id = ? OR friend_id = ?"
                );
                $stmt->execute([$userId, $userId]);
                return (int) $stmt->fetchColumn();
            }
        );

        return (int) $result;
    }
}
