<?php

declare(strict_types=1);

namespace NewSite\Cache;

use Exception;
use NewSite\Config\SetupService;
use PDO;

/**
 * Caching Layer
 *
 * Supports Redis (primary) with APCu and file-based fallbacks.
 * Reduces database load by 95 %+ for frequently accessed data.
 *
 * Migrated from includes/cache.php to PSR-4 namespace NewSite\Cache.
 * Legacy call sites may still reference the unqualified `Cache` class
 * via the backward-compatible alias registered in includes/autoload.php.
 */
class CacheService
{
    /** @var \Redis|null */
    private static $redis = null;
    private static bool $connected = false;
    private static bool $useFile = false;
    private static ?string $cacheDir = null;
    private const FILE_CACHE_GLOB = '/*.cache';

    // ------------------------------------------------------------------
    //  Initialisation
    // ------------------------------------------------------------------

    /**
     * Initialize cache connection.
     *
     * Tries Redis first, then APCu, then falls back to file-based caching.
     */
    public static function init(): void
    {
        if (self::$connected) {
            return;
        }

        self::$cacheDir = DATA_PATH . '/cache';
        if (!is_dir(self::$cacheDir)) {
            @mkdir(self::$cacheDir, 0755, true);
        }

        if (self::tryRedis()) {
            return;
        }

        if (self::tryApcu()) {
            return;
        }

        // Fallback to file cache
        self::$connected = true;
        self::$useFile   = true;
    }

    /**
     * Attempt a Redis connection.
     */
    private static function tryRedis(): bool
    {
        $redisHost = SetupService::getEnvOrConfig('REDIS_HOST', '');
        $redisPort = (int) SetupService::getEnvOrConfig('REDIS_PORT', '6379');
        $redisPass = SetupService::getEnvOrConfig('REDIS_PASS', '');

        if ($redisHost === '' || !extension_loaded('redis') || !class_exists('Redis')) {
            return false;
        }

        try {
            $redisClass  = 'Redis';
            self::$redis = new $redisClass();
            self::$redis->connect($redisHost, $redisPort, 2.0);

            if ($redisPass !== '') {
                self::$redis->auth($redisPass);
            }

            // Configure Redis prefix and serializer options
            $prefixConst        = $redisClass . '::OPT_PREFIX';
            $serializerOptConst = $redisClass . '::OPT_SERIALIZER';
            $serializerPhpConst = $redisClass . '::SERIALIZER_PHP';

            if (defined($prefixConst)) {
                self::$redis->setOption((int) constant($prefixConst), 'site:');
            }

            if (defined($serializerOptConst) && defined($serializerPhpConst)) {
                self::$redis->setOption(
                    (int) constant($serializerOptConst),
                    (int) constant($serializerPhpConst)
                );
            }

            self::$connected = true;
            self::$useFile   = false;
            return true;
        } catch (Exception $e) {
            error_log('Redis connection failed: ' . $e->getMessage());
            self::$redis = null;
            return false;
        }
    }



    /**
     * Attempt APCu initialisation.
     */
    private static function tryApcu(): bool
    {
        if (
            extension_loaded('apcu')
            && function_exists('apcu_enabled')
            && (bool) call_user_func('apcu_enabled')
        ) {
            self::$connected = true;
            self::$useFile   = false;
            return true;
        }

        return false;
    }

    // ------------------------------------------------------------------
    //  Core CRUD
    // ------------------------------------------------------------------

    /**
     * Get cache file path for a given key.
     */
    private static function getCacheFile(string $key): string
    {
        return self::$cacheDir . '/' . md5($key) . '.cache';
    }

    /**
     * Get value from cache.
     *
     * A static re-entrancy guard prevents infinite recursion if a cache
     * callback triggers another read for the same key.
     */
    public static function get(string $key, mixed $default = null): mixed
    {
        static $getting = [];

        if (isset($getting[$key])) {
            return $default;
        }

        $getting[$key] = true;
        self::init();

        $value = self::readFromBackend($key, $default);

        unset($getting[$key]);
        return $value;
    }

    /**
     * Read a value from the active cache backend.
     */
    private static function readFromBackend(string $key, mixed $default): mixed
    {
        if (self::$redis) {
            $value = self::$redis->get($key);
            return $value !== false ? $value : $default;
        }

        if (!self::$useFile && extension_loaded('apcu') && function_exists('apcu_fetch')) {
            $success = false;
            $tmp = call_user_func('apcu_fetch', $key, $success);
            return $success ? $tmp : $default;
        }

        return self::readFileCache($key, $default);
    }

    /**
     * Read a value from the file-based cache.
     */
    private static function readFileCache(string $key, mixed $default): mixed
    {
        $file = self::getCacheFile($key);
        $raw  = file_exists($file) ? file_get_contents($file) : false;

        if ($raw === false) {
            return $default;
        }

        // @unserialize: corrupt cache files must not emit warnings;
        // returning the default value is the correct fallback.
        $data = @unserialize($raw);

        if (is_array($data) && isset($data['expires']) && $data['expires'] > time()) {
            return $data['value'];
        }

        // Expired — clean up
        if (is_file($file)) {
            @unlink($file);
        }

        return $default;
    }

    /**
     * Set value in cache.
     */
    public static function set(string $key, mixed $value, int $ttl = 60): bool
    {
        static $setting = [];

        if (isset($setting[$key])) {
            return false;
        }

        $setting[$key] = true;
        self::init();

        $result = self::writeToBackend($key, $value, $ttl);

        unset($setting[$key]);
        return $result;
    }

    /**
     * Write a value to the active cache backend.
     */
    private static function writeToBackend(string $key, mixed $value, int $ttl): bool
    {
        if (self::$redis) {
            return (bool) self::$redis->setex($key, $ttl, $value);
        }

        if (!self::$useFile && extension_loaded('apcu') && function_exists('apcu_store')) {
            return (bool) call_user_func('apcu_store', $key, $value, $ttl);
        }

        return self::writeFileCache($key, $value, $ttl);
    }

    /**
     * Write a value to the file-based cache.
     */
    private static function writeFileCache(string $key, mixed $value, int $ttl): bool
    {
        $file = self::getCacheFile($key);
        $data = serialize(['value' => $value, 'expires' => time() + $ttl]);
        return @file_put_contents($file, $data, LOCK_EX) !== false;
    }

    /**
     * Delete value from cache.
     */
    public static function delete(string $key): bool
    {
        self::init();

        if (self::$redis) {
            return (bool) self::$redis->del($key);
        }

        if (!self::$useFile && extension_loaded('apcu') && function_exists('apcu_delete')) {
            return (bool) call_user_func('apcu_delete', $key);
        }

        $file = self::getCacheFile($key);
        return !is_file($file) || @unlink($file);
    }

    /**
     * Delete all keys matching a pattern.
     *
     * Only Redis supports pattern-based deletion.  For APCu / file
     * backends this falls back to exact-key deletion.
     */
    public static function deletePattern(string $pattern): bool
    {
        self::init();

        if (self::$redis) {
            $keys = self::$redis->keys($pattern);
            if (!empty($keys)) {
                return (bool) self::$redis->del($keys);
            }
            return true;
        }

        return self::delete($pattern);
    }

    // ------------------------------------------------------------------
    //  Convenience helpers
    // ------------------------------------------------------------------

    /**
     * Increment a counter.
     */
    public static function increment(string $key, int $by = 1): int
    {
        self::init();

        if (self::$redis) {
            return (int) self::$redis->incrBy($key, $by);
        }

        $current = (int) self::get($key, 0);
        $new     = $current + $by;
        self::set($key, $new, 3600);
        return $new;
    }

    /**
     * Get or set — fetch from cache, or call the callback and cache the result.
     */
    public static function remember(string $key, int $ttl, callable $callback): mixed
    {
        $value = self::get($key);
        if ($value !== null) {
            return $value;
        }

        $value = $callback();
        self::set($key, $value, $ttl);
        return $value;
    }

    /**
     * Check if cache is using Redis.
     */
    public static function isRedis(): bool
    {
        self::init();
        return self::$redis !== null;
    }

    // ------------------------------------------------------------------
    //  Status / maintenance
    // ------------------------------------------------------------------

    /**
     * Get cache status information.
     */
    public static function getStatus(): array
    {
        self::init();

        if (self::$redis) {
            return self::getRedisStatus();
        }

        if (!self::$useFile && extension_loaded('apcu') && function_exists('apcu_cache_info')) {
            return self::getApcuStatus();
        }

        $files = glob(self::$cacheDir . self::FILE_CACHE_GLOB);
        return [
            'type'      => 'file',
            'connected' => true,
            'files'     => count($files ?: []),
            'dir'       => self::$cacheDir,
        ];
    }

    /**
     * Get Redis-specific status.
     */
    private static function getRedisStatus(): array
    {
        try {
            $info = self::$redis->info();
            return [
                'type'      => 'redis',
                'connected' => true,
                'memory'    => $info['used_memory_human'] ?? 'unknown',
                'keys'      => $info['db0']['keys'] ?? 0,
            ];
        } catch (Exception $e) {
            return [
                'type'      => 'redis',
                'connected' => false,
                'error'     => $e->getMessage(),
            ];
        }
    }

    /**
     * Get APCu-specific status.
     */
    private static function getApcuStatus(): array
    {
        $info = (array) call_user_func('apcu_cache_info', true);
        return [
            'type'      => 'apcu',
            'connected' => true,
            'memory'    => round(($info['mem_size'] ?? 0) / 1024 / 1024, 2) . 'MB',
            'keys'      => $info['num_entries'] ?? 0,
        ];
    }

    /**
     * Clear all cached data.
     */
    public static function flush(): bool
    {
        self::init();

        if (self::$redis) {
            return (bool) self::$redis->flushDB();
        }

        if (!self::$useFile && extension_loaded('apcu') && function_exists('apcu_clear_cache')) {
            return (bool) call_user_func('apcu_clear_cache');
        }

        $files = glob(self::$cacheDir . self::FILE_CACHE_GLOB);
        if ($files) {
            foreach ($files as $file) {
                if (is_file($file)) {
                    @unlink($file);
                }
            }
        }
        return true;
    }

    /**
     * Clean up expired file-cache entries.
     *
     * No-op when Redis or APCu is in use (they handle expiry internally).
     *
     * @return int Number of expired files deleted.
     */
    public static function cleanup(): int
    {
        self::init();

        if (!self::$useFile) {
            return 0;
        }

        $files = glob(self::$cacheDir . self::FILE_CACHE_GLOB);
        if (!$files) {
            return 0;
        }

        $deleted = 0;
        $now     = time();

        foreach ($files as $file) {
            if (!is_file($file)) {
                continue;
            }

            $raw = file_get_contents($file);
            if ($raw === false) {
                continue;
            }

            // @unserialize: corrupt cache files must degrade gracefully.
            $data = @unserialize($raw);

            if ((!$data || !isset($data['expires']) || $data['expires'] < $now) && @unlink($file)) {
                $deleted++;
            }
        }

        return $deleted;
    }

}
