<?php

declare(strict_types=1);

namespace NewSite\Auth;

use NewSite\Database\DatabaseManager;
use NewSite\Logging\LogService;
use NewSite\Security\CsrfService;
use NewSite\Security\IpService;

/**
 * SiteAuth — front-end user authentication (login / logout / session checks).
 *
 * Security: Sessions are rotated on login and logout via CsrfService to
 * prevent fixation. IP addresses are GDPR-masked before storage. All
 * database queries use prepared statements.
 */
final class SiteAuth
{
    public static function isLoggedIn(): bool
    {
        return !empty($_SESSION['site_user_id']);
    }

    public static function requireLogin(string $redirect = '/login'): void
    {
        if (!self::isLoggedIn()) {
            header('Location: ' . $redirect);
            exit;
        }
    }

    public static function login(int $userId, ?string $ip = null): void
    {
        $db = DatabaseManager::getWriteConnection();

        CsrfService::rotateSession();

        $_SESSION['site_user_id'] = $userId;

        $storedIp = $ip !== null ? IpService::gdprStore($ip) : null;

        $stmt = $db->prepare("UPDATE site_users SET last_login_at = ?, last_login_ip = ? WHERE id = ?");
        $stmt->execute([time(), $storedIp, $userId]);

        LogService::add('login', "User login: ID $userId", "IP: " . ($ip ?? 'unknown'));
    }

    public static function logout(): void
    {
        $userId = $_SESSION['site_user_id'] ?? null;

        if ($userId !== null) {
            LogService::add('logout', "User logout: ID $userId", null);
        }

        unset($_SESSION['site_user_id']);

        CsrfService::rotateSession();
    }
}
