<?php

declare(strict_types=1);

namespace NewSite\Auth;

use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;

/**
 * Google OAuth 2.0 server-side authorization code flow.
 *
 * Security: Uses state parameter to prevent CSRF, exchanges auth codes
 * server-side only (no JS / client-side token handling), validates tokens
 * via Google's tokeninfo endpoint. Client secret is stored in the DB
 * settings table — never exposed to the browser.
 *
 * This implementation requires no external libraries — it communicates
 * directly with Google's OAuth 2.0 endpoints via cURL.
 */
final class GoogleOAuthService
{
    private const AUTH_URL    = 'https://accounts.google.com/o/oauth2/v2/auth';
    private const TOKEN_URL   = 'https://oauth2.googleapis.com/token';
    private const USERINFO_URL = 'https://www.googleapis.com/oauth2/v3/userinfo';

    /**
     * Check whether Google OAuth is enabled and configured in admin settings.
     */
    public static function isEnabled(): bool
    {
        return SettingsService::get('google_oauth_enabled', '0') === '1'
            && self::getClientId() !== ''
            && self::getClientSecret() !== '';
    }

    /**
     * Build the Google authorization URL the user should be redirected to.
     * A random state token is stored in the session to prevent CSRF.
     */
    public static function getAuthUrl(): string
    {
        $state = bin2hex(random_bytes(16));
        $_SESSION['google_oauth_state'] = $state;

        $params = [
            'client_id'     => self::getClientId(),
            'redirect_uri'  => self::getRedirectUri(),
            'response_type' => 'code',
            'scope'         => 'openid email profile',
            'access_type'   => 'online',
            'state'         => $state,
            'prompt'        => 'select_account',
        ];

        return self::AUTH_URL . '?' . http_build_query($params);
    }

    /**
     * Handle the OAuth callback: verify state, exchange code for token,
     * fetch user info from Google.
     *
     * @return array{google_id: string, email: string, name: string}|null
     *         null on any error (logged internally)
     */
    public static function handleCallback(string $code, string $state): ?array
    {
        // CSRF check: compare state to session
        $expectedState = $_SESSION['google_oauth_state'] ?? '';
        unset($_SESSION['google_oauth_state']);

        if ($state === '' || !hash_equals($expectedState, $state)) {
            LogService::add('google_oauth', 'State mismatch — possible CSRF', null);
            return null;
        }

        // Exchange code for access token, then fetch user profile
        $accessToken = self::obtainAccessToken($code);
        return $accessToken !== null ? self::fetchUserInfo($accessToken) : null;
    }

    /**
     * Exchange an authorization code and return just the access_token string.
     */
    private static function obtainAccessToken(string $code): ?string
    {
        $tokenData = self::exchangeCode($code);
        $accessToken = $tokenData['access_token'] ?? '';

        if ($accessToken === '') {
            LogService::add('google_oauth', 'No access_token in token response', null);
            return null;
        }

        return $accessToken;
    }

    /**
     * Exchange an authorization code for tokens via Google's token endpoint.
     *
     * @return array<string, mixed>  Decoded JSON response or empty array on failure
     */
    private static function exchangeCode(string $code): array
    {
        $postFields = [
            'code'          => $code,
            'client_id'     => self::getClientId(),
            'client_secret' => self::getClientSecret(),
            'redirect_uri'  => self::getRedirectUri(),
            'grant_type'    => 'authorization_code',
        ];

        $response = self::curlPost(self::TOKEN_URL, $postFields);
        $data     = $response !== null ? json_decode($response, true) : null;

        if (!is_array($data)) {
            LogService::add('google_oauth', 'Token exchange failed or invalid JSON', null);
            return [];
        }

        if (isset($data['error'])) {
            LogService::add(
                'google_oauth',
                'Token error: ' . ($data['error'] ?? 'unknown'),
                $data['error_description'] ?? null,
            );
            return [];
        }

        return $data;
    }

    /**
     * Fetch authenticated user info from Google using an access token.
     *
     * @return array{google_id: string, email: string, name: string}|null
     */
    private static function fetchUserInfo(string $accessToken): ?array
    {
        $response = self::curlGet(self::USERINFO_URL, $accessToken);
        if ($response === null) {
            LogService::add('google_oauth', 'Userinfo cURL failed', null);
            return null;
        }

        $data = json_decode($response, true);
        if (!is_array($data) || empty($data['sub']) || empty($data['email'])) {
            LogService::add('google_oauth', 'Userinfo missing sub or email', null);
            return null;
        }

        return [
            'google_id' => (string) $data['sub'],
            'email'     => (string) $data['email'],
            'name'      => (string) ($data['name'] ?? ''),
        ];
    }

    // ── Helpers ─────────────────────────────────────────────────────

    private static function getClientId(): string
    {
        return trim((string) SettingsService::get('google_oauth_client_id', ''));
    }

    private static function getClientSecret(): string
    {
        return trim((string) SettingsService::get('google_oauth_client_secret', ''));
    }

    /**
     * Build the redirect URI pointing to the site's OAuth callback.
     * Uses HTTPS and the current host.
     */
    private static function getRedirectUri(): string
    {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
        return $scheme . '://' . $host . '/auth/google/callback';
    }

    /**
     * Perform a POST request using cURL.
     *
     * @param  array<string, string> $fields  Form fields to send
     * @return string|null  Response body or null on failure
     */
    private static function curlPost(string $url, array $fields): ?string
    {
        $ch = curl_init($url);
        if ($ch === false) {
            return null;
        }

        curl_setopt_array($ch, [
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => http_build_query($fields),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 15,
            CURLOPT_HTTPHEADER     => ['Content-Type: application/x-www-form-urlencoded'],
        ]);

        $result = curl_exec($ch);
        $httpCode = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (!is_string($result) || $httpCode < 200 || $httpCode >= 300) {
            return null;
        }

        return $result;
    }

    /**
     * Perform a GET request with a Bearer token using cURL.
     *
     * @return string|null  Response body or null on failure
     */
    private static function curlGet(string $url, string $bearerToken): ?string
    {
        $ch = curl_init($url);
        if ($ch === false) {
            return null;
        }

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 15,
            CURLOPT_HTTPHEADER     => ['Authorization: Bearer ' . $bearerToken],
        ]);

        $result = curl_exec($ch);
        $httpCode = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (!is_string($result) || $httpCode < 200 || $httpCode >= 300) {
            return null;
        }

        return $result;
    }
}
