<?php

declare(strict_types=1);

namespace NewSite\Auth;

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Security\IpService;

/**
 * AdminAuth — admin-panel authentication and IP-allowlist enforcement.
 *
 * Security: All queries use prepared statements. IP allow-listing prevents
 * access from non-approved addresses. Denied requests receive a generic 404
 * to avoid leaking the admin panel's existence.
 */
final class AdminAuth
{
    public static function isLoggedIn(): bool
    {
        return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
    }

    public static function isAllowedIp(string $ip): bool
    {
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            return false;
        }

        $db    = DatabaseManager::getWriteConnection();
        $count = (int) $db->query("SELECT COUNT(*) FROM admin_allowed_ips")->fetchColumn();

        if ($count === 0) {
            return false;
        }

        $stmt = $db->prepare("SELECT 1 FROM admin_allowed_ips WHERE ip_address = ? LIMIT 1");
        $stmt->execute([$ip]);

        return $stmt->fetchColumn() !== false;
    }

    public static function isIpAllowed(?string $ip = null): bool
    {
        $ip = $ip ?? IpService::getClientIP();

        return self::isAllowedIp($ip);
    }

    public static function addAllowedIp(string $ip, ?string $label = null): bool
    {
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            return false;
        }

        $db   = DatabaseManager::getWriteConnection();
        $now  = DbHelper::nowString();
        $stmt = $db->prepare(
            "INSERT INTO admin_allowed_ips (ip_address, label, added_at) VALUES (?, ?, ?) "
            . "ON CONFLICT (ip_address) DO UPDATE SET label = EXCLUDED.label, added_at = EXCLUDED.added_at"
        );

        return $stmt->execute([$ip, $label, $now]);
    }

    public static function removeAllowedIp(string $ip): bool
    {
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            return false;
        }

        $db   = DatabaseManager::getWriteConnection();
        $stmt = $db->prepare("DELETE FROM admin_allowed_ips WHERE ip_address = ?");

        return $stmt->execute([$ip]);
    }

    public static function getAllowedIps(): array
    {
        $db = DatabaseManager::getWriteConnection();

        return $db->query("SELECT * FROM admin_allowed_ips ORDER BY added_at DESC")->fetchAll();
    }

    public static function denyAccess(): never
    {
        http_response_code(404);
        echo '<!DOCTYPE html><html><head><title>404 Not Found</title></head>'
            . '<body><h1>Page not found</h1>'
            . '<p>The page you are looking for does not exist.</p></body></html>';
        exit;
    }

    public static function requireIp(): void
    {
        if (!self::isIpAllowed()) {
            self::denyAccess();
        }
    }

    public static function requireLogin(): void
    {
        self::requireIp();

        if (!self::isLoggedIn()) {
            header('Location: /admin/login.php');
            exit;
        }
    }
}
