<?php

declare(strict_types=1);

namespace NewSite\Admin;

use NewSite\Cache\CacheService;
use NewSite\Database\DatabaseManager;
use NewSite\Settings\SettingsService;
use NewSite\User\UserMessageService;
use NewSite\Util\AssetVersioning;

/**
 * AdminLayout — centralised admin header rendering.
 *
 * Replaces per-page include_once calls for header.php with a single
 * autoloaded class that outputs identical HTML via echo.
 *
 * Security: all dynamic output is escaped via e(). CSRF tokens and CSP
 * nonces are injected from the global helpers set up by _init.php.
 */
class AdminLayout
{
    // ── SVG icon wrapper constants (avoid string duplication) ──────

    private const SVG_OPEN      = '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">';
    private const SVG_OPEN_ARIA = '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" aria-hidden="true">';
    private const SVG_CLOSE     = '</svg>';
    private const SPAN_CLOSE    = '</span>';

    // ── Cached icon paths (populated once on first call) ──────────

    /** @var array<string, string>|null */
    private static ?array $iconPaths = null;

    // ================================================================
    //  PUBLIC API
    // ================================================================

    /**
     * Render the full admin header — DOCTYPE through opening main tag.
     *
     * Call this instead of  include_once __DIR__ . '/includes/header.php';
     */
    public static function renderHeader(): void
    {
        self::renderDocumentHead();
        $ctx = self::gatherSidebarContext();
        echo '<body class="admin-body">' . "\n";
        self::renderSidebar($ctx);
    }

    // ================================================================
    //  DOCUMENT HEAD
    // ================================================================

    private static function renderDocumentHead(): void
    {
        $csrf  = \e(\getCsrfToken());
        $css   = \e(AssetVersioning::url('/admin/assets/css/admin.css'));

        echo '<!DOCTYPE html>' . "\n"
            . '<html lang="en">' . "\n"
            . '<head>' . "\n"
            . '    <meta charset="UTF-8">' . "\n"
            . '    <meta name="viewport" content="width=device-width, initial-scale=1.0">' . "\n"
            . '    <meta name="csrf-token" content="' . $csrf . '">' . "\n"
            . '    <title>Admin Panel</title>' . "\n";

        $favicon = SettingsService::get('site_favicon', '');
        if ($favicon !== '') {
            $ext  = strtolower(pathinfo($favicon, PATHINFO_EXTENSION));
            $mime = match ($ext) {
                'svg'          => 'image/svg+xml',
                'png'          => 'image/png',
                'gif'          => 'image/gif',
                'jpg', 'jpeg'  => 'image/jpeg',
                'webp'         => 'image/webp',
                default        => 'image/x-icon',
            };
            echo '    <link rel="icon" href="' . \e($favicon) . '" type="' . $mime . '">' . "\n";
        } else {
            echo "    <link rel=\"icon\" href=\"data:image/svg+xml,"
                . "<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'>"
                . "<text y='.9em' font-size='90'>⚡</text></svg>\">" . "\n";
        }

        echo '    <link rel="stylesheet" href="' . $css . '">' . "\n"
            . '</head>' . "\n";
    }

    // ================================================================
    //  CONTEXT GATHERING
    // ================================================================

    /**
     * Collect the data the sidebar needs (forum status + pending count).
     *
     * The mini bar was removed — admins don't need currency/language/
     * notification widgets. Only sidebar-relevant context is gathered.
     *
     * @return array{forumEnabled: bool, pendingForum: int}
     */
    private static function gatherSidebarContext(): array
    {
        $forumEnabled = SettingsService::get('forum_enabled', '0') === '1';
        $pendingForum = $forumEnabled ? self::fetchPendingForumCount() : 0;

        return [
            'forumEnabled' => $forumEnabled,
            'pendingForum' => $pendingForum,
        ];
    }

    /**
     * Fetch the pending forum post count with a 60-second cache.
     */
    private static function fetchPendingForumCount(): int
    {
        $cached = CacheService::get('forum_pending_count', null);
        if (is_int($cached)) {
            return $cached;
        }

        $count = 0;
        try {
            $stmt  = DatabaseManager::getReadConnection()->query("SELECT COUNT(*) FROM forum_posts WHERE status = 'pending'");
            $count = (int) $stmt->fetchColumn();
        } catch (\Throwable) {
            // DB error — fall back to zero
        }

        CacheService::set('forum_pending_count', $count, 60);
        return $count;
    }

    // ================================================================
    //  SIDEBAR
    // ================================================================

    private static function renderSidebar(array $ctx): void
    {
        // Sidebar toggle + overlay
        echo '<div class="admin-wrapper">' . "\n"
            . '<button type="button" class="admin-sidebar-toggle" data-admin-sidebar-toggle aria-label="Toggle sidebar">'
            . self::SVG_OPEN
            . '<line x1="3" y1="6" x2="21" y2="6"></line>'
            . '<line x1="3" y1="12" x2="21" y2="12"></line>'
            . '<line x1="3" y1="18" x2="21" y2="18"></line>'
            . self::SVG_CLOSE . '</button>' . "\n"
            . '<div class="admin-sidebar-overlay" data-admin-sidebar-overlay></div>' . "\n";

        // Sidebar aside
        echo '<aside class="admin-sidebar">' . "\n"
            . '<div class="sidebar-header">'
            . '<a href="/admin/" class="admin-logo">'
            . '<span class="logo-icon">⚡' . self::SPAN_CLOSE
            . '<span class="logo-text">Admin' . self::SPAN_CLOSE
            . '</a></div>' . "\n";

        // Navigation
        echo '<nav class="sidebar-nav"><ul>' . "\n";
        $navItems  = self::buildSidebarNav($ctx['forumEnabled'], $ctx['pendingForum']);
        $iconPaths = self::getNavIconPaths();
        foreach ($navItems as $item) {
            self::renderNavItem($item, $iconPaths);
        }
        echo '</ul></nav>' . "\n";

        // Sidebar footer
        echo '<div class="sidebar-footer">'
            . '<a href="/" target="_blank" class="view-site"><span>View Site' . self::SPAN_CLOSE
            . '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">'
            . '<path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path>'
            . '<polyline points="15 3 21 3 21 9"></polyline>'
            . '<line x1="10" y1="14" x2="21" y2="3"></line>'
            . self::SVG_CLOSE . '</a>'
            . '<a href="/admin/logout.php" class="logout-link">Logout</a>'
            . '</div>' . "\n"; // sidebar-footer

        echo '</aside>' . "\n"
            . '<main class="admin-main">' . "\n";
    }

    // ================================================================
    //  NAV ITEMS
    // ================================================================

    /**
     * Build the ordered list of sidebar navigation items.
     *
     * @return list<array{file: string, href: string, label: string,
     *                     icon: string, badge?: bool, linkClass?: string,
     *                     ariaIcon?: bool}>
     */
    private static function buildSidebarNav(bool $forumEnabled, int $pendingForum): array
    {
        $unreadTickets = UserMessageService::getAdminUnreadTicketCount();

        // Count pending refund requests for the badge indicator
        $pendingRefunds = 0;
        try {
            $pendingRefunds = (int) DatabaseManager::getReadConnection()
                ->query("SELECT COUNT(*) FROM refund_requests WHERE status = 'pending'")
                ->fetchColumn();
        } catch (\PDOException) {
            // Table may not exist yet during migration
        }

        return [
            ['file' => 'index.php',          'href' => '/admin/',                  'label' => 'Dashboard',       'icon' => 'home'],
            ['file' => 'pages.php',          'href' => '/admin/pages.php',         'label' => 'Pages',           'icon' => 'page'],
            ['file' => 'legal-pages.php',    'href' => '/admin/legal-pages.php',   'label' => 'Legal Pages',     'icon' => 'legal'],
            ['file' => 'products.php',       'href' => '/admin/products.php',      'label' => 'Products',        'icon' => 'grid'],
            ['file' => 'attributes.php',     'href' => '/admin/attributes.php',    'label' => 'Attributes',      'icon' => 'tag'],
            ['file' => 'purchases.php',      'href' => '/admin/purchases.php',     'label' => 'Purchases',       'icon' => 'receipt'],
            ['file' => 'refunds.php',        'href' => '/admin/refunds.php',       'label' => 'Refunds',         'icon' => 'refund',
             'badge' => $pendingRefunds > 0, 'linkClass' => 'nav-link-with-indicator'],
            ['file' => 'collections.php',    'href' => '/admin/collections.php',   'label' => 'Collections',     'icon' => 'list'],
            ['file' => 'forum.php',          'href' => '/admin/forum.php',         'label' => 'Forum',           'icon' => 'chat',
             'badge' => $forumEnabled && $pendingForum > 0, 'linkClass' => 'nav-link-with-indicator'],
            ['file' => 'menus.php',          'href' => '/admin/menus.php',         'label' => 'Menus',           'icon' => 'menu'],
            ['file' => 'files.php',          'href' => '/admin/files.php',         'label' => 'Files',           'icon' => 'folder'],
            ['file' => 'theme-editor.php',   'href' => '/admin/theme-editor.php',  'label' => 'Theme Editor',    'icon' => 'gear'],
            ['file' => 'translations.php',   'href' => '/admin/translations.php',  'label' => 'Translations',    'icon' => 'translate'],
            ['file' => 'settings.php',       'href' => '/admin/settings.php',      'label' => 'Settings',        'icon' => 'info-circle'],
            ['file' => 'email-templates.php','href' => '/admin/email-templates.php','label' => 'Email Templates', 'icon' => 'email'],
            ['file' => 'logs.php',           'href' => '/admin/logs.php',          'label' => 'Logs',            'icon' => 'log'],
            ['file' => 'security-scan.php',  'href' => '/admin/security-scan.php', 'label' => 'Security Scan',   'icon' => 'shield'],
            ['file' => 'vendor-libs.php',    'href' => '/admin/vendor-libs.php',   'label' => 'Vendor Libs',     'icon' => 'diamond', 'ariaIcon' => true],
            ['file' => 'mini-games.php',     'href' => '/admin/mini-games.php',    'label' => 'Mini-Games',      'icon' => 'layers',  'ariaIcon' => true],
            ['file' => 'statistics.php',     'href' => '/admin/statistics.php',    'label' => 'Statistics',       'icon' => 'bar-chart'],
            ['file' => 'messages.php',       'href' => '/admin/messages.php',      'label' => 'User Messages',   'icon' => 'bubble'],
            ['file' => 'tickets.php',        'href' => '/admin/tickets.php',       'label' => 'Tickets',         'icon' => 'ticket',
             'badge' => $unreadTickets > 0, 'linkClass' => 'nav-link-with-indicator'],
            ['file' => 'dmca.php',           'href' => '/admin/dmca.php',          'label' => 'DMCA Requests',   'icon' => 'clipboard'],
            ['file' => 'site-files.php',     'href' => '/admin/site-files.php',    'label' => 'Site Files',      'icon' => 'file-text'],
            ['file' => 'currency-rates.php', 'href' => '/admin/currency-rates.php','label' => 'Currency Rates',  'icon' => 'currency'],
            ['file' => 'backup.php',         'href' => '/admin/backup.php',        'label' => 'Backups',         'icon' => 'download'],
            ['file' => 'info.php',           'href' => '/admin/info.php',          'label' => 'Info',            'icon' => 'info'],
        ];
    }

    /**
     * Render a single sidebar <li> navigation item.
     *
     * @param array<string, string> $iconPaths icon-key → SVG inner paths
     */
    private static function renderNavItem(array $item, array $iconPaths): void
    {
        $activeAttr = self::isActivePage($item['file']) ? ' class="active"' : '';
        $linkClass  = isset($item['linkClass']) ? ' class="' . \e($item['linkClass']) . '"' : '';
        $svgOpen    = !empty($item['ariaIcon']) ? self::SVG_OPEN_ARIA : self::SVG_OPEN;
        $paths      = $iconPaths[$item['icon']] ?? '';

        echo '<li' . $activeAttr . '>'
            . '<a href="' . \e($item['href']) . '"' . $linkClass . '>'
            . '<span class="nav-icon">' . $svgOpen . $paths . self::SVG_CLOSE . self::SPAN_CLOSE
            . '<span class="nav-text">' . \e($item['label']) . self::SPAN_CLOSE;

        if (!empty($item['badge'])) {
            echo '<span class="nav-unread-indicator">' . self::SPAN_CLOSE;
        }

        echo '</a></li>' . "\n";
    }

    /**
     * Check whether the current request matches the given admin filename.
     *
     * Uses REQUEST_URI (preserved by mod_rewrite) so active-page detection
     * works correctly when requests are routed through the admin front controller.
     */
    private static function isActivePage(string $filename): bool
    {
        $path = parse_url((string) ($_SERVER['REQUEST_URI'] ?? ''), PHP_URL_PATH) ?? '';
        $base = basename($path);

        // Bare /admin/ resolves to the dashboard (index.php)
        if ($filename === 'index.php' && ($base === 'admin' || $base === '')) {
            return true;
        }

        return $base === $filename;
    }

    // ================================================================
    //  ICON PATHS (inner SVG elements without wrapper)
    // ================================================================

    /**
     * @return array<string, string> icon-key → SVG inner markup
     */
    private static function getNavIconPaths(): array
    {
        if (self::$iconPaths !== null) {
            return self::$iconPaths;
        }

        self::$iconPaths = [
            'home'        => '<path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"></path>'
                           . '<polyline points="9 22 9 12 15 12 15 22"></polyline>',
            'page'        => '<path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>'
                           . '<polyline points="14 2 14 8 20 8"></polyline>',
            'legal'       => '<path d="M4 4h16v16H4z"></path><path d="M8 8h8"></path>'
                           . '<path d="M8 12h6"></path><path d="M8 16h4"></path>',
            'grid'        => '<rect x="3" y="3" width="7" height="7"></rect>'
                           . '<rect x="14" y="3" width="7" height="7"></rect>'
                           . '<rect x="3" y="14" width="7" height="7"></rect>'
                           . '<rect x="14" y="14" width="7" height="7"></rect>',
            'tag'         => '<path d="M20.59 13.41l-7.17 7.17a2 2 0 0 1-2.83 0L2 12V2h10l8.59 8.59a2 2 0 0 1 0 2.82z"></path>'
                           . '<line x1="7" y1="7" x2="7.01" y2="7"></line>',
            'receipt'     => '<path d="M6 2h12a2 2 0 0 1 2 2v16a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2z"></path>'
                           . '<path d="M8 6h8"></path><path d="M8 10h8"></path><path d="M8 14h5"></path>',
            'list'        => '<path d="M3 6h18"></path><path d="M3 12h18"></path><path d="M3 18h18"></path>',
            'chat'        => '<path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>'
                           . '<path d="M7 9h10"></path><path d="M7 13h6"></path>',
            'menu'        => '<line x1="3" y1="12" x2="21" y2="12"></line>'
                           . '<line x1="3" y1="6" x2="21" y2="6"></line>'
                           . '<line x1="3" y1="18" x2="21" y2="18"></line>',
            'folder'      => '<path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"></path>',
            'gear'        => '<circle cx="12" cy="12" r="3"></circle>'
                           . '<path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06'
                           . 'a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09'
                           . 'A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83'
                           . 'l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09'
                           . 'A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0'
                           . 'l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09'
                           . 'a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83'
                           . 'l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09'
                           . 'a1.65 1.65 0 0 0-1.51 1z"></path>',
            'translate'   => '<path d="M4 5h7"></path><path d="M4 9h7"></path><path d="M4 13h4"></path>'
                           . '<path d="M14 6h6"></path><path d="M14 10h6"></path><path d="M14 14h6"></path>',
            'info-circle' => '<circle cx="12" cy="12" r="10"></circle>'
                           . '<line x1="12" y1="8" x2="12" y2="12"></line>'
                           . '<line x1="12" y1="16" x2="12.01" y2="16"></line>',
            'email'       => '<path d="M4 4h16v16H4z"></path><path d="M4 7l8 5 8-5"></path>',
            'log'         => '<path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>'
                           . '<polyline points="14 2 14 8 20 8"></polyline>'
                           . '<line x1="16" y1="13" x2="8" y2="13"></line>'
                           . '<line x1="16" y1="17" x2="8" y2="17"></line>'
                           . '<polyline points="10 9 9 9 8 9"></polyline>',
            'shield'      => '<path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"></path>'
                           . '<path d="M9 12l2 2 4-4"></path>',
            'diamond'     => '<path d="M12 2l4 4-4 4-4-4 4-4z"></path>'
                           . '<path d="M2 12l4-4 4 4-4 4-4-4z"></path>'
                           . '<path d="M22 12l-4-4-4 4 4 4 4-4z"></path>'
                           . '<path d="M12 22l-4-4 4-4 4 4-4 4z"></path>',
            'layers'      => '<path d="M12 2l9 5-9 5-9-5 9-5z"></path>'
                           . '<path d="M3 12l9 5 9-5"></path>'
                           . '<path d="M3 17l9 5 9-5"></path>',
            'bar-chart'   => '<path d="M18 20V10"></path><path d="M12 20V4"></path><path d="M6 20v-6"></path>',
            'bubble'      => '<path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>',
            'ticket'      => '<path d="M21 10V6a2 2 0 0 0-2-2H5a2 2 0 0 0-2 2v4"></path>'
                           . '<path d="M3 10h18"></path>'
                           . '<path d="M7 15h10"></path><path d="M7 19h6"></path>',
            'clipboard'   => '<path d="M4 4h16v16H4z"></path><path d="M8 9h8"></path>'
                           . '<path d="M8 13h6"></path><path d="M8 17h4"></path>',
            'file-text'   => '<path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>'
                           . '<polyline points="14 2 14 8 20 8"></polyline>'
                           . '<line x1="8" y1="13" x2="16" y2="13"></line>'
                           . '<line x1="8" y1="17" x2="16" y2="17"></line>',
            'currency'    => '<circle cx="12" cy="12" r="9"></circle>'
                           . '<path d="M8 12h8"></path><path d="M12 8v8"></path>',
            'download'    => '<path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>'
                           . '<polyline points="7 10 12 15 17 10"></polyline>'
                           . '<line x1="12" y1="15" x2="12" y2="3"></line>',
            'info'        => '<circle cx="12" cy="12" r="10"></circle>'
                           . '<path d="M12 16v-4"></path><path d="M12 8h.01"></path>',
            'refund'      => '<polyline points="1 4 1 10 7 10"></polyline>'
                           . '<path d="M3.51 15a9 9 0 1 0 2.13-9.36L1 10"></path>',
        ];

        return self::$iconPaths;
    }

    // ================================================================
    //  FOOTER + MEDIA PICKER
    // ================================================================

    /**
     * Render the admin footer — media picker modal, scripts, closing tags.
     *
     * Call this instead of  include_once __DIR__ . '/includes/footer.php';
     */
    public static function renderFooter(): void
    {
        echo "        </main>\n    </div>\n";
        $pickerData = self::gatherMediaPickerData();
        self::renderMediaPickerModal($pickerData);
        $nonce = \e(\getCspNonce());
        $mpSrc = \e(AssetVersioning::url('/admin/assets/js/media-picker.js'));
        $adSrc = \e(AssetVersioning::url('/admin/assets/js/admin.js'));
        echo '    <script nonce="' . $nonce . '" src="' . $mpSrc . '"></script>' . "\n";
        echo '    <script nonce="' . $nonce . '" src="' . $adSrc . '"></script>' . "\n";
        echo "</body>\n</html>\n";
    }

    /**
     * Query media files and thumbnail setting for the media picker.
     *
     * @return array{pickerFiles: list<array<string,mixed>>, imageFiles: list<array<string,mixed>>, videoFiles: list<array<string,mixed>>, videoThumbnailMode: string}
     */
    private static function gatherMediaPickerData(): array
    {
        $pickerFiles = DatabaseManager::getWriteConnection()->query(
            "SELECT * FROM files ORDER BY uploaded_at DESC"
        )->fetchAll();

        $imageFiles = array_values(array_filter(
            $pickerFiles,
            static fn (array $f): bool => str_starts_with($f['file_type'], 'image/')
        ));

        $videoFiles = array_values(array_filter(
            $pickerFiles,
            static fn (array $f): bool => str_starts_with($f['file_type'], 'video/')
        ));

        $mode = SettingsService::get('video_thumbnail_mode', 'auto');
        $allowed = ['auto', 'client', 'server', 'off'];
        if (!in_array($mode, $allowed, true)) {
            $mode = 'auto';
        }

        return [
            'pickerFiles'        => array_values($pickerFiles),
            'imageFiles'         => $imageFiles,
            'videoFiles'         => $videoFiles,
            'videoThumbnailMode' => $mode,
        ];
    }

    /**
     * Output the media picker modal HTML with embedded data attributes.
     *
     * Data is passed to the external media-picker.js via data-* attributes,
     * avoiding any inline script blocks. This is fully CSP-safe.
     *
     * @param array{pickerFiles: list<array<string,mixed>>, imageFiles: list<array<string,mixed>>, videoFiles: list<array<string,mixed>>, videoThumbnailMode: string} $data
     */
    private static function renderMediaPickerModal(array $data): void
    {
        $libraryJson = json_encode([
            'images' => $data['imageFiles'],
            'videos' => $data['videoFiles'],
            'files'  => $data['pickerFiles'],
        ], JSON_THROW_ON_ERROR | JSON_UNESCAPED_SLASHES);

        $mode    = \e($data['videoThumbnailMode']);
        $library = \e($libraryJson);

        echo <<<HTML
        <!-- Media Picker Modal -->
        <div id="media-picker-modal" class="modal is-hidden"
             data-video-thumbnail-mode="{$mode}"
             data-media-library="{$library}">
            <div class="modal-backdrop" data-action="close-media-picker"></div>
            <div class="modal-content modal-lg">
                <div class="modal-header">
                    <h3 id="media-picker-title">Select Media</h3>
                    <button type="button" class="modal-close" data-action="close-media-picker">&times;</button>
                </div>
                <div class="modal-body">
                    <div class="media-picker-tabs">
                        <button type="button" class="tab-btn active" data-tab="upload">&#x1F4E4; Upload New</button>
                        <button type="button" class="tab-btn" data-tab="library">&#x1F4C1; Media Library</button>
                        <button type="button" class="tab-btn" data-tab="url">&#x1F517; External URL</button>
                        <button type="button" class="tab-btn tab-video-only is-hidden" data-tab="youtube">&#x25B6;&#xFE0F; YouTube</button>
                    </div>
                    <div id="media-tab-upload" class="media-tab-content active">
                        <div class="upload-zone" id="media-upload-zone">
                            <div class="upload-zone-content">
                                <div class="upload-icon">&#x1F4C1;</div>
                                <p>Drag &amp; drop files here or click to browse</p>
                                <div class="file-input-button">
                                    <input type="file" id="media-picker-file" accept="image/*,video/*" multiple>
                                    <span class="btn btn-primary" aria-hidden="true">Choose File</span>
                                </div>
                            </div>
                            <div class="upload-progress is-hidden" id="media-upload-progress">
                                <progress class="progress-bar" id="media-progress-fill" value="0" max="100"></progress>
                                <p id="media-upload-status">Uploading...</p>
                            </div>
                        </div>
                    </div>
                    <div id="media-tab-library" class="media-tab-content">
                        <div class="media-library-grid" id="media-library-grid"></div>
                        <p class="text-muted text-center empty-state is-hidden" id="media-library-empty">No media files found. Upload some first!</p>
                    </div>
                    <div id="media-picker-actions" class="media-picker-actions is-hidden">
                        <span class="media-picker-selected-count" id="media-picker-selected-count">0 selected</span>
                        <div class="media-picker-action-buttons">
                            <button type="button" class="btn btn-secondary" data-action="clear-media-selection">Clear</button>
                            <button type="button" class="btn btn-primary" data-action="confirm-media-selection">Add Selected</button>
                        </div>
                    </div>
                    <div id="media-tab-url" class="media-tab-content">
                        <div class="form-group">
                            <label id="media-url-label">Enter Image/Video URL</label>
                            <input type="url" id="media-external-url" class="form-control" placeholder="https://example.com/image.jpg">
                        </div>
                        <div id="media-url-preview" class="media-url-preview is-hidden"></div>
                        <button type="button" class="btn btn-primary" data-action="select-external-url">Use This URL</button>
                    </div>
                    <div id="media-tab-youtube" class="media-tab-content">
                        <div class="form-group">
                            <label>YouTube Video URL or ID</label>
                            <input type="text" id="media-youtube-url" class="form-control" placeholder="https://www.youtube.com/watch?v=VIDEO_ID">
                            <small class="text-muted">Paste a YouTube URL or video ID</small>
                        </div>
                        <div id="youtube-preview" class="youtube-preview is-hidden">
                            <iframe id="youtube-preview-frame" width="100%" height="250" frameborder="0" allowfullscreen></iframe>
                        </div>
                        <button type="button" class="btn btn-primary" data-action="select-youtube-video">Use This Video</button>
                    </div>
                </div>
            </div>
        </div>

        HTML;
    }
}
