<?php
$toolFlashKey = 'public_text_tool_flash_v1';

$toolMessage = '';
$toolError = '';

$toolSearchText = '';
$toolReplaceText = '';
$toolInputText = '';
$toolCaseSensitive = false;

$toolCountPreview = null;

$toolOutputText = null;
$toolOutputCount = 0;
$toolFileRows = [];

$toolMaxPasteBytes = 2 * 1024 * 1024; // 2 MB
$toolMaxSingleFileBytes = 1024 * 1024; // 1 MB per text file
$toolMaxArchiveBytes = 10 * 1024 * 1024; // 10 MB per archive upload
$toolMaxUploadsPerRun = 15;
$toolMaxProcessedEntries = 120;
$toolMaxRenderedOutputBytes = 2 * 1024 * 1024;

if (!function_exists('toolNormalizeUploadedFiles')) {
    function toolNormalizeUploadedFiles(array $files): array {
        $normalized = [];
        if (!isset($files['name'])) {
            return $normalized;
        }

        if (is_array($files['name'])) {
            foreach ($files['name'] as $index => $name) {
                $normalized[] = [
                    'name' => (string)($name ?? ''),
                    'type' => (string)($files['type'][$index] ?? ''),
                    'tmp_name' => (string)($files['tmp_name'][$index] ?? ''),
                    'error' => (int)($files['error'][$index] ?? UPLOAD_ERR_NO_FILE),
                    'size' => (int)($files['size'][$index] ?? 0),
                ];
            }

            return $normalized;
        }

        $normalized[] = [
            'name' => (string)($files['name'] ?? ''),
            'type' => (string)($files['type'] ?? ''),
            'tmp_name' => (string)($files['tmp_name'] ?? ''),
            'error' => (int)($files['error'] ?? UPLOAD_ERR_NO_FILE),
            'size' => (int)($files['size'] ?? 0),
        ];

        return $normalized;
    }
}

if (!function_exists('toolSanitizeArchiveEntryPath')) {
    function toolSanitizeArchiveEntryPath(string $entryPath): string {
        $normalized = str_replace('\\', '/', trim($entryPath));
        $normalized = ltrim($normalized, '/');

        $isValid = true;
        if ($normalized === '' || str_ends_with($normalized, '/')) {
            $isValid = false;
        }
        if ($isValid && (strpos($normalized, '../') !== false || strpos($normalized, '/..') !== false || str_starts_with($normalized, '..'))) {
            $isValid = false;
        }

        return $isValid ? $normalized : '';
    }
}

if (!function_exists('toolApplyReplace')) {
    function toolApplyReplace(string $content, string $needle, string $replacement, bool $caseSensitive, int &$count): string {
        $count = 0;
        if ($needle === '') {
            return $content;
        }

        if ($caseSensitive) {
            $count = substr_count($content, $needle);
            return str_replace($needle, $replacement, $content);
        }

        $pattern = '/' . preg_quote($needle, '/') . '/iu';
        $result = preg_replace_callback(
            $pattern,
            static function () use ($replacement) {
                return $replacement;
            },
            $content,
            -1,
            $count
        );

        return is_string($result) ? $result : $content;
    }
}

if (!function_exists('toolCountMatches')) {
    function toolCountMatches(string $content, string $needle, bool $caseSensitive): int {
        if ($needle === '') {
            return 0;
        }

        if ($caseSensitive) {
            return substr_count($content, $needle);
        }

        $pattern = '/' . preg_quote($needle, '/') . '/iu';
        $matchResult = preg_match_all($pattern, $content, $matches);

        return is_int($matchResult) && $matchResult > 0 ? $matchResult : 0;
    }
}

if (!function_exists('toolSafeDownloadName')) {
    function toolSafeDownloadName(string $name): string {
        $clean = preg_replace('/[^a-zA-Z0-9._-]/', '_', $name);
        $clean = trim((string)$clean, '._-');
        if ($clean === '') {
            $clean = 'result.txt';
        }
        if (!str_ends_with(strtolower($clean), '.txt')) {
            $clean .= '.txt';
        }
        return $clean;
    }
}

if (!function_exists('toolConsumeRateLimit')) {
    function toolConsumeRateLimit(string &$error): bool {
        $sessionKey = 'public_text_tool_runs';
        $now = time();
        $runs = $_SESSION[$sessionKey] ?? [];
        if (!is_array($runs)) {
            $runs = [];
        }

        $runs = array_values(array_filter($runs, static function ($ts) use ($now): bool {
            return is_int($ts) && ($now - $ts) <= 60;
        }));

        $lastRun = empty($runs) ? 0 : (int)end($runs);
        if ($lastRun > 0 && ($now - $lastRun) < 2) {
            $error = 'Please wait a moment before running another replacement.';
            return false;
        }

        if (count($runs) >= 20) {
            $error = 'Rate limit reached. Please wait one minute and try again.';
            return false;
        }

        $runs[] = $now;
        $_SESSION[$sessionKey] = $runs;
        return true;
    }
}

if (!function_exists('toolStoreFlashState')) {
    function toolStoreFlashState(string $key, array $state): void {
        $_SESSION[$key] = $state;
    }
}

if (!function_exists('toolPullFlashState')) {
    function toolPullFlashState(string $key): array {
        $state = $_SESSION[$key] ?? [];
        unset($_SESSION[$key]);

        return is_array($state) ? $state : [];
    }
}

if (!function_exists('toolBuildRedirectTarget')) {
    function toolBuildRedirectTarget(string $fallback): string {
        $requestUri = (string)($_SERVER['REQUEST_URI'] ?? '');
        if ($requestUri === '') {
            return $fallback;
        }

        $hashPos = strpos($requestUri, '#');
        $candidate = $hashPos === false ? $requestUri : substr($requestUri, 0, $hashPos);
        if (!is_string($candidate) || $candidate === '' || !str_starts_with($candidate, '/')) {
            return $fallback;
        }

        return $candidate;
    }
}

$toolFlashState = toolPullFlashState($toolFlashKey);
if (!empty($toolFlashState)) {
    $toolMessage = (string)($toolFlashState['toolMessage'] ?? '');
    $toolError = (string)($toolFlashState['toolError'] ?? '');
    $toolSearchText = (string)($toolFlashState['toolSearchText'] ?? '');
    $toolReplaceText = (string)($toolFlashState['toolReplaceText'] ?? '');
    $toolInputText = (string)($toolFlashState['toolInputText'] ?? '');
    $toolCaseSensitive = (bool)($toolFlashState['toolCaseSensitive'] ?? false);

    $toolCountPreviewRaw = $toolFlashState['toolCountPreview'] ?? null;
    $toolCountPreview = is_int($toolCountPreviewRaw) ? $toolCountPreviewRaw : null;

    $toolOutputTextRaw = $toolFlashState['toolOutputText'] ?? null;
    $toolOutputText = is_string($toolOutputTextRaw) ? $toolOutputTextRaw : null;

    $toolOutputCount = (int)($toolFlashState['toolOutputCount'] ?? 0);

    $fileRows = $toolFlashState['toolFileRows'] ?? [];
    $toolFileRows = is_array($fileRows) ? $fileRows : [];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = (string)($_POST['tool_action'] ?? '');
    $toolSearchText = (string)($_POST['tool_search_text'] ?? '');
    $toolReplaceText = (string)($_POST['tool_replace_text'] ?? '');
    $toolInputText = (string)($_POST['tool_input_text'] ?? '');
    $toolCaseSensitive = isset($_POST['tool_case_sensitive']);

    $toolMessage = '';
    $toolError = '';
    $toolCountPreview = null;
    $toolOutputText = null;
    $toolOutputCount = 0;
    $toolFileRows = [];

    if ($toolSearchText === '') {
        $toolError = 'Find text is required.';
    } elseif (!toolConsumeRateLimit($toolError)) {
        // Rate limit error already set by helper.
    } elseif ($action === 'count_text') {
        if (strlen($toolInputText) > $toolMaxPasteBytes) {
            $toolError = 'Input text is too large (max 2 MB).';
        } else {
            $toolCountPreview = toolCountMatches($toolInputText, $toolSearchText, $toolCaseSensitive);
            $toolMessage = 'Current matches found: ' . $toolCountPreview . '.';
        }
    } elseif ($action === 'replace_text') {
        if (strlen($toolInputText) > $toolMaxPasteBytes) {
            $toolError = 'Input text is too large (max 2 MB).';
        } else {
            $toolOutputText = toolApplyReplace($toolInputText, $toolSearchText, $toolReplaceText, $toolCaseSensitive, $toolOutputCount);
            $toolMessage = 'Done. Replacements made: ' . $toolOutputCount . '.';
        }
    } elseif ($action === 'replace_files') {
        $uploads = toolNormalizeUploadedFiles($_FILES['tool_source_files'] ?? []);
        $uploads = array_values(array_filter($uploads, static function (array $file): bool {
            return (int)($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE;
        }));

        if (empty($uploads)) {
            $toolError = 'Please upload .txt files or .zip archives containing .txt files.';
        } elseif (count($uploads) > $toolMaxUploadsPerRun) {
            $toolError = 'Too many files. Maximum uploads per run: ' . $toolMaxUploadsPerRun . '.';
        } else {
            $processedEntries = 0;
            $skippedEntries = 0;
            $totalReplacementCount = 0;
            $totalRenderedBytes = 0;

            foreach ($uploads as $upload) {
                $uploadName = basename((string)($upload['name'] ?? ''));
                $uploadName = $uploadName !== '' ? $uploadName : 'upload.txt';
                $uploadError = (int)($upload['error'] ?? UPLOAD_ERR_NO_FILE);
                $uploadSize = (int)($upload['size'] ?? 0);
                $tmpName = (string)($upload['tmp_name'] ?? '');

                if ($uploadError !== UPLOAD_ERR_OK || $tmpName === '' || !is_uploaded_file($tmpName)) {
                    $toolFileRows[] = [
                        'source' => $uploadName,
                        'name' => $uploadName,
                        'replacements' => 0,
                        'note' => 'Upload failed.',
                        'output' => ''
                    ];
                    $skippedEntries++;
                    continue;
                }

                $ext = strtolower((string)pathinfo($uploadName, PATHINFO_EXTENSION));
                if ($ext === 'txt') {
                    if ($uploadSize > $toolMaxSingleFileBytes) {
                        $toolFileRows[] = [
                            'source' => $uploadName,
                            'name' => $uploadName,
                            'replacements' => 0,
                            'note' => 'Skipped: file larger than 1 MB.',
                            'output' => ''
                        ];
                        $skippedEntries++;
                        continue;
                    }

                    $content = @file_get_contents($tmpName);
                    if (!is_string($content)) {
                        $toolFileRows[] = [
                            'source' => $uploadName,
                            'name' => $uploadName,
                            'replacements' => 0,
                            'note' => 'Skipped: could not read file.',
                            'output' => ''
                        ];
                        $skippedEntries++;
                        continue;
                    }

                    $replaceCount = 0;
                    $output = toolApplyReplace($content, $toolSearchText, $toolReplaceText, $toolCaseSensitive, $replaceCount);
                    $outputBytes = strlen($output);

                    if (($totalRenderedBytes + $outputBytes) > $toolMaxRenderedOutputBytes) {
                        $output = '';
                        $note = 'Processed, but output hidden because total output limit was reached.';
                    } else {
                        $totalRenderedBytes += $outputBytes;
                        $note = 'Processed.';
                    }

                    $toolFileRows[] = [
                        'source' => $uploadName,
                        'name' => toolSafeDownloadName($uploadName),
                        'replacements' => $replaceCount,
                        'note' => $note,
                        'output' => $output
                    ];

                    $processedEntries++;
                    $totalReplacementCount += $replaceCount;
                    if ($processedEntries >= $toolMaxProcessedEntries) {
                        break;
                    }
                    continue;
                }

                if ($ext !== 'zip') {
                    $toolFileRows[] = [
                        'source' => $uploadName,
                        'name' => $uploadName,
                        'replacements' => 0,
                        'note' => 'Skipped: unsupported file type.',
                        'output' => ''
                    ];
                    $skippedEntries++;
                    continue;
                }

                if ($uploadSize > $toolMaxArchiveBytes) {
                    $toolFileRows[] = [
                        'source' => $uploadName,
                        'name' => $uploadName,
                        'replacements' => 0,
                        'note' => 'Skipped archive: larger than 10 MB.',
                        'output' => ''
                    ];
                    $skippedEntries++;
                    continue;
                }

                if (!class_exists('ZipArchive')) {
                    $toolError = 'Zip archives are not supported on this server right now. Please upload plain .txt files.';
                    break;
                }

                $zip = new ZipArchive();
                $openResult = $zip->open($tmpName);
                if ($openResult !== true) {
                    $toolFileRows[] = [
                        'source' => $uploadName,
                        'name' => $uploadName,
                        'replacements' => 0,
                        'note' => 'Skipped archive: cannot open.',
                        'output' => ''
                    ];
                    $skippedEntries++;
                    continue;
                }

                $zipPrefix = pathinfo($uploadName, PATHINFO_FILENAME);
                $zipPrefix = trim((string)$zipPrefix) !== '' ? trim((string)$zipPrefix) : 'archive';

                for ($i = 0; $i < $zip->numFiles; $i++) {
                    if ($processedEntries >= $toolMaxProcessedEntries) {
                        break;
                    }

                    $entryNameRaw = (string)$zip->getNameIndex($i);
                    $entryName = toolSanitizeArchiveEntryPath($entryNameRaw);
                    $entryExt = strtolower((string)pathinfo($entryName, PATHINFO_EXTENSION));
                    if ($entryName === '' || $entryExt !== 'txt') {
                        $skippedEntries++;
                        continue;
                    }

                    $entryStat = $zip->statIndex($i);
                    $entrySize = (int)($entryStat['size'] ?? 0);
                    if ($entrySize > $toolMaxSingleFileBytes) {
                        $skippedEntries++;
                        continue;
                    }

                    $entryContent = $zip->getFromIndex($i);
                    if (!is_string($entryContent)) {
                        $skippedEntries++;
                        continue;
                    }

                    $replaceCount = 0;
                    $entryOutput = toolApplyReplace($entryContent, $toolSearchText, $toolReplaceText, $toolCaseSensitive, $replaceCount);
                    $outputBytes = strlen($entryOutput);
                    $renderedOutput = $entryOutput;
                    $entryNote = 'Processed from archive.';

                    if (($totalRenderedBytes + $outputBytes) > $toolMaxRenderedOutputBytes) {
                        $renderedOutput = '';
                        $entryNote = 'Processed from archive, but output hidden because total output limit was reached.';
                    } else {
                        $totalRenderedBytes += $outputBytes;
                    }

                    $toolFileRows[] = [
                        'source' => $uploadName,
                        'name' => toolSafeDownloadName($zipPrefix . '-' . str_replace('/', '-', $entryName)),
                        'replacements' => $replaceCount,
                        'note' => $entryNote,
                        'output' => $renderedOutput
                    ];

                    $processedEntries++;
                    $totalReplacementCount += $replaceCount;
                }

                $zip->close();
            }

            if ($toolError === '') {
                if ($processedEntries > 0) {
                    $toolMessage = 'Done. Processed text files: ' . $processedEntries
                        . '. Total replacements: ' . $totalReplacementCount
                        . '. Skipped entries: ' . $skippedEntries . '.';
                } else {
                    $toolError = 'No valid .txt content could be processed from your uploads.';
                }
            }
        }
    } else {
        $toolError = 'Invalid action.';
    }

    toolStoreFlashState($toolFlashKey, [
        'toolMessage' => $toolMessage,
        'toolError' => $toolError,
        'toolSearchText' => $toolSearchText,
        'toolReplaceText' => $toolReplaceText,
        'toolInputText' => $toolInputText,
        'toolCaseSensitive' => $toolCaseSensitive,
        'toolCountPreview' => $toolCountPreview,
        'toolOutputText' => $toolOutputText,
        'toolOutputCount' => $toolOutputCount,
        'toolFileRows' => $toolFileRows,
    ]);

    $toolRedirectTarget = toolBuildRedirectTarget('/text-finder-replacer');
    if (!headers_sent()) {
        header('Location: ' . $toolRedirectTarget, true, 303);
        exit;
    }
}
?>

<section class="section-text-box">
    <div class="container">
        <div class="text-box-card">
            <h1>Text Finder &amp; Replacer</h1>
            <p class="text-box-content text-tool-intro">Find and replace text in pasted content, text files, or zip archives with text files.</p>

            <?php if ($toolMessage !== ''): ?>
            <div class="contact-alert contact-alert-success"><?php echo e($toolMessage); ?></div>
            <?php endif; ?>

            <?php if ($toolError !== ''): ?>
            <div class="contact-alert contact-alert-error"><?php echo e($toolError); ?></div>
            <?php endif; ?>

            <div class="text-tool-stack">
            <div class="text-tool-panel">
                <div class="text-tool-panel-header">
                    <h3>Replace in Pasted Text</h3>
                </div>
                <div class="text-tool-panel-body">
                    <form method="POST" action="/<?php echo e($pageSlug ?? 'text-finder-replacer'); ?>" class="contact-form">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">

                        <div class="contact-grid">
                            <div class="form-group">
                                <label for="tool-search-text">Find</label>
                                <input id="tool-search-text" type="text" name="tool_search_text" class="form-control" value="<?php echo e($toolSearchText); ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="tool-replace-text">Replace with</label>
                                <input id="tool-replace-text" type="text" name="tool_replace_text" class="form-control" value="<?php echo e($toolReplaceText); ?>">
                            </div>
                        </div>

                        <div class="form-group">
                            <label class="text-tool-check">
                                <input type="checkbox" name="tool_case_sensitive" <?php echo $toolCaseSensitive ? 'checked' : ''; ?>>
                                <span>Case-sensitive match</span>
                            </label>
                        </div>

                        <div class="form-group">
                            <label for="tool-input-text">Input text</label>
                            <textarea id="tool-input-text" name="tool_input_text" class="form-control" rows="10" placeholder="Paste your text here..."><?php echo e($toolInputText); ?></textarea>
                        </div>

                        <?php if ($toolCountPreview !== null): ?>
                        <p class="text-tool-count">Current matches found: <?php echo (int)$toolCountPreview; ?></p>
                        <?php endif; ?>

                        <div class="text-tool-actions">
                            <button type="submit" name="tool_action" value="replace_text" class="btn btn-primary">Run Replace</button>
                            <button type="submit" name="tool_action" value="count_text" class="btn btn-secondary">Count Matches</button>
                        </div>
                    </form>
                </div>
            </div>

            <?php if ($toolOutputText !== null): ?>
            <div class="text-tool-panel">
                <div class="text-tool-panel-header">
                    <h3>Result (Pasted Text)</h3>
                </div>
                <div class="text-tool-panel-body">
                    <div class="contact-form">
                        <p class="text-muted">Replacements made: <?php echo (int)$toolOutputCount; ?></p>
                        <div class="form-group">
                            <label for="tool-output-text">Output</label>
                            <textarea id="tool-output-text" class="form-control" rows="10"><?php echo e($toolOutputText); ?></textarea>
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <div class="text-tool-panel">
                <div class="text-tool-panel-header">
                    <h3>Replace in Files</h3>
                </div>
                <div class="text-tool-panel-body">
                    <form method="POST" action="/<?php echo e($pageSlug ?? 'text-finder-replacer'); ?>" enctype="multipart/form-data" class="contact-form">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">

                        <div class="contact-grid">
                            <div class="form-group">
                                <label for="tool-file-search-text">Find</label>
                                <input id="tool-file-search-text" type="text" name="tool_search_text" class="form-control" value="<?php echo e($toolSearchText); ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="tool-file-replace-text">Replace with</label>
                                <input id="tool-file-replace-text" type="text" name="tool_replace_text" class="form-control" value="<?php echo e($toolReplaceText); ?>">
                            </div>
                        </div>

                        <div class="form-group">
                            <label class="text-tool-check">
                                <input type="checkbox" name="tool_case_sensitive" <?php echo $toolCaseSensitive ? 'checked' : ''; ?>>
                                <span>Case-sensitive match</span>
                            </label>
                        </div>

                        <div class="form-group">
                            <label for="tool-source-files">Upload .txt or .zip files</label>
                            <input id="tool-source-files" type="file" name="tool_source_files[]" class="form-control form-control-file" accept=".txt,.zip" multiple required>
                            <small class="text-tool-note">Supports .txt files and .zip archives with .txt files inside.</small>
                        </div>

                        <button type="submit" name="tool_action" value="replace_files" class="btn btn-primary">Process Files</button>
                    </form>
                </div>
            </div>

            <?php if (!empty($toolFileRows)): ?>
            <div class="text-tool-panel">
                <div class="text-tool-panel-header">
                    <h3>File Results</h3>
                </div>
                <div class="text-tool-panel-body">
                    <div class="text-tool-table-wrap">
                        <table class="text-tool-table">
                            <thead>
                                <tr>
                                    <th>Source</th>
                                    <th>Replacements</th>
                                    <th>Note</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($toolFileRows as $row): ?>
                                <tr>
                                    <td><?php echo e((string)($row['name'] ?? 'result.txt')); ?></td>
                                    <td><?php echo (int)($row['replacements'] ?? 0); ?></td>
                                    <td><?php echo e((string)($row['note'] ?? '')); ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <div class="contact-form">
                    <?php foreach ($toolFileRows as $row): ?>
                        <?php
                        $outputText = (string)($row['output'] ?? '');
                        if ($outputText === '') {
                            continue;
                        }
                        $downloadName = toolSafeDownloadName((string)($row['name'] ?? 'result.txt'));
                        $dataHref = 'data:text/plain;charset=utf-8,' . rawurlencode($outputText);
                        ?>
                        <div class="text-tool-output-block">
                            <div class="text-tool-output-name"><?php echo e($downloadName); ?></div>
                            <textarea class="form-control" rows="8"><?php echo e($outputText); ?></textarea>
                            <div class="text-tool-download">
                                <a class="btn btn-secondary btn-sm" href="<?php echo e($dataHref); ?>" download="<?php echo e($downloadName); ?>">Download .txt</a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            </div>
        </div>
    </div>
</section>
