<?php
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;

$sectionTitle = trim($sectionSettings['section_title'] ?? '');
$showTitle = isset($sectionSettings['show_title']) ? (bool)$sectionSettings['show_title'] : true;
$titleAlign = $sectionSettings['title_align'] ?? 'center';
$titleAlign = in_array($titleAlign, ['left', 'center', 'right'], true) ? $titleAlign : 'center';
$columns = (int)($sectionSettings['columns'] ?? 3);
$columns = max(2, min(4, $columns));
$showPrice = isset($sectionSettings['show_price']) ? (bool)$sectionSettings['show_price'] : true;
$showDescription = isset($sectionSettings['show_description']) ? (bool)$sectionSettings['show_description'] : true;
$showFilters = isset($sectionSettings['show_filters']) ? (bool)$sectionSettings['show_filters'] : true;
$showSort = isset($sectionSettings['show_sort']) ? (bool)$sectionSettings['show_sort'] : true;
$showViewToggle = isset($sectionSettings['show_view_toggle']) ? (bool)$sectionSettings['show_view_toggle'] : true;
$hoverSwapMedia = isset($sectionSettings['hover_swap_media']) ? (bool)$sectionSettings['hover_swap_media'] : true;
$defaultView = $sectionSettings['default_view'] ?? 'grid';
$defaultView = in_array($defaultView, ['grid', 'list'], true) ? $defaultView : 'grid';

function formatProductPrice(string $price): string
{
    return CurrencyService::formatValue($price, CurrencyService::getCurrent());
}

function isImageMediaUrl(string $url): bool
{
    $url = strtolower(trim($url));
    if ($url === '') {
        return false;
    }
    if (preg_match('~^https?://(www\.)?(youtube\.com|youtu\.be)/~', $url)) {
        return false;
    }
    return (bool)preg_match('/\.(jpg|jpeg|png|gif|webp|avif|svg)(\?.*)?$/', $url);
}

function isVideoMediaUrl(string $url): bool
{
    $url = strtolower(trim($url));
    if ($url === '') {
        return false;
    }
    if (preg_match('~^https?://(www\.)?(youtube\.com|youtu\.be)/~', $url)) {
        return true;
    }
    return (bool)preg_match('/\.(mp4|webm|mov|m4v|avi|mkv)(\?.*)?$/', $url);
}

$selectedCollectionId = isset($_GET['collection']) ? (int)$_GET['collection'] : 0;
$collectionTitle = '';
$baseTitle = $sectionTitle;
if ($selectedCollectionId > 0) {
    $stmt = $db->prepare("SELECT title FROM collections WHERE id = ? LIMIT 1");
    $stmt->execute([$selectedCollectionId]);
    $collectionTitle = trim((string)$stmt->fetchColumn());
    if ($collectionTitle !== '' && $showTitle && $baseTitle !== '') {
        $sectionTitle = $collectionTitle;
    }
}
$productsQuery = "SELECT p.*, pg.slug AS page_slug, pm.media_url AS gallery_media FROM products p LEFT JOIN pages pg ON pg.id = p.page_id LEFT JOIN product_media pm ON pm.product_id = p.id AND pm.sort_order = 0";
$productsParams = [];
if ($selectedCollectionId > 0) {
    $productsQuery .= " WHERE p.collection_id = ?";
    $productsParams[] = $selectedCollectionId;
}
$productsQuery .= " ORDER BY p.created_at DESC";
$stmt = $db->prepare($productsQuery);
$stmt->execute($productsParams);
$products = $stmt->fetchAll();
$activeProducts = array_values(array_filter($products, function ($product) {
    return (int)($product['is_active'] ?? 1) === 1;
}));
if (empty($activeProducts)) {
    return;
}

$productIds = array_map(function ($product) {
    return (int)$product['id'];
}, $activeProducts);
$productIds = array_values(array_filter($productIds));

$variantsByProduct = [];
$variantStockByProduct = [];
$blocksByProduct = [];
$attributesByProduct = [];
$mediaByProduct = [];
$filterGroups = [];

if (!empty($productIds)) {
    $placeholders = implode(',', array_fill(0, count($productIds), '?'));

    $stmt = $db->prepare("SELECT product_id, label, stock FROM product_variants WHERE product_id IN ($placeholders) ORDER BY sort_order ASC, id ASC");
    $stmt->execute($productIds);
    foreach ($stmt->fetchAll() as $variant) {
        $pid = (int)$variant['product_id'];
        $variantsByProduct[$pid][] = $variant;
        $variantStockByProduct[$pid][] = $variant['stock'];
    }

    $stmt = $db->prepare("SELECT product_id, title, content FROM product_blocks WHERE product_id IN ($placeholders) ORDER BY sort_order ASC, id ASC");
    $stmt->execute($productIds);
    foreach ($stmt->fetchAll() as $block) {
        $pid = (int)$block['product_id'];
        $blocksByProduct[$pid][] = $block;
    }

    $stmt = $db->prepare("SELECT product_id, name, value FROM product_attributes WHERE product_id IN ($placeholders) ORDER BY name ASC, value ASC");
    $stmt->execute($productIds);
    foreach ($stmt->fetchAll() as $attr) {
        $pid = (int)$attr['product_id'];
        $name = trim($attr['name']);
        $value = trim($attr['value']);
        if ($name === '' || $value === '') {
            continue;
        }
        if (!isset($attributesByProduct[$pid])) {
            $attributesByProduct[$pid] = [];
        }
        $attributesByProduct[$pid][$name] = $attributesByProduct[$pid][$name] ?? [];
        $attributesByProduct[$pid][$name][] = $value;
        $filterGroups[$name] = $filterGroups[$name] ?? [];
        $filterGroups[$name][$value] = ($filterGroups[$name][$value] ?? 0) + 1;
    }

    $stmt = $db->prepare("SELECT product_id, media_url, sort_order FROM product_media WHERE product_id IN ($placeholders) ORDER BY sort_order ASC, id ASC");
    $stmt->execute($productIds);
    foreach ($stmt->fetchAll() as $media) {
        $pid = (int)$media['product_id'];
        if (!isset($mediaByProduct[$pid])) {
            $mediaByProduct[$pid] = [];
        }
        $mediaByProduct[$pid][] = $media['media_url'];
    }
}

$availabilityCounts = ['in' => 0, 'out' => 0];
foreach ($activeProducts as $product) {
    $pid = (int)$product['id'];
    $stock = $product['stock'];
    $variantStocks = $variantStockByProduct[$pid] ?? [];
    $inStock = false;
    if (!empty($variantStocks)) {
        foreach ($variantStocks as $variantStock) {
            if ($variantStock === null || (int)$variantStock > 0) {
                $inStock = true;
                break;
            }
        }
    } else {
        $inStock = $stock === null || (int)$stock > 0;
    }
    if ($inStock) {
        $availabilityCounts['in']++;
    } else {
        $availabilityCounts['out']++;
    }
}
?>
<section class="section-products-list">
    <div class="container">
        <?php if ($showTitle && $sectionTitle !== ''): ?>
            <div class="products-header text-align-<?php echo e($titleAlign); ?>">
                <h2 class="text-align-<?php echo e($titleAlign); ?>"><?php echo e($sectionTitle); ?></h2>
            </div>
        <?php endif; ?>
        <div class="products-list-layout <?php echo $showFilters ? '' : 'no-filters'; ?>" data-products-list data-hover-media="<?php echo $hoverSwapMedia ? '1' : '0'; ?>">
            <?php if ($showSort || $showViewToggle || $showFilters): ?>
            <div class="products-list-toolbar">
                <div class="toolbar-left">
                    <?php if ($showFilters): ?>
                    <div class="toolbar-group">
                        <label class="filter-toggle">
                            <input type="checkbox" data-filters-toggle checked>
                            <span data-filters-label>Filters</span>
                        </label>
                    </div>
                    <?php endif; ?>
                    <div class="toolbar-group products-count" data-products-count>
                        Showing <?php echo count($activeProducts); ?> of <?php echo count($activeProducts); ?> Products
                    </div>
                </div>
                <div class="toolbar-right">
                    <?php if ($showSort): ?>
                    <div class="toolbar-group">
                        <span class="sort-label">Sort by</span>
                        <div class="sort-dropdown" data-products-sort>
                            <button type="button" class="sort-trigger" data-sort-trigger>
                                <span data-sort-label>Featured</span>
                                <svg viewBox="0 0 24 24" aria-hidden="true">
                                    <path d="M6 9l6 6 6-6"></path>
                                </svg>
                            </button>
                            <div class="sort-menu" data-sort-menu>
                                <button type="button" class="sort-option" data-sort-option="featured">Featured</button>
                                <button type="button" class="sort-option" data-sort-option="bestselling">Bestselling</button>
                                <button type="button" class="sort-option" data-sort-option="price_asc">Price: Low to High</button>
                                <button type="button" class="sort-option" data-sort-option="price_desc">Price: High to Low</button>
                                <button type="button" class="sort-option" data-sort-option="date_new">Date: New to Old</button>
                                <button type="button" class="sort-option" data-sort-option="date_old">Date: Old to New</button>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    <?php if ($showViewToggle): ?>
                    <div class="toolbar-group view-toggle" data-view-toggle>
                        <button type="button" class="view-btn <?php echo $defaultView === 'grid' ? 'active' : ''; ?>" data-view="grid" aria-label="Grid view" data-tooltip="Grid view">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <rect x="3" y="3" width="7" height="7" rx="1"></rect>
                                <rect x="14" y="3" width="7" height="7" rx="1"></rect>
                                <rect x="3" y="14" width="7" height="7" rx="1"></rect>
                                <rect x="14" y="14" width="7" height="7" rx="1"></rect>
                            </svg>
                        </button>
                        <button type="button" class="view-btn <?php echo $defaultView === 'list' ? 'active' : ''; ?>" data-view="list" aria-label="List view" data-tooltip="List view">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <rect x="4" y="5" width="16" height="3" rx="1"></rect>
                                <rect x="4" y="10.5" width="16" height="3" rx="1"></rect>
                                <rect x="4" y="16" width="16" height="3" rx="1"></rect>
                            </svg>
                        </button>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <div class="products-list-divider"></div>
            <?php endif; ?>
            <?php if ($showFilters): ?>
            <aside class="products-filters">
                <div class="filters-card">
                    <div class="filters-card-header">
                        <h4>Availability</h4>
                        <button type="button" class="filter-reset" data-filter-reset>Reset</button>
                    </div>
                    <label class="filter-option">
                        <input type="checkbox" value="in" data-filter="availability">
                        <span>In Stock</span>
                        <small><?php echo (int)$availabilityCounts['in']; ?></small>
                    </label>
                    <label class="filter-option">
                        <input type="checkbox" value="out" data-filter="availability">
                        <span>Out of Stock</span>
                        <small><?php echo (int)$availabilityCounts['out']; ?></small>
                    </label>
                </div>
                <?php foreach ($filterGroups as $groupName => $values): ?>
                    <div class="filters-card">
                        <div class="filters-card-header">
                            <h4><?php echo e($groupName); ?></h4>
                            <button type="button" class="filter-reset" data-filter-reset>Reset</button>
                        </div>
                        <?php foreach ($values as $valueLabel => $count): ?>
                            <label class="filter-option">
                                <input type="checkbox" value="<?php echo e($valueLabel); ?>" data-filter="attr" data-filter-group="<?php echo e($groupName); ?>">
                                <span><?php echo e($valueLabel); ?></span>
                                <small><?php echo (int)$count; ?></small>
                            </label>
                        <?php endforeach; ?>
                    </div>
                <?php endforeach; ?>
            </aside>
            <?php endif; ?>
            <div class="products-list-content">
                <div class="product-list-grid columns-<?php echo $columns; ?> view-<?php echo e($defaultView); ?>" data-products-grid>
                    <?php foreach ($activeProducts as $product):
                        $pid = (int)$product['id'];
                        $name = trim($product['name'] ?? '');
                        $price = trim($product['price'] ?? '');
                        $displayPrice = formatProductPrice($price);
                        $defaultMediaRaw = trim($product['gallery_media'] ?? $product['media_url'] ?? '');
                        $mediaList = $mediaByProduct[$pid] ?? [];
                        $mediaUrl = '';
                        $primaryMediaRaw = '';

                        if (!empty($mediaList)) {
                            foreach ($mediaList as $mediaItem) {
                                $mediaItem = trim((string)$mediaItem);
                                if ($mediaItem === '' || isVideoMediaUrl($mediaItem) || !isImageMediaUrl($mediaItem)) {
                                    continue;
                                }

                                $primaryCandidate = toPublicFilePath($mediaItem);
                                if ($primaryCandidate !== '') {
                                    $mediaUrl = $primaryCandidate;
                                    $primaryMediaRaw = $mediaItem;
                                    break;
                                }
                            }
                        }

                        if ($mediaUrl === '' && $defaultMediaRaw !== '' && !isVideoMediaUrl($defaultMediaRaw) && isImageMediaUrl($defaultMediaRaw)) {
                            $primaryCandidate = toPublicFilePath($defaultMediaRaw);
                            if ($primaryCandidate !== '') {
                                $mediaUrl = $primaryCandidate;
                                $primaryMediaRaw = $defaultMediaRaw;
                            }
                        }

                        $hoverMedia = '';
                        if (!empty($mediaList)) {
                            foreach ($mediaList as $mediaItem) {
                                $mediaItem = trim((string)$mediaItem);
                                if ($mediaItem === '' || $mediaItem === $primaryMediaRaw) {
                                    continue;
                                }
                                if (isVideoMediaUrl($mediaItem) || !isImageMediaUrl($mediaItem)) {
                                    continue;
                                }
                                $hoverCandidate = toPublicFilePath($mediaItem);
                                if ($hoverCandidate !== '' && $hoverCandidate !== $mediaUrl) {
                                    $hoverMedia = $hoverCandidate;
                                    break;
                                }
                            }
                        }
                        $pageSlug = trim($product['page_slug'] ?? '');
                        $productSlug = trim((string)($product['product_slug'] ?? ''));
                        $actionType = $product['action_type'] ?? 'cart';
                        $actionType = in_array($actionType, ['cart', 'download', 'external_link'], true) ? $actionType : 'cart';
                        $externalUrl = trim($product['download_url'] ?? '');
                        $isExternalLink = $actionType === 'external_link' && $externalUrl !== '';
                        if ($pageSlug !== '' && $productSlug !== '') {
                            $productUrl = '/' . $pageSlug . '/' . rawurlencode($productSlug);
                        } else {
                            $productUrl = $pageSlug !== '' ? '/' . $pageSlug . '?product=' . $pid : '/product?product=' . $pid;
                        }
                        $productTarget = '';
                        $productRel = '';
                        $variants = $variantsByProduct[$pid] ?? [];
                        $blocks = $blocksByProduct[$pid] ?? [];
                        $description = trim($product['description'] ?? '');
                        $shortDescription = $description !== '' ? trim(preg_replace('/\s+/', ' ', strip_tags($description))) : '';
                        $attrs = $attributesByProduct[$pid] ?? [];
                        $attrsJson = htmlspecialchars(json_encode($attrs), ENT_QUOTES, 'UTF-8');
                        $createdAt = strtotime($product['created_at'] ?? 'now');
                        $featured = !empty($product['is_featured']) ? 1 : 0;
                        $salesCount = (int)($product['sales_count'] ?? 0);
                        $stock = $product['stock'];
                        $variantStocks = $variantStockByProduct[$pid] ?? [];
                        $inStock = false;
                        if (!empty($variantStocks)) {
                            foreach ($variantStocks as $variantStock) {
                                if ($variantStock === null || (int)$variantStock > 0) {
                                    $inStock = true;
                                    break;
                                }
                            }
                        } else {
                            $inStock = $stock === null || (int)$stock > 0;
                        }
                        $numericPrice = $price !== '' ? (float)preg_replace('/[^0-9.]/', '', $price) : 0;
                        $numericPrice = CurrencyService::convert($numericPrice, SettingsService::get('store_currency', 'USD'), CurrencyService::getCurrent());

                        // Responsive image: sizes match CSS grid layout, srcset covers 1x-2x DPR
                        $prodSizesMap = [2 => '(max-width: 640px) 100vw, (max-width: 900px) 50vw, 50vw', 3 => '(max-width: 640px) 100vw, (max-width: 900px) 50vw, 33vw', 4 => '(max-width: 640px) 100vw, (max-width: 900px) 50vw, 25vw'];
                        $prodSizes = $prodSizesMap[$columns] ?? $prodSizesMap[3];
                        $prodWebpSm  = $mediaUrl . '?w=400&format=webp';
                        $prodWebpMd  = $mediaUrl . '?w=700&format=webp';
                        $prodWebpLg  = $mediaUrl . '?w=1000&format=webp';
                        ?>
                        <?php if ($productUrl !== ''): ?>
                            <a class="product-list-card <?php echo ($hoverSwapMedia && $hoverMedia !== '') ? 'has-hover-media' : ''; ?>" href="<?php echo e($productUrl); ?>"<?php echo $productTarget !== '' ? ' target="' . e($productTarget) . '"' : ''; ?><?php echo $productRel !== '' ? ' rel="' . e($productRel) . '"' : ''; ?> data-in-stock="<?php echo $inStock ? '1' : '0'; ?>" data-featured="<?php echo $featured; ?>" data-bestselling="<?php echo $salesCount; ?>" data-price="<?php echo e((string)$numericPrice); ?>" data-date="<?php echo e((string)$createdAt); ?>" data-attrs="<?php echo $attrsJson; ?>" data-hover-media="<?php echo e($hoverMedia); ?>">
                                <?php if ($mediaUrl !== ''): ?>
                                    <div class="product-list-media">
                                        <img src="<?php echo e($prodWebpMd); ?>" srcset="<?php echo e($prodWebpSm); ?> 400w, <?php echo e($prodWebpMd); ?> 700w, <?php echo e($prodWebpLg); ?> 1000w" sizes="<?php echo e($prodSizes); ?>" alt="<?php echo e($name); ?>" data-default-media="<?php echo e($prodWebpMd); ?>" loading="lazy" decoding="async">
                                    </div>
                                <?php endif; ?>
                                <div class="product-list-body">
                                    <?php if ($name !== ''): ?>
                                        <h3 class="product-list-title"><?php echo e($name); ?></h3>
                                    <?php endif; ?>
                                    <?php if (!empty($variants)): ?>
                                        <div class="product-list-variants">
                                            <?php foreach ($variants as $variant):
                                                $variantLabel = $variant['label'] ?? '';
                                                $variantStock = $variant['stock'];
                                                $variantOut = $variantStock !== null && (int)$variantStock <= 0;
                                                ?>
                                                <span class="<?php echo $variantOut ? 'is-out-of-stock' : ''; ?>"><?php echo e($variantLabel); ?></span>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endif; ?>
                                    <?php if ($showPrice && $displayPrice !== '' && !$isExternalLink): ?>
                                        <div class="product-list-price"><?php echo e($displayPrice); ?></div>
                                    <?php endif; ?>
                                    <?php if ($showDescription && $shortDescription !== ''): ?>
                                        <p class="product-list-description"><?php echo e($shortDescription); ?></p>
                                    <?php endif; ?>
                                </div>
                            </a>
                        <?php else: ?>
                            <article class="product-list-card <?php echo ($hoverSwapMedia && $hoverMedia !== '') ? 'has-hover-media' : ''; ?>" data-in-stock="<?php echo $inStock ? '1' : '0'; ?>" data-featured="<?php echo $featured; ?>" data-bestselling="<?php echo $salesCount; ?>" data-price="<?php echo e((string)$numericPrice); ?>" data-date="<?php echo e((string)$createdAt); ?>" data-attrs="<?php echo $attrsJson; ?>" data-hover-media="<?php echo e($hoverMedia); ?>">
                                <?php if ($mediaUrl !== ''): ?>
                                    <div class="product-list-media">
                                        <img src="<?php echo e($prodWebpMd); ?>" srcset="<?php echo e($prodWebpSm); ?> 400w, <?php echo e($prodWebpMd); ?> 700w, <?php echo e($prodWebpLg); ?> 1000w" sizes="<?php echo e($prodSizes); ?>" alt="<?php echo e($name); ?>" data-default-media="<?php echo e($prodWebpMd); ?>" loading="lazy" decoding="async">
                                    </div>
                                <?php endif; ?>
                                <div class="product-list-body">
                                    <?php if ($name !== ''): ?>
                                        <h3 class="product-list-title"><?php echo e($name); ?></h3>
                                    <?php endif; ?>
                                    <?php if (!empty($variants)): ?>
                                        <div class="product-list-variants">
                                            <?php foreach ($variants as $variant):
                                                $variantLabel = $variant['label'] ?? '';
                                                $variantStock = $variant['stock'];
                                                $variantOut = $variantStock !== null && (int)$variantStock <= 0;
                                                ?>
                                                <span class="<?php echo $variantOut ? 'is-out-of-stock' : ''; ?>"><?php echo e($variantLabel); ?></span>
                                            <?php endforeach; ?>
                                        </div>
                                    <?php endif; ?>
                                    <?php if ($showPrice && $displayPrice !== '' && !$isExternalLink): ?>
                                        <div class="product-list-price"><?php echo e($displayPrice); ?></div>
                                    <?php endif; ?>
                                    <?php if ($showDescription && $shortDescription !== ''): ?>
                                        <p class="product-list-description"><?php echo e($shortDescription); ?></p>
                                    <?php endif; ?>
                                </div>
                            </article>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
</section>
